/**
 * $Id: MacroInstanceCaller.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.UnknownElement;
import  org.apache.tools.ant.taskdefs.Ant;
import  org.apache.tools.ant.taskdefs.Property;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.helpers.InnerNameValuePair;
import  com.idaremedia.antx.ownhelpers.ProjectDependentSkeleton;
import  com.idaremedia.antx.ownhelpers.ScopedProperties;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;

/**
 * Implementation of {@linkplain TargetCaller} for local macrodefs. Without any
 * overlayed properties and/or attributes, this caller is an expensive adapter of the
 * standard macro invocation mechanism (so you really want to use it only if
 * overlaying is likely).
 *
 * @since    JWare/AntX 0.5
 * @author   ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public final class MacroInstanceCaller extends ProjectDependentSkeleton
    implements TargetCaller, ConfigurableCaller
{
    private static final String IAM_= AntX.flow+"MacroInstanceCaller";


    /**
     * Initializes a new local macro instance caller. This
     * caller's target and project must be defined before it
     * is used.
     * @see #setCaller setCaller(&#8230;)
     **/
    public MacroInstanceCaller()
    {
    }


    /**
     * Initializes a new local macro instance caller. This
     * caller's target must be defined before it is used.
     * @param task controlling task (non-null)
     * @see #setCaller setCaller(&#8230;)
     **/
    public MacroInstanceCaller(CallerTask task)
    {
        setCaller(task);
    }



    /**
     * Initializes this caller's controlling task.
     * @param task controlling task (non-null)
     * @.sideeffect This caller's target name is initialized
     *     to the task's enclosing target's name (if exists).
     **/
    public void setCaller(Task task)
    {
        AntX.require_(task!=null,IAM_,"setCalr- nonzro task");
        Target parent = task.getOwningTarget();
        if (parent!=null) {
            setTarget(parent.getName());
        }
        setProject(task.getProject());
        m_owningTask = task;
    }



    /**
     * (Re)Sets the target macro's name for this caller.
     **/
    public void setTarget(String targetName)
    {
        m_targetName = targetName;
    }


    /**
     * Returns this caller's target macro's name.
     **/
    public String getTargetName()
    {
        return m_targetName;
    }



    /**
     * Unsupported operation; only top-level macros can be called.
     * @throws UnsupportedOperationException always.
     **/
    public final void setStepName(String stepName)
    {
        throw new UnsupportedOperationException();
    }


    /**
     * Returns <i>null</i> always; can only call top-level macros.
     **/
    public final String getStepName()
    {
        return null;
    }



    /**
     * Create a new property to passthru to called macro.
     **/
    public Property createProperty()
    {
        Property p = new Property();
        p.setProject(getProject());
        if (m_plist==null) {
            m_plist = AntXFixture.newList();
        }
        m_plist.add(p);
        return p;
    }



    /**
     * Create a new property set to passthru to called macro.
     **/
    public PropertySet createPropertySet()
    {
        PropertySet p = new PropertySet();//?use just one?
        p.setProject(getProject());
        if (m_plist==null) {
            m_plist = AntXFixture.newList();
        }
        m_plist.add(p);
        return p;
    }



    /**
     * Unsupported operation; only macro parameters can be created.
     * @throws UnsupportedOperationException always.
     **/
    public Ant.Reference createReference()
    {
        throw new UnsupportedOperationException();
    }



    /**
     * Create a new macro attribute to passthru to called macro.
     **/
    public InnerNameValuePair createAttribute()
    {
        InnerNameValuePair a = new InnerNameValuePair();
        if (m_plist==null) {
            m_plist = AntXFixture.newList();
        }
        m_plist.add(a);
        return a;
    }



    /**
     * Installs all of our property declarations as a PropertyHelper
     * hook. Any macro attribute definitions are returned to be
     * given directly to macro instance factory.
     **/
    private Map installOverlay()
    {
        Project P= getProjectNoNull();
        Map attrs = null;
        if (m_plist!=null) {
            m_overlay = new ScopedProperties(P,true);
            Iterator itr= m_plist.iterator();
            while (itr.hasNext()) {
                Object o = itr.next();
                if (o instanceof InnerNameValuePair) {
                    InnerNameValuePair nv = (InnerNameValuePair)o;
                    nv.verifyNamed();
                    if (attrs==null) {
                        attrs = AntXFixture.newMap();
                    }
                    attrs.put(nv.getName(), nv.getValue(P,true));
                }
                else if (o instanceof Property) {
                    m_overlay.put((Property)o);
                }
                else {
                    m_overlay.put((PropertySet)o);
                }
            }
            m_overlay.install();
        }
        return attrs;
    }



    /**
     * Undoes the effect of {@linkplain #installOverlay}.
     **/
    private void uninstallOverlay()
    {
        if (m_overlay!=null) {
            m_overlay.uninstall(null);
            m_overlay=null;
        }
    }




    /**
     * Executes a new instance of this caller's named macro. Any custom
     * properties are converted into property overlay properties; named
     * attribute values are passed directly into the macro instance
     * factory.
     **/
    public void run()
        throws BuildException
    {
        Map attrs = installOverlay();
        try {
            UnknownElement launch = TaskExaminer.newUEProxy
                (getTargetName(),attrs,(String)null,getProject());
            launch.setOwningTarget(m_owningTask.getOwningTarget());
            launch.perform();
        } finally {
            uninstallOverlay();
        }
    }



    /**
     * Convenient {@linkplain #run() run} alternative that defines
     * a required attribute of the macro instance before we execute
     * it.
     * @param property property name (non-null)
     * @param value property value-- used as-is no additional
     *          replacements are done (non-null)
     **/
    public final void run(String property, String value)
        throws BuildException
    {
        Property p= this.createProperty();
        p.setName(property);
        p.setValue(value);
        run();
    }



    /**
     * Convenient {@linkplain #run() run} alternative that defines
     * a set of macro instance attributes before we execute it.
     * @param properties properties file (non-null)
     **/
    public final void run(Map properties)
        throws BuildException
    {
        if (!properties.isEmpty()) {
            Iterator itr = properties.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry item = (Map.Entry)itr.next();
                Property p= this.createProperty();
                p.setName(item.getKey().toString());
                p.setValue(item.getValue().toString());
            }
        }
        run();
    }



    /**
     * Convenient {@linkplain #run() run} alternative that defines
     * non-trivial macro instance attributes before we execute it.
     * @param prep client preparation snippet (non-null)
     **/
    public final void run(TargetCaller.Prep prep)
        throws BuildException
    {
        prep.prepare(this);
        run();
    }


    private String m_targetName;
    private List m_plist;
    private Task m_owningTask;
    private ScopedProperties m_overlay;//as-propertyhook
}

/* end-of-MacroInstanceCaller.java */
