/**
 * $Id: LocalTargetTask.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.solo.CopyPropertyTask;

/**
 * Task that will execute a set of targets <em>within</em> the enclosing project. A
 * new subproject is not created and any modifications to the fixture are seen. The
 * named targets are <em>not</em> checked for duplication (directly or indirectly).
 * This task is fairly straightforwardd: it just executes the targets <em>in the order
 * you list them</em>. If you would like to execute the named targets in their own
 * partitioned space, use the {@linkplain CallOnceTask}.
 * <p>
 * If at least one of the named targets does not exist in the enclosing project,
 * <em>no</em> targets are run and a build exception is thrown.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;callinline targets="--init_a,--init_b"/&gt;
 *    &lt;callinline targets="--init_a,--init_b" haltiferror="no" tryeach="yes"/&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class LocalTargetTask extends CallerTask
{
    /**
     * Initializes a new inlined target caller task.
     **/
    public LocalTargetTask()
    {
        super(AntX.flow+"callinline");
    }



    /**
     * Initializes a new enclosed target caller task.
     * @param iam CV-label (non-null)
     **/
    public LocalTargetTask(String iam)
    {
        super(iam);
    }

//  ---------------------------------------------------------------------------------------
//  Unique Script-facing Parameters:
//  ---------------------------------------------------------------------------------------


    /**
     * Initializes the names of targets to be executed inlined.
     * @param setOfTargetNames comma-delimited list of names (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setTargets(String setOfTargetNames)
    {
        require_(setOfTargetNames!=null,"setTargets- nonzro names");
        if (m_targetNames!=null) {
            String error = uistrs().get("task.one.or.other.attr","targets","steps");
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        m_targetNames = setOfTargetNames;
        m_topLevel = true;
    }



    /**
     * Shorthand way to initialize the target name of a overridden,
     * imported target of the same name as this task's owning target.
     * @param importedProjectName name of imported project whose target
     *            we'd like to call (non-null)
     * @since JWare/AntX 0.5
     **/
    public final void setSuper(String importedProjectName)
    {
        require_(importedProjectName!=null,"setSuper- nonzro projectname");
        verifyInTarget_("setsuper");
        String mytarget = getOwningTarget().getName();
        setTargets(importedProjectName+"."+mytarget);
    }



    /**
     * Returns the comma-delimited names of targets to be
     * executed inlined. Will return <i>null</i> if never
     * set explicitly.
     **/
    public final String getTargetNamesList()
    {
        return m_topLevel ? m_targetNames : null;
    }



    /**
     * Initializes the list of steps to be executed inlined.
     * @param setOfStepNames comma-delimited list of names (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setSteps(String setOfStepNames)
    {
        require_(setOfStepNames!=null,"setSteps- nonzro names");
        if (m_targetNames!=null) {
            String error = uistrs().get("task.one.or.other.attr","targets","steps");
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        m_targetNames = setOfStepNames;
        m_topLevel = false;
    }



    /**
     * Returns the comma-delimited names of steps to be
     * executed inlined. Will return <i>null</i> if never
     * set explicitly.
     **/
    public final String getStepNamesList()
    {
        return m_topLevel ? null : m_targetNames;
    }



    /**
     * Adds a new fixture overlay property to this task.
     * This property is automatically passed to each target's
     * environment as it is executed.
     * @param param initialized property (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addProperty(CopyPropertyTask param)
    {
        require_(param!=null,"addProperty- nonzro item");
        getParametersKeeper().addParameter(param);
    }



    /**
     * Adds a new fixture overlay property set to this task. The
     * properties named by the property set are automatically
     * copied to  each target's environment before it is executed.
     * @param param property set information (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addPropertySet(PropertySet param)
    {
        require_(param!=null,"addPropertySet- nonzro item");
        getParametersKeeper().addParameter(param);
    }



    /**
     * Ensures this task's list of targets is set and each target
     * exists in enclosing project.
     * @throws BuildException if target list is missing or at least
     *         one target does not exist.
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (m_targetNames==null) {
            String error = uistrs().get("task.needs.this.attr",
                                        getTaskName(),"targets|steps");
            log(error,Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }

        List l= Tk.splitList(m_targetNames);
        m_targets = l;

        if (m_topLevel) {
            String tn;
            for (int i=0,N=l.size();i<N;i++) {
                tn = l.get(i).toString();
                if (!targetExists(tn)) {
                    String error = uistrs().get
                        ("flow.steplaunch.missing.target",tn,getProject().getName());
                    log(error,Project.MSG_ERR);
                    throw new BuildException(error,getLocation());
                }
            }
        }
    }


    /**
     * Returns a filled in list of target callers for this
     * task's list.
     * @throws BuildException if name list invalid or a called
     *         target fails.
     */
    protected List copyOfOrderedTargetCallers()
    {
        verify_(m_targets!=null,"copyOf- been verified");

        List l= m_targets;
        List callers= AntXFixture.newList(l.size());
        String tn;
        LocalTargetCaller caller;

        for (int i=0,N=l.size();i<N;i++) {
            tn = l.get(i).toString();
            caller = new LocalTargetCaller(this);
            if (m_topLevel) {
                caller.setTarget(tn);
            } else {
                caller.setStepName(tn);
            }
            transferOverlayParameters(caller);
            callers.add(caller);
        }

        return callers;
    }



    private String m_targetNames;
    private List m_targets;
    private boolean m_topLevel=true;//NB:false if steps
}

/* end-of-LocalTargetTask.java */
