/**
 * $Id: LocalTargetCaller.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.taskdefs.Ant;
import  org.apache.tools.ant.taskdefs.Property;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.ownhelpers.ProjectDependentSkeleton;
import  com.idaremedia.antx.ownhelpers.ScopedProperties;

/**
 * Implementation of {@linkplain TargetCaller} for {@linkplain LocalTargetTask}.
 * Without any overlayed properties, this caller is an expensive adapter of the
 * <span class="src">Project</span> object's <span class="src">executeTarget</span>
 * method (so you really want to use it only if overlaying is likely).
 * <p>
 * When you need to overlay properties for multiple targets, it is much more
 * efficient to run multiple targets from a single target caller (by resetting
 * the <span class="src">targetName</span> attribute) than it is to create a new
 * target caller everytime. The caller will cache the property and propertyset
 * declarations for (re)installation for every target.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public final class LocalTargetCaller extends ProjectDependentSkeleton
    implements TargetCaller
{
    private static final String IAM_= AntX.flow+"LocalTargetCaller";


    /**
     * Initializes a new local target caller instance. This
     * caller's target and project must be defined before it
     * is used.
     * @see #setCaller setCaller(&#8230;)
     **/
    public LocalTargetCaller()
    {
    }


    /**
     * Initializes a new local target caller instance. This
     * caller's target must be defined before it is used.
     * @param task controlling task (non-null)
     * @see #setCaller setCaller(&#8230;)
     **/
    public LocalTargetCaller(CallerTask task)
    {
        setCaller(task);
    }



    /**
     * Initializes this caller's controlling task. Inlined
     * steps will be read from this task's owning target.
     * @param task controlling task (non-null)
     * @.sideeffect This caller's target name is initialized
     *     to the task's enclosing target's name (if exists).
     **/
    public void setCaller(Task task)
    {
        AntX.require_(task!=null,IAM_,"setCalr- nonzro task");
        Target parent = task.getOwningTarget();
        if (parent!=null) {
            setTarget(parent.getName());
        }
        setProject(task.getProject());
        m_owningTask = task;
    }



    /**
     * (Re)Sets the target name of this caller.
     **/
    public void setTarget(String targetName)
    {
        m_targetName = targetName;
    }


    /**
     * Returns this caller's target's name.
     **/
    public String getTargetName()
    {
        return m_targetName;
    }


    /**
     * Sets the step on whose behave this caller exists.
     **/
    public final void setStepName(String stepName)
    {
        m_stepName = stepName;
    }


    /**
     * Returns the step on whose behave this caller exists.
     **/
    public final String getStepName()
    {
        return m_stepName;
    }


    /**
     * Create a new property to overlay called target.
     **/
    public Property createProperty()
    {
        Property p = new Property();
        p.setProject(getProject());
        if (m_plist==null) {
            m_plist = AntXFixture.newList();
        }
        m_plist.add(p);
        return p;
    }


    /**
     * Create a new property set to overlay called target.
     **/
    public PropertySet createPropertySet()
    {
        PropertySet p = new PropertySet();//?use just one?
        p.setProject(getProject());
        if (m_plist==null) {
            m_plist = AntXFixture.newList();
        }
        m_plist.add(p);
        return p;
    }



    /**
     * Create a new reference to overlay called target.
     * Currently ignored.
     **/
    public Ant.Reference createReference()
    {
        return null;
    }



    /**
     * Installs all of our property declarations as a
     * PropertyHelper hook.
     **/
    private void installOverlay()
    {
        Project P= getProjectNoNull();
        if (m_plist!=null) {
            m_overlay = new ScopedProperties(P,true);
            Iterator itr= m_plist.iterator();
            while (itr.hasNext()) {
                Object o = itr.next();
                if (o instanceof Property) {
                    m_overlay.put((Property)o);
                } else {
                    m_overlay.put((PropertySet)o);
                }
            }
            m_overlay.install();
        }
    }


    /**
     * Undoes the effect of {@linkplain #installOverlay}.
     **/
    private void uninstallOverlay()
    {
        if (m_overlay!=null) {
            m_overlay.uninstall(null);
            m_overlay=null;
        }
    }


    private Task[] getFreshTargetTasks(String targetName)
    {
        Target t = (Target)getProject().getTargets().get(getTargetName());
        //@.todo Add call to 'reconfigure' if tasks != UnknownElement
        return t.getTasks();
    }


    /**
     * Executes the target this caller references with
     * overlay properties if necessary.
     * @.bug Update to try to reconfigure tasks that are Reconfigurable
     **/
    public void run()
        throws BuildException
    {
        installOverlay();
        try {
            if (getStepName()==null) {
                getProject().executeTarget(getTargetName());
            }
            else {
                StepLauncher steplaunch = new StepLauncher();
                steplaunch.setProject(getProject());

                Task[] fromTasks = getFreshTargetTasks(getTargetName());

                InlineStep step = (InlineStep)steplaunch.getInlinedTarget
                    (getStepName(),InlineStep.class,fromTasks,1,1);

                steplaunch = null;//gc

                if (step!=null) {
                    step.maybeConfigure();
                    step.run();
                } else {
                    String error = AntX.uistrs().
                        get("flow.steplaunch.missing.step", getStepName(),
                            getTargetName());
                    m_owningTask.log(error,Project.MSG_ERR);
                    throw new BuildException
                        (error, m_owningTask.getLocation());
                }
            }
        } finally {
            uninstallOverlay();
        }
    }



    /**
     * Convenient {@linkplain #run() run} alternative that auto-installs
     * a required property into the to-be-called target's runtime
     * environmentbefore it's executed.
     * @param property property name (non-null)
     * @param value property value-- used as-is no additional
     *          replacements are done (non-null)
     **/
    public final void run(String property, String value)
        throws BuildException
    {
        Property p= this.createProperty();
        p.setName(property);
        p.setValue(value);
        run();
    }



    /**
     * Convenient {@linkplain #run() run} alternative that auto-installs
     * a set of properties into the to-be-called target's runtime
     * environment before it's executed.
     * @param properties properties file (non-null)
     **/
    public final void run(Map properties)
        throws BuildException
    {
        if (!properties.isEmpty()) {
            Iterator itr = properties.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry item = (Map.Entry)itr.next();
                Property p= this.createProperty();
                p.setName(item.getKey().toString());
                p.setValue(item.getValue().toString());
            }
        }
        run();
    }



    /**
     * Convenient {@linkplain #run() run} alternative that auto-installs
     * arbitrary kinds of fixture data into the to-be-called target's
     * runtime environment before it's executed.
     * @param prep client preparation snippet (non-null)
     **/
    public final void run(TargetCaller.Prep prep)
        throws BuildException
    {
        prep.prepare(this);
        run();
    }


    private String m_targetName, m_stepName;
    private List m_plist;//raw-property-decl
    private ScopedProperties m_overlay;//as-propertyhook
    private Task m_owningTask;
}

/* end-of-LocalTargetCaller.java */
