/**
 * $Id: InlineForEachTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.ComponentHelper;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.helpers.Empties;
import  com.idaremedia.antx.helpers.SIDs;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;
import  com.idaremedia.antx.parameters.ExecutionMode;
import  com.idaremedia.antx.parameters.FixturePassthru;

/**
 * Special-case extension of a {@linkplain ForEachTask} task that allows the iteration
 * steps to be nested inside the foreach taskset itself. This task exists to address some
 * of the readability and maintenance issues lots of functions-masking-as-targets tend
 * to create. Usally defined <span class="src">&lt;foreach&gt;</span>.
 * <p>
 * To use this task you <em>must</em> define the special AntX step-launch target unless
 * the mode is set to "local"; see {@linkplain StepLauncherInstallTask}.
 * <p>
 * <b>Example Usage:</b><pre>
 * &lt;target name="preview"&gt;
 *    ...
 *    &lt;foreach name="compile-test" i="build.type" list="debug,opt"
 *             haltiferror="no" tryeach="true"
 *             failproperty="notclean-compile-test"&gt;
 *
 *      &lt;property name="_rootdir" value="${preview.root}/${build.type}"/&gt;
 *      &lt;callinline targets="--setup.${build.type}"/&gt;
 *
 *      &lt;property name="_classes" value="${_rootdir}/classes"/&gt;
 *      &lt;mkdir dir="${_classes}"/&gt;
 *
 *      &lt;javac srcdir="${_source}" destdir="${_classes}"
 *             debug="${javac.isdebug}" deprecation="${javac.isdebug}"
 *             optimize="${javac.isopt}"
 *               .../&gt;
 *   &lt;/foreach&gt;
 * &lt;/target&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      StepLauncher
 * @.caveat  An inlined foreach task must be uniquely named within its <em>target's</em>
 *           scope so that a StepLauncher can find it (even it's nested within another task).
 **/

public final class InlineForEachTask extends ForEachTask
    implements InlinedTarget, InlinedTargetContainer
{
    /**
     * Maximum search depth for nested inlined-foreach tasks. This
     * limit is hard-coded for now (until it's shown to be useful
     * as a general parameter).
     **/
    public final static int MAX_SEARCH_DEPTH= 3;


    /**
     * Initializes new inline-foreach task.
     **/
    public InlineForEachTask()
    {
        super(AntX.flow+"foreach");
    }


    /**
     * Initializes new CV-labeled inline-foreach task.
     * @param iam CV-label (non-null)
     **/
    public InlineForEachTask(String iam)
    {
        super(iam);
    }



    /**
     * Sets this task's inline step-name. Must be called before
     * this task can be executed.
     * @param name new name (non-null)
     **/
    public void setName(String name)
    {
        require_(name!=null,"setName- nonzro name");
        m_name = name;
    }



    /**
     * Returns this task's inline step-name; returns <i>null</i>
     * if never set.
     **/
    public String getName()
    {
        return m_name;
    }



    /**
     * Instructs this task what kinds of fixture informaton
     * should be passed to the child project. By default only
     * properties are passed on.
     * @since JWare/AntX 0.4
     **/
    public void setPassthru(FixturePassthru passthru)
    {
        require_(passthru!=null,"setPassthru- nonzro option");
        m_passthru = FixturePassthru.from(passthru.getIndex());//NB:normalize
    }



    /**
     * Returns the fixture passthru setting to be used when this
     * caller runs its targets. Never returns <i>null</i>;
     * will return <span class="src">PROPERTIES</span> if
     * not set explicitly.
     * @since JWare/AntX 0.4
     **/
    public final FixturePassthru getPassthruOption()
    {
        return m_passthru;
    }



    /**
     * Tells this task to execute self locally like a macrodef.
     * @param Xmodestring either "local" or "isolated"
     * @since JWare/AntX 0.5
     **/
    public void setMode(String Xmodestring)
    {
        require_(Xmodestring!=null,"setMode- nonzro string");
        m_Xmode = ExecutionMode.from(Xmodestring,m_Xmode);
        if (asMacrodef() && m_name==null) {
            m_name = SIDs.next("foreach");
            m_generatedName = true;
        }
    }



    /**
     * Returns this task's execution mode. Never returns <i>null</i>;
     * will return <span class="src">ISOLATED</span> if never set.
     * @since JWare/AntX 0.5
     **/
    public ExecutionMode getMode()
    {
        return m_Xmode;
    }



    /**
     * @return True if this foreach should operation like a macrodef.
     * @since JWare/AntX 0.5
     **/
    private boolean asMacrodef()
    {
        return ExecutionMode.LOCAL.equals(getMode());
    }



    /**
     * Returns this task's implementation macrodef's name iff is
     * set to execute in local mode. Will return <i>null</i> if
     * set to work in isolated mode.
     * @since JWare/AntX 0.5
     **/
    public String getMacroNamesList()
    {
        return asMacrodef() ? getName()+"_macrodef" : null;
    }



    /**
     * Returns this task's step-name (can be the string "null").
     **/
    public String toString()
    {
        return getName();
    }



    /**
     * Allow any task to be nested within this task. Note that there
     * is no special processing for properties or reference.
     * @param task task to be added (non-null)
     **/
    public void addTask(Task task)
    {
        m_nestedTasks.add(task);
    }



    /**
     * Returns this taskset's list of nested tasks.
     **/
    protected final List getTasksList()
    {
        return m_nestedTasks;
    }



    /**
     * Returns a new array of this taskset's nested tasks.
     * The array is new, the tasks are live references.
     **/
    public Task[] getTasks()
    {
        return (Task[])getTasksList().toArray(Empties.EMPTY_TASK_ARRAY);
    }




    /**
     * Returns an overlay keeper that redirects to this taskset's
     * container. Never returns <i>null</i>.
     * @since JWare/AntX 0.4
     **/
    protected OverlayParametersHelper getParametersKeeper()
    {
        if (m_paramsHelper==null) {
            m_paramsHelper= new OverlayParametersHelper(getProject());
            m_paramsHelper.addOverride(this);
        }
        return m_paramsHelper;
    }



    /**
     * Always returns one (1) since the target steps are any tasks
     * nested within this task itself.
     **/
    protected final int getKindOfRunnablesSpecified()
    {
        return 1;
    }



    /**
     * Runs this task's nested tasks in order. Each task's
     * <span class="src">perform()</span> method is called.
     * @throws BuildException if any nested task does
     **/
    protected final void performTheTasksList()
    {
        Iterator itr = getTasksList().listIterator();
        while (itr.hasNext()) {
            Task nestedTask = (Task)itr.next();
            nestedTask.perform();
        }
    }
 


    /**
     * Ensures this foreach task has been named.
     * @throws BuildException if this task has no step-name, loop-
     *         control, or is not in a valid target.
     * @throws BuildException if is local mode and cannot install a
     *         matching macrodef.
     **/
    protected void verifyCanExecute_(String calr)
        throws BuildException
    {
        if (Tk.isWhitespace(getName())) {
            String ermsg = getAntXMsg("flow.task.needs.name",getTaskName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }

        super.verifyCanExecute_(calr);

        if (asMacrodef()) {
            if (!m_generatedName) {
                String ermsg = getAntXMsg("task.unsupported.attr",
                    getTaskName()+"(mode=local)","name");
                log(ermsg, Project.MSG_ERR);
                throw new BuildException(ermsg, getLocation());
            }
            installMacrodef();
        }
    }



    /**
     * Returns a list-of-one TargetCaller that calls back into this
     * foreach task via its '<i>run</i>' method.
     * @throws BuildException if unable to create target caller
     * @see #MAX_SEARCH_DEPTH
     **/
    protected List copyOfOrderedTargetCallers()
        throws BuildException
    {
        TargetCaller caller;
        
        if (asMacrodef()) {
            verify_(m_macrodefInstalled!=null,"getCalrs- macrodef inited");
            MacroInstanceCaller mcaller = new MacroInstanceCaller(this);
            mcaller.setTarget(m_macrodefInstalled);
            caller = mcaller;
        }
        else {
            String targetName = StepLauncherCaller.findSpecialTarget(this);
            caller = StepLauncherCaller.create(this,true,targetName,
                                          this.getName(),
                                          InlineForEachTask.class.getName(),
                                          MAX_SEARCH_DEPTH);
        }

        transferOverlayParameters(caller);

        List l = AntXFixture.newList(2);
        l.add(caller);

        return l;
    }



    /**
     * Performs this task's nested tasks from a StepLauncher's call.
     * Unused if mode is local.
     * @throws BuildException if any nested task does
     **/
    public void run() throws BuildException
    {
        performTheTasksList();
    }



    /**
     * Executes foreach taskset. If local inlined, will also try to
     * remove the macrodef we installed.
     * @since JWare/AntX 0.5
     **/
    public void execute()
    {
        boolean uninstallMacrodef = asMacrodef();
        super.execute();

        if (uninstallMacrodef) {
            String macroname = getMacroNamesList();
            ComponentHelper ch = ComponentHelper.getComponentHelper(getProject());
            Object was = ch.getAntTypeTable().remove(macroname);
            if (was!=null) {
                log("Removed macrodef("+macroname+") from type table",
                    Project.MSG_DEBUG);
            } else {
                log("Unable to remove macrodef("+macroname+") from type table",
                    Project.MSG_DEBUG);
            }
        }
    }



    /**
     * Installs a macrodef comprised of our internals. This installation
     * is done once in task's lifetime (on verify). If a foreach is run
     * repeatedly as part of a macrodef itself, it installs a <em>single</em>
     * macrodef the first time it is run. Subsequent calls to the foreach's
     * parent macrodef reuses the previously installed foreach macrodef. 
     **/
    private void installMacrodef()
    {
        if (m_macrodefInstalled==null) {
            String macroname = getMacroNamesList();

            ComponentHelper ch = ComponentHelper.getComponentHelper(getProject());
            if (ch.getDefinition(macroname)==null) {
                InlineForEachMacroDef macrodefFactory = new InlineForEachMacroDef();
                TaskExaminer.initTaskFrom(macrodefFactory,this);

                macrodefFactory.setName(macroname);
 
                Iterator itr = getTasksList().listIterator();
                while (itr.hasNext()) {
                    Task nestedTask = (Task)itr.next();
                    macrodefFactory.addTask(nestedTask);
                }
    
                log("Installing macrodef("+macrodefFactory.getName()+
                    ") as foreach implementation.",Project.MSG_DEBUG);
    
                macrodefFactory.perform();

            } else {
                log("Reusing existing macrodef("+macroname+
                    ") as foreach implementation.",Project.MSG_DEBUG);
            }
            m_macrodefInstalled = macroname;//NB: latch
        }
    }


    private FixturePassthru m_passthru = FixturePassthru.PROPERTIES;/* NB: Ant dflt */
    private String m_name;//NB:required
    private boolean m_generatedName;
    private List m_nestedTasks= AntXFixture.newList(10);
    private ExecutionMode m_Xmode= ExecutionMode.ISOLATED;
    private String m_macrodefInstalled;//NB:null=> macrodef might need installin'
}

/* end-of-InlineForEachTask.java */
