/**
 * $Id: CallerTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;
import  com.idaremedia.antx.parameters.RecoveryEnabled;

/**
 * Superclass of any task that can run either a set of nested tasksets or a set
 * of real top level targets . Whether the targets are run from an isolated sub-project
 * is subclass dependent. This class was originally named
 * "<span class="src">StepCaller</span>" but was renamed to eliminate the confusion
 * caused by the "Step" prefix. For simplicity in our documentation we refer to both
 * top-level targets and nested tasksets as "targets". The CallerTask class implements
 * basic support for the following common behavior:<ul>
 *   <li>Target verification (ensure named steps or top-level targets actually exist).</li>
 *   <li>The default fixture passthru options (like 'inheritRefs', and 'inheritAll').</li>
 *   <li>Implementation for basic property-based overlay instructions.</li>
 *   <li>The standard execution of named targets in order. The support is based on
 *       the {@linkplain TargetCaller} interface. Whether targets are nested or
 *       top-level is subclass defined.</li>
 *   <li>Problem handling when a called target generates an error. This support is
 *       based on the 'haltiferror' and 'tryeach' options. </li>
 * </ul>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @see      InlineStep
 * @see      TargetCaller
 **/

public abstract class CallerTask extends AssertableTask
    implements RecoveryEnabled
{
    /**
     * Initializes a new caller task.
     **/
    protected CallerTask()
    {
        super(AntX.flow+"call");
    }



    /**
     * Iniitializes a new, custom caller task.
     * @param iam CV-label
     **/
    protected CallerTask(String iam)
    {
        super(iam);
    }


    /**
     * Optionally delays configuration of nested tasks until they're
     * executed (like standard '<i>sequential</i>' task).
     * @see TaskExaminer#ignoreConfigure TaskExaminer.ignoreConfigure(&#8230;)
     * @since JWare/AntX 0.4
     **/
    public void maybeConfigure()
        throws BuildException
    {
        if (!TaskExaminer.ignoreConfigure(this)) {//@since AntX0.4+Ant1.6
            super.maybeConfigure();
        }
    }


// ---------------------------------------------------------------------------------------
// Script-facing Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets name of a property to update in event any target fails.
     * @param property the property's name (non-null)
     **/
    public void setFailProperty(String property)
    {
        require_(property!=null,"setFailProperty- nonzro name");
        m_failProperty = property;
    }


    /**
     * Returns name of a property to update in event any caller
     * fails. Returns <i>null</i> if never set.  This property is set
     * irregardless of this task's "haltiferror" setting.
     **/
    public final String getFailProperty()
    {
        return m_failProperty;
    }


    /**
     * Sets the name of a variable to update in event any
     * target caller fails.
     * @see #setFailValue
     * @since JWare/AntX 0.3
     **/
    public void setFailVariable(String variable)
    {
        require_(variable!=null,"setFailVariable- nonzro name");
        m_failVariable = variable;
    }



    /**
     * Returns name of the variable to update in event
     * any target caller fails. Returns <i>null</i> if never set.
     * @since JWare/AntX 0.3
     **/
    public final String getFailVariable()
    {
        return m_failVariable;
    }


    /**
     * Sets value used for {@linkplain #setFailProperty failproperty}
     * and/or {@linkplain #setFailVariable failvariable}
     * in event of failure.
     * @param value the value to be used (non-null)
     **/
    public void setFailValue(String value)
    {
        require_(value!=null,"setFailValu- nonzro valu");
        m_failValue = value;
    }


    /**
     * Returns value used for {@linkplain #setFailProperty failproperty}
     * and/or {@linkplain #setFailVariable failvariable}
     * in event of failure. Defaults to the string "true".
     **/
    public final String getFailValue()
    {
        return m_failValue;
    }



    /**
     * Sets whether this task will continue to unwind the call
     * stack and eventually fail with a top-level build exception.
     * Defaults to <i>true</i>.
     **/
    public void setHaltIfError(boolean balk)
    {
        m_haltIfError = balk;
    }


    /**
     * Returns <i>true</i> if this task will propagate any
     * build exceptions unconditionally. Defaults to <i>true</i>.
     **/
    public final boolean isHaltIfError()
    {
        return m_haltIfError;
    }


    /**
     * Sets whether this task will ignore any target failures
     * and continue executing subsequent targets.
     **/
    public void setTryEach(boolean tryEach)
    {
        m_tryEach = tryEach;
    }


    /**
     * Returns whether this task will ignore any target failures
     * and continue executing subsequent targets. Defaults to
     * <i>false</i>.
     **/
    public final boolean isTryEach()
    {
        return m_tryEach;
    }


//  ---------------------------------------------------------------------------------------
//  Execution:
//  ---------------------------------------------------------------------------------------


    /**
     * Returns this task's overlay instructions tracker. Never returns
     * <i>null</i>. Must be called after this task initialized.
     * @since JWare/AntX 0.4
     **/
    protected OverlayParametersHelper getParametersKeeper()
    {
        if (m_paramsHelper==null) {
            m_paramsHelper= new OverlayParametersHelper(getProject());
        }
        return m_paramsHelper;
    }



    /**
     * Returns <i>true</i> if this task has been assigned
     * specific overlay instructions.
     * @since JWare/AntX 0.4
     **/
    protected final boolean haveOverlayParameters()
    {
        if (m_paramsHelper==null) {
            return false;
        }
        return !m_paramsHelper.getParameters().isEmpty();
    }



    /**
     * Copies this task's fixture configuration instructions to the
     * target caller's environment.
     * @since JWare/AntX 0.4
     **/
    protected final void transferOverlayParameters(TargetCaller caller)
    {
        if (m_paramsHelper!=null) {
            m_paramsHelper.transferParameters(caller);
        }
    }



    /**
     * Factory-method that returns an ordered, filtered list of configured
     * step or top-level target callers. Subclasses must provide. Never
     * returns <i>null</i> but can return an empty list. Caller assumes
     * ownership of the returned list.
     * @see #getFilteredStepsLike getFilteredStepsLike(&#8230;)
     * @throws BuildException if unable to determine targets.
     **/
    protected abstract List copyOfOrderedTargetCallers()
        throws BuildException;



    /**
     * Template-method that executes this caller's targets in natural order.
     * @param selector [optional] callback to select which run method to use
     * @see #copyOfOrderedTargetCallers
     * @throws BuildException if unable to determine targets or any target
     *         does and "haltiferror" is <i>false</i>
     **/
    protected void runTargetCallers(TargetCaller.RunSelector selector)
        throws BuildException
    {
        List callers = copyOfOrderedTargetCallers();
        verify_(callers!=null,"runcalrs- nonzro calrs to run");

        if (!callers.isEmpty()) {
            Iterator itr= callers.listIterator();
            boolean started=false;
            TargetCaller caller=null;

            while (itr.hasNext()) {
                caller = (TargetCaller)itr.next();
                try {

                    callerStarted(caller);
                    started=true;
                    log("Caller for '"+callerName(caller)+
                        "' about to run.",Project.MSG_DEBUG);

                    if (selector!=null) {
                        selector.run(caller);
                    } else {
                        caller.run();
                    }

                    started=false;
                    callerFinished(caller,null);

                } catch(RuntimeException rtX) {
                    callerFailed(caller, rtX, callers);
                    if (started) {
                        callerFinished(caller,rtX);
                    }
                    if (isHaltIfError()) {
                        throw rtX;
                    } else if (!isTryEach()) {
                        break;
                    }
                }
            }//foreach

            callers.clear();

        }//!=0
    }



    /**
     * Generates a series of (possibly partitioned) calls to either
     * a set of top-level targets or a set of nested steps. By default
     * just calls {@linkplain #runTargetCallers runTargetCallers}.
     * @throws BuildException if any caller does.
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");
        runTargetCallers(null);
    }



    /**
     * Called before a target caller is executed. By default
     * does nothing.
     * @param caller the verified (about-to-be)started caller (non-null)
     **/
    protected void callerStarted(TargetCaller caller)
    {
        //do-nothing
    }



    /**
     * Called after a target caller returns (successfully or
     * not). By default does nothing.
     * @param caller the finished caller (non-null)
     * @param cause [optional] if non-null caller aborted with this
     **/
    protected void callerFinished(TargetCaller caller, Throwable cause)
    {
        //do-nothing
    }



    /**
     * Called if this caller catches a runtime exception while
     * executing a sequence of target callers. By default logs an
     * error and updates failproperty if one was requested.
     * @param caller the target caller that failed (non-null)
     * @param failure the barfage (non-null)
     * @param callers list of all targets being executed (non-null)
     **/
    protected void callerFailed(TargetCaller caller, Throwable failure,
                                List callers)
    {
        String targetName = callerName(caller);

        if (getMsgId()!=null) {
            log(getMsg(newMsgGetter(targetName)), Project.MSG_ERR);
        } else {
            log(uistrs().get("flow.runsteps.failure",getTaskName(),targetName),
                Project.MSG_ERR);
        }

        String what = getFailValue();

        if (getFailProperty()!=null) {
            checkIfProperty_(getFailProperty(),true);
            getProject().setNewProperty(getFailProperty(), what);
        }
        if (getFailVariable()!=null) {
            ExportedProperties.set(getFailVariable(), what);
        }
    }


// ---------------------------------------------------------------------------------------
// Determining(sp?) Call Targets:
// ---------------------------------------------------------------------------------------


    /**
     * Returns <i>true</i> if the named target exists within this
     * task's project.
     * @param targetName the name of candidate target (non-null)
     **/
    protected boolean targetExists(String targetName)
    {
        return getProject().getTargets().get(targetName)!=null;
    }


    /**
     * Returns a new list of all possible tasks-- not just steps. By
     * default returns task list of enclosing target. Never returns
     * <i>null</i>. Caller assumes ownership of returned array. Returned
     * list can include task placeholders (UnknownElements).
     **/
    protected Task[] getAllPossibleTasks()
    {
        return getOwningTarget().getTasks();
    }



    /**
     * Returns a new list of all possible steps of a particular class.
     * The filter class must be a kind-of {@linkplain InlineStep} class. Caller
     * assumes ownership of returned list. The returned list of tasks can
     * contain task placeholders (instead of true steps). It is assumed
     * that the caller is examining task attributes. If the caller needs
     * additional information, they will have to force the placeholders
     * to be replaced by true tasks.
     * @param thisClass required step class (non-null)
     * @.impl This method must not convert proxies to their real objects YET.
     *        If this conversion were done here, it's very likely to be
     *        premature as tasks subsequent to the this task may depend
     *        on side-effects of this task (like properties being defined,
     *        etc.)
     **/
    protected List getAllPossibleStepsLike(Class thisClass)
    {
        require_(InlineStep.class.isAssignableFrom(thisClass),
                 "getStpsLik- kindof(inlineStep)");
        Task[] tasks = getAllPossibleTasks();
        List l = AntXFixture.newList(tasks.length);
        for (int i=0;i<tasks.length;i++) {
            Class taskClass = TaskExaminer.trueClass(tasks[i]);//@.impl
            if (taskClass!=null && thisClass.isAssignableFrom(taskClass)) {
                l.add(tasks[i]);
            }
        }
        return l;
    }



    /**
     * Extracts a step's name from a placeholder task or if not a
     * placeholder the real step object.
     * @param t placeholder or step task (non-null)
     * @return the value of the 'name' attribute
     * @.impl  Expects Ant 1.6.1
     * @since  JWare/AntX 0.4
     **/
    private String getStepName(Object t)
    {
        if (t instanceof InlineStep) {
            return ((InlineStep)t).getName();
        }
        return TaskExaminer.trueAttr((Task)t,"name");
    }



    /**
     * Returns a new list of steps (of particular class) that match a
     * supplied name list. Caller assumes ownership of returned list. The
     * returned list of tasks can contain task placeholders (instead of
     * actual steps).
     * @param thisClass required step class (non-null)
     * @param filterList the set of step names (non-null)
     * @return list of matching steps
     * @see #getAllPossibleStepsLike getAllPossibleStepsLike(&#8230;)
     * @.impl See {@linkplain #getAllPossibleStepsLike getAllPossibleStepsLike(&#8230;)}
     **/
    protected final List getFilteredStepsLike(Class thisClass, List filterList)
    {
        require_(filterList!=null,"filterlst- nonzro list");
        List steps = getAllPossibleStepsLike(thisClass);
        if (!steps.isEmpty()) {
            Iterator itr= steps.iterator();
            while (itr.hasNext()) {
                String sn = getStepName(itr.next());
                if (!filterList.contains(sn)) {
                    itr.remove();
                }
            }
        }
        return steps;
    }



    /**
     * Utility method to get a useful name for caller.
     * @param caller the caller in question (non-null)
     **/
    static final String callerName(TargetCaller caller)
    {
        String name = caller.getStepName();
        if (name==null) {
            name = caller.getTargetName();
        }
        if (name==null) {
            name = "n/d";
        }
        return name;
    }



    OverlayParametersHelper m_paramsHelper;//NB: lazy-inited
    private boolean m_haltIfError= true;  /* NB: unrolls and barfs by default */
    private boolean m_tryEach=false;      /* NB: abort if any task fails */
    private String  m_failProperty, m_failVariable;
    private String  m_failValue = Strings.TRUE;
}

/* end-of-CallerTask.java */
