/**
 * $Id: CallOnceTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.helpers.InnerNameValuePair;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.ExecutionMode;
import  com.idaremedia.antx.parameters.FixturePassthru;
import  com.idaremedia.antx.solo.CopyPropertyTask;
import  com.idaremedia.antx.solo.CopyReferenceTask;

/**
 * The public task definition of {@linkplain OnceTask}; usually defined as &lt;call&gt;.
 * The CallOnceTask allows only &lt;propertyset&gt;, &lt;[copy]property&gt;, and
 * &lt;[copy]reference&gt; declarations to be nested within it as fixture instructions
 * for the child project.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;call steps="scrub,co,merge" failproperty="prepare.failed"&gt;
 *     &lt;property name="scrub.disable" value="false"/&gt;
 *     &lt;reference refid="copy.filters"/&gt;
 *   &lt/call&gt;
 *
 *   &lt;call targets="--clobber,--checkout,--merge" failproperty="prepare.failed"&gt;
 *     &lt;property name="scrub.disable" value="false"/&gt;
 *     &lt;property name="start.time" variable="local.start.time"/&gt;
 *   &lt/call&gt;
 * 
 *   &lt;call macros="scrub,co,merge"&gt;
 *      &lt;attribute name="project" value="${subproject}"/&gt;
 *      &lt;property name="scrub.disable" value="false"/&gt;
 *   &lt;/call&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.4
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,infra
 **/

public final class CallOnceTask extends OnceTask
{
    /**
     * Initializes a new CallOnceTask instance.
     **/
    public CallOnceTask()
    {
        super(AntX.flow+"call");
    }



    /**
     * Initializes a new helper CallOnceTask instance.
     * @param iam CV-label (non-null)
     **/
    public CallOnceTask(String iam)
    {
        super(iam);
    }


//  ---------------------------------------------------------------------------------------
//  Unique Script-facing Parameters:
//  ---------------------------------------------------------------------------------------

    /**
     * Adds a new fixture overlay property to this task.
     * This property is automatically passed to each target's
     * environment as it is executed.
     * @param param initialized property (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addProperty(CopyPropertyTask param)
    {
        require_(param!=null,"addProperty- nonzro item");
        getParametersKeeper().addParameter(param);
    }



    /**
     * Adds a new fixture overlay property set to this task. The
     * properties named by the property set are automatically
     * copied to  each target's environment before it is executed.
     * @param param property set information (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addPropertySet(PropertySet param)
    {
        require_(param!=null,"addPropertySet- nonzro item");
        getParametersKeeper().addParameter(param);
    }


    /**
     * Adds a new configuration reference to this task. This
     * reference is automatically copied to each target's
     * environment as each is executed.
     * @param param reference information (non-null)
     * @since JWare/AntX 0.4
     **/
    public void addReference(CopyReferenceTask param)
    {
        require_(param!=null,"addReference- nonzro item");
        getParametersKeeper().addParameter(param);
    }



    /**
     * Adds a new macro attribute default value to this task.
     * This attribute is automatically passed to ,em>each</em>
     * macro when it is executed.
     * @param attr initialized macro attribute (non-null)
     * @since JWare/AntX 0.5
     * @throws BuildException if this caller is not setup for macros.
     **/
    public void addConfiguredAttribute(InnerNameValuePair attr)
    {
        require_(attr!=null,"addAttribute- nonzro item");
        if (getMacroNamesList()==null) {
            String err = getAntXMsg("flow.attritem.formacros");
            log(err,Project.MSG_ERR);
            throw new BuildException(err,getLocation());
        }
        getParametersKeeper().addParameter(attr);
    }



    /**
     * Instructs this task what kinds of fixture informaton
     * should be passed to the child project. By default only
     * properties are passed on.
     * @since JWare/AntX 0.4
     **/
    public void setPassthru(FixturePassthru passthru)
    {
        require_(passthru!=null,"setPassthru- nonzro option");
        m_passthru = FixturePassthru.from(passthru.getIndex());//NB:normalize
    }



    /**
     * Returns the fixture passthru setting to be used when this
     * caller runs its targets. Never returns <i>null</i>;
     * will return <span class="src">PROPERTIES</span> if
     * not set explicitly.
     * @since JWare/AntX 0.4
     **/
    public final FixturePassthru getPassthruOption()
    {
        return m_passthru;
    }



    /**
     * Sets this task's list of steps. These steps will be
     * called in order from a partitioned child project
     * when this task is executed.
     * @param setOfStepNames the comma-delimited list of names
     * @since JWare/AntX 0.4
     **/
    public void setSteps(String setOfStepNames)
    {
        require_(!Tk.isWhitespace(setOfStepNames),
                 "setStps- nonwspc list");
        m_stepNames = setOfStepNames;
    }


    /**
     * Returns this task's comma-delimited list of step names.
     * Will return <i>null</i> if never set. By default, all
     * steps must exist within the same target as this task.
     * @since JWare/AntX 0.4
     **/
    public String getStepNamesList()
    {
        return m_stepNames;
    }



    /**
     * Sets this task's list of targets. These targets will
     * be called in order from a partitioned child project
     * when this task is executed.
     * @param setOfTargetNames the comma-delimited list of names
     * @since JWare/AntX 0.4
     **/
    public void setTargets(String setOfTargetNames)
    {
        require_(!Tk.isWhitespace(setOfTargetNames),
                 "setTrgs- nonwspc list");
        m_targetNames = setOfTargetNames;
    }


    /**
     * Returns this task's comma-delimited list of target names.
     * Will return <i>null</i> if never set.
     * @since JWare/AntX 0.4
     **/
    public String getTargetNamesList()
    {
        return m_targetNames;
    }



    /**
     * Sets this task's list of macros. These macros will be
     * called in order from the current project's partition when
     * this task is executed.
     * @param setOfMacroNames the comma-delimited list of macros
     * @since JWare/AntX 0.5
     * @.sideeffect Will default mode to "local" if not set.
     **/
    public void setMacros(String setOfMacroNames)
    {
        require_(!Tk.isWhitespace(setOfMacroNames),
                 "setMacros- nonwspc list");
        m_macroNames = setOfMacroNames;
        if (!m_modeInited) {
            setMode(ExecutionMode.LOCAL.getValue());
        }
    }


    /**
     * Returns this task's comma-delimited list of macro names.
     * Will return <i>null</i> if never set. By default, all
     * macros must exist within the same project as this task.
     * @since JWare/AntX 0.5
     **/
    public String getMacroNamesList()
    {
        return m_macroNames;
    }



    /**
     * Tells this tasks to execute items locally (within current
     * project).
     * @param Xmodestring either "local" or "isolated"
     * @since JWare/AntX 0.4
     **/
    public void setMode(String Xmodestring)
    {
        require_(Xmodestring!=null,"setMode- nonzro string");
        m_Xmode = ExecutionMode.from(Xmodestring,m_Xmode);
        m_modeInited = true;//latch
    }



    /**
     * Returns this task's execution mode. Never returns <i>null</i>;
     * will return <span class="src">ISOLATED</span> if never set.
     * @since JWare/AntX 0.4
     **/
    public ExecutionMode getMode()
    {
        return m_Xmode;
    }



    //Required sequence of steps (can be specified by subclass "later")
    //Default way of specifying a set of steps is using a comma-delimited list
    //in "steps", "macros", or "targets" attribute.
    private String m_stepNames;
    private String m_targetNames;
    private String m_macroNames;
    private FixturePassthru m_passthru = FixturePassthru.PROPERTIES;/* NB: Ant dflt */
    private ExecutionMode m_Xmode= ExecutionMode.ISOLATED;
    private boolean m_modeInited;
}

/* end-of-CallOnceTask.java */
