/**
 * $Id: RepeatedTaskSet.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.condition.solo.RulesTk;
import  com.idaremedia.antx.helpers.GenericParameters;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.ScopedProperties;
import  com.idaremedia.antx.parameters.FeedbackLevel;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * Taskset that is executed as long as a named condition evaluates true. Usually defined
 * as <span class="src">&lt;dowhile&gt;</span>.
 * <p>
 * While the RepeatedTaskSet can provide a set of overlaid properties to its test,
 * unlike the <span class="src">foreach</span> type loop tasks, a RepeatedTaskSet does
 * not (optionally) isolate its sub-tasks' execution from the current project. This
 * means that a RepeatedTaskSet nested tasks can always access (and alter) fixture
 * elements created and/or modified by the loop criteria task. (Of course you can
 * manually isolate the nested tasks inside an AntX
 * <span class="src">&lt;isolate&gt;</span> taskset.)
 * <p>
 * Setting a RepeatedTaskSet quiet does not affect(ie&#46; default) the feedback
 * levels of its nested tasks.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;criteria id="moreLogs"...&gt;
 *       ...
 *    &lt;<b>dowhile</b> test="moreLogs" maxloops="25" haltifmax="yes"&gt;
 *       ...<i>[Tasks to process each log (next in $var:nextlogfile)]</i>
 *    &lt;/<b>dowhile</b>&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,infra
 **/

public class RepeatedTaskSet extends TaskSet implements HardLimitEnabled
{
    /**
     * Initializes a new RepeatedTaskSet instance.
     **/
    public RepeatedTaskSet()
    {
        super(AntX.flow+"RepeatedTaskSet");
    }



    /**
     * Initializes a new RepeatedTaskSet subclassed instance.
     * @param iam CV-label (non-null)
     **/
    protected RepeatedTaskSet(String iam)
    {
        super(iam);
    }



    /**
     * Defines the test for this repeat task. The referred to
     * object must implement either the <span class="src">Condition</span>
     * or <span class="src">ShareableCondition</span> interface.
     * The reference is verified when this task is executed.
     * @param testRef reference of test definition (non-null)
     **/
    public void setCriteria(Reference testRef)
    {
        require_(testRef!=null,"setTest- nonzro test ref");
        m_conditionId = testRef;
    }



    /**
     * Synonym for {@linkplain #setCriteria(Reference) setCriteria}.
     * @param testRef reference to test definition (non-null)
     **/
    public final void setTest(Reference testRef)
    {
        setCriteria(testRef);
    }



    /**
     * Returns this task's test's reference. Will return
     * <i>null</i> if never set explicitly. Must be defined before
     * this task is executed.
     **/
    public final String getCriteria()
    {
        return m_conditionId.getRefId();
    }



    /**
     * Gives this loop task a hard limit to the number of times
     * it can execute. This attribute serves mostly as an error
     * handling mechanism to prevent infinite loops.
     **/
    public void setMaxLoops(int maxLoops)
    {
        m_maxLoops = maxLoops>=0 ? maxLoops : Integer.MAX_VALUE;
    }



    /**
     * Returns <i>true</i> if this loop tasks has a hard limit.
     **/
    public final boolean hasMax()
    {
        return m_maxLoops!=Integer.MAX_VALUE;
    }



    /**
     * Returns this loop tasks hard limit or <span class="src">-1</span>
     * if no such limit exists.
     **/
    public final int getMaxLoops()
    {
        return hasMax() ? m_maxLoops : -1;
    }



    /**
     * Tells this tasks whether to fail if the hard limit is
     * reached. Ignored if this loop task does not have a hard limit.
     * @see #setMaxLoops setMaxLoops(LIMIT)
     **/
    public void setHaltIfMax(boolean halt)
    {
        m_haltIfMax = halt ? Boolean.TRUE : Boolean.FALSE;
    }



    /**
     * Returns this task's fail-if-max option setting. Will
     * return <i>null</i> if never set explicitly. By default this
     * task will <em>not</em> fail if its hard limit is hit; it
     * just stops.
     **/
    public final Boolean getHaltIfMax()
    {
        return m_haltIfMax;
    }



    /**
     * Tells this task how much non-diagnostic feedback to generate.
     * Really only has "loud" vs. "quiet-ish" interpretation. If
     * set quiet, this task will not issue a warning if it hits a
     * hard limit.
     * @param level feedback level (non-null)
     **/
    public void setFeedback(String level)
    {
        require_(level!=null,"setFeedback- nonzro level");
        FeedbackLevel fbl = FeedbackLevel.from(level);
        if (fbl==null) {
            String e = getAntXMsg("task.illegal.param.value",
                           getTaskName(), level,"feedbacklevel");
            log(e, Project.MSG_ERR);
            throw new BuildException(e, getLocation());
        }
        m_fbLevel = fbl;
    }



    /**
     * Returns this task's assigned feedback level. Will return
     * <i>null</i> if never set explicitly.
     **/
    public final FeedbackLevel getFeedbackLevel()
    {
        return m_fbLevel;
    }




    /**
     * Tells this loop task to overlay its current loop cursor value
     * as a property with the given name when its test is called. Unlike
     * the foreach type loops, this is <em>not</em> a required
     * parameter for repeated tasksets.
     * @param cursorName the loop cursor name (as seen by the test)
     **/
    public void setI(String cursorName)
    {
        require_(cursorName!=null,"setI- nonzro name");
        m_iName = cursorName;
    }



    /**
     * Returns this loop's cursor overlay property's name. Will
     * return <i>null</i> if not set explicitly.
     **/
    public final String getCursorName()
    {
        return m_iName;
    }



    /**
     * Assigns an evaluation context to this loop task. How the
     * context is used by the test is unknown to this task.
     * @param refid reference to collection of key-value settings
     * @see com.idaremedia.antx.solo.PropertiesList PropertiesList
     **/
    public void setContext(Reference refid)
    {
        m_evalContext = refid;
    }



    /**
     * Returns this task's evaluation context. Will return <i>null</i>
     * if no context was assigned explicitly.
     **/
    public final Reference getContext()
    {
        return m_evalContext;
    }



    /**
     * Ensures we have been given a test reference and that it is
     * valid. Will also verify that we do not have a always fail
     * situation.
     * @throws BuildException if improperly configured.
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (m_conditionId==null) {
            String error = getAntXMsg("task.needs.this.attr",
                getTaskName(),"criteria");
            log(error, Project.MSG_ERR);
            throw new BuildException(error,getLocation());
        }
        String testId = m_conditionId.getRefId();
        RulesTk.verifyTest(testId,m_rqlink);

        if (m_maxLoops==0 && m_haltIfMax==Boolean.TRUE) {
            if (getFeedbackLevel()!=FeedbackLevel.NONE) {//?always-warn?
                String e = getAntXMsg("flow.loop.alwaysfail");
                log(e, Project.MSG_WARN);
            }
        }
    }



    /**
     * Performs this task's nested elements until its named criteria
     * is no longer met.
     * @throws BuildException if verification fails.
     * @throws BuildException if any nested task does.
     * @throws BuildException if hard limit it and haltifmax is turned on.
     **/
    protected void performNestedTasks() throws BuildException
    {
        String testId = m_conditionId.getRefId();
        int i=0;
        final int n=m_maxLoops;

        while (i<n) {
            boolean continu;
            ScopedProperties overlay = installOverlay(i,getContext());
            try {
                continu = RulesTk.evalTest(testId,m_rqlink);
            } finally {
                uninstallOverlay(overlay);
                overlay = null;
            }
            if (!continu) {
                break;
            }
            performIterationOfTheTasksList();
            i++;
        }

        if (i==n) {
            String msg = getMsg();
            if (Tk.isWhitespace(msg)) {
                msg = getAntXMsg("flow.loop.overflow",String.valueOf(i));
            }
            if (m_haltIfMax==Boolean.TRUE) {
                log(msg, Project.MSG_ERR);
                throw new BuildException(msg, getLocation());
            }
            if (!FeedbackLevel.isQuietish(m_fbLevel,false)) {
                log(msg, Project.MSG_WARN);
            }
        }
    }



    /**
     * Installs all of our context properties and switch value
     * as an overlay before calling condition.
     **/
    private ScopedProperties installOverlay(int i, Reference context)
    {
        Project P= getProject();
        ScopedProperties overlay=null;

        //NB: Ordering is important; let the switch.value prevail!
        if (context!=null) {
            Object o = P.getReference(context.getRefId());
            if (o instanceof GenericParameters) {
                overlay = new ScopedProperties(P,true);
                overlay.put((GenericParameters)o);
            }
        }
        if (m_iName!=null) {
            if (overlay==null) {
                overlay = new ScopedProperties(P,true);
            }
            overlay.put(m_iName, String.valueOf(i));
        }
        if (overlay!=null) {
            overlay.install();
        }
        return overlay;
    }




    /**
     * Undoes the effect of {@linkplain #installOverlay installOverlay}.
     **/
    private void uninstallOverlay(ScopedProperties overlay)
    {
        if (overlay!=null) {
            overlay.uninstall(null);
            overlay=null;
        }
    }



    /**
     * Resets this taskset as if never configured. Allows this taskset
     * to be re-run with different parameters and nested elements.
     **/
    public void unconfigure()
    {
        super.unconfigure();
        m_conditionId = null;
        m_maxLoops = Integer.MAX_VALUE;
        m_haltIfMax = null;
        m_fbLevel = null;
        m_iName = null;
        m_evalContext = null;
    }


    private Requester m_rqlink = new Requester.ForComponent(this);
    private Reference m_conditionId;
    private int m_maxLoops = Integer.MAX_VALUE;
    private Boolean m_haltIfMax;//NB: *NO*
    private FeedbackLevel m_fbLevel;
    private String m_iName;//NB: null => none
    private Reference  m_evalContext;
}


/* end-of-WhileTask.java */