/**
 * $Id: LJDiagnosticsEmitter.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2003 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                                 EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  org.apache.log4j.Level;
import  org.apache.log4j.Logger;
import  org.apache.log4j.LogManager;
import  org.apache.log4j.spi.LoggerFactory;

import  com.idaremedia.apis.DiagnosticsEmitter;
import  com.idaremedia.apis.DiagnosticsEmitterFactory;

import  com.idaremedia.antx.NoiseLevel;

/**
 * Implementation of a {@linkplain DiagnosticsEmitter DiagnosticsEmitter} that uses a
 * <a href="http://jakarta.apache.org/log4j">log4j</a> category as the underlying
 * output-generating mechanism.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,infra
 **/

public class LJDiagnosticsEmitter implements DiagnosticsEmitter
{
    /**
     * Creates a new root diagnostics emitter. This emitter uses
     * the LogManager's root logger.
     **/
    public LJDiagnosticsEmitter()
    {
        m_logger = LogManager.getRootLogger();
    }


    /**
     * Creates a new diagnostics emitter for an existing Log4j logger.
     * @param logger the underlying logger (non-null)
     **/
    public LJDiagnosticsEmitter(Logger logger)
    {
        if (logger==null) {
            throw new IllegalArgumentException();
        }
        m_logger = logger;
    }


    /**
     * Creates a new diagnostics emitter for a particular grouping.
     * @param grpId this emitters log4j category (non-null)
     **/
    public LJDiagnosticsEmitter(String grpId)
    {
        if (grpId==null) {
            throw new IllegalArgumentException();
        }
        m_logger = LogManager.getLogger(grpId);
    }


    /**
     * Creates a new diagnostics emitter for a particular grouping
     * using a special Logger factory to create the underlying Logger.
     * @param lf the Logger factory (non-null)
     * @param grpId the target class (non-null)
     **/
    public LJDiagnosticsEmitter(LoggerFactory lf, String grpId)
    {
        if (grpId==null || lf==null) {
            throw new IllegalArgumentException();
        }
        m_logger = LogManager.getLogger(grpId, lf);
    }



    /**
     * Converts an Antisque msg-level to its log4j equivalent.
     **/
    public static final Level levelFrom(int nl, Level dflt)
    {
        switch (nl) {
            case NoiseLevel.FATAL_INDEX:
                return Level.FATAL;
            case NoiseLevel.ERROR_INDEX:
                return Level.ERROR;
            case NoiseLevel.WARNING_INDEX:
                return Level.WARN;
            case NoiseLevel.INFO_INDEX:
                return Level.INFO;
            case NoiseLevel.VERBOSE_INDEX:
                return Level.INFO;
            case NoiseLevel.DEBUG_INDEX:
                return Level.DEBUG;
            default:
                return dflt;
        }
    }


    /**
     * Determines if this emitter will ignore requests at given
     * noise level.
     **/
    public boolean emits(int level)
    {
        return m_logger.isEnabledFor(levelFrom(level,Level.INFO));
    }


    /**
     * Determines if this emitter will ignore requests at given
     * Log4J Level.
     **/
    public boolean emits(Level w)
    {
        return m_logger.isEnabledFor(w);
    }


    /**
     * Returns the underlying Logger's category name.
     * @since JWare/AntX 0.3b4
     **/
    public String identifier()
    {
        return m_logger.getName();
    }



    /**
     * Records a user-categorized message and an optional call
     * stack.
     * @since JWare/AntX 0.3b4
     **/
    public void report(int nl, Object msg, Throwable t)
    {
        m_logger.log(FQCN,levelFrom(nl,Level.INFO),msg,t);
    }



    /**
     * Records an unrecoverable failure has occured.
     **/
    public void failure(Object msg)
    {
        m_logger.log(FQCN,Level.FATAL,msg,null);
    }


    /**
     * Records an unrecoverable failure has occured including the
     * stack trace of given exception.
     **/
    public void failure(Throwable t, Object msg)
    {
        m_logger.log(FQCN,Level.FATAL,msg,t);
    }




    /**
     * Records an unexpected problem condition has occured.
     **/
    public void error(Object msg)
    {
        m_logger.log(FQCN,Level.ERROR,msg,null);
    }


    /**
     * Records an unexpected problem condition has occured including
     * the call stack of given exception.
     **/
    public void error(Throwable t, Object msg)
    {
        m_logger.log(FQCN,Level.ERROR,msg,t);
    }



    /**
     * Records an atypical but locally recoverable problem has occured.
     **/
    public void warning(Object msg)
    {
        m_logger.log(FQCN,Level.WARN,msg,null);
    }


    /**
     * Records an atypical but locally recoverable problem has occured
     * including the given stack trace informatin.
     **/
    public void warning(Throwable t, Object msg)
    {
        m_logger.log(FQCN,Level.WARN,msg,t);
    }



    /**
     * Records an alert notification. This level is not natively supported
     * by log4j; for now mapped to {@linkplain #warning(Object) warning}.
     **/
    public void alert(Object msg)
    {
        m_logger.log(FQCN,Level.WARN,msg,null);
    }


    /**
     * Records an alert notification including stack trace of
     * given exception. This level is not natively supported by log4j; for
     * now mapped to {@linkplain #warning(Throwable,Object) warning}.
     **/
    public void alert(Throwable t, Object msg)
    {
        m_logger.log(FQCN,Level.WARN,msg,t);
    }


    /**
     * Records a status or progress information message.
     **/
    public void note(Object msg)
    {
        m_logger.log(FQCN,Level.INFO,msg,null);
    }



    /**
     * Records an internal diagnostics message.
     **/
    public void trace(Object msg)
    {
        m_logger.log(FQCN,Level.DEBUG,msg,null);
    }


    /**
     * Records an internal diagnostics message including stack
     * trace of given exception.
     **/
    public void trace(Throwable t, Object msg)
    {
        m_logger.log(FQCN,Level.DEBUG,msg,t);
    }


    /**
     * Records a very fine-grained trace message. For AntX,
     * same as a {@linkplain #trace(Object) trace}.
     **/
    public final void finetrace(Object msg)
    {
        trace(msg);
    }


    /**
     * Records a very fine-grained trace message including
     * stack trace of given exception. For AntX, same as
     * {@linkplain #trace(Throwable,Object) trace}.
     **/
    public final void finetrace(Throwable t, Object msg)
    {
        trace(t,msg);
    }



    /**
     * Returns the underlying logger for this emitter object.
     * Never returns <i>null</i>.
     **/
    public final Logger getLogger()
    {
        return m_logger;
    }


    /**
     * Returns the fully-qualified-class-name of this emitter's
     * wrapper class (or if no wrapper, this class's name).
     **/
    public final String getFQCN()
    {
        return FQCN;
    }


    /**
     * Initializes the fully-qualified-class-name of this emitter's
     * wrapper class. Should be done once at (wrapper's) construction time.
     * @param fqcn the class name (non-null)
     **/
    public final void setFQCN(String fqcn)
    {
        if (fqcn==null) {
            throw new IllegalArgumentException();
        }
        FQCN = fqcn;;
    }


    /**
     * Builtin factory for instance of LJDiagnostisEmitters.
     **/
    public static final DiagnosticsEmitterFactory FACTORY=
        new DiagnosticsEmitterFactory() {
                public DiagnosticsEmitter newEmitter() {
                    return new LJDiagnosticsEmitter();
                }
                public DiagnosticsEmitter newEmitter(String grpId) {
                    return new LJDiagnosticsEmitter(grpId);
                }
            };


    /** The underlying log4j logger stuff. **/
    private final Logger m_logger;
    private String FQCN = LJDiagnosticsEmitter.class.getName();
}

/* end-of-LJDiagnosticsEmitter.java */
