/**
 * $Id: EmitLogsTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2003 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.starters.TaskSet;

/**
 * Bridge between Ant's build monitoring infrastructure and Apache's log4j.
 * <p>
 * <b>Examples:</b><pre>
 *    &lt;emitlogs with="log4j.config" labels="log4j.groups"&gt;
 *        &lt;echo message="foo"/&gt;
 *        &lt;property name="p" value="foo"/&gt;
 *    &lt;/emitlogs&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      EventEmitConduit
 **/

public class EmitLogsTask extends TaskSet implements EmitConfigurable
{
    /**
     * Initializes a new EmitLogsTask instance.
     **/
    public EmitLogsTask()
    {
        super(AntX.feedback);
    }


    /**
     * Returns the fallback emit configuration this task will
     * use to generate diagnostics emitters, etc. Never returns
     * <i>null</i>; usually an enclosing emit configure task.
     **/
    public EmitConfiguration getDefaults()
    {
        return EmitContext.getConfigurationNoNull();
    }


    /**
     * Assigns a primary emit configuration definition to this
     * task. This task will use this configuration instead of
     * any inherited context configuration.
     * @param r the {@linkplain EmitConfiguration} reference
     **/
    public void setWith(Reference r)
    {
        require_(r!=null,"setWith- nonzro refid");
        m_withEC= r;
    }


    /**
     * Returns this task's user-defined emit configuration or
     * <i>null</i> if none defined.
     * @see #setWith
     **/
    public final Reference getWith()
    {
        return m_withEC;
    }


    /**
     * Sets a custom grouping mapper for messages captured
     * by this task.
     * @param r the {@linkplain GroupingMapper} reference
     **/
    public void setLabels(Reference r)
    {
        require_(r!=null,"setLabels- nonzro refid");
        m_withLabels = r;
    }


    /**
     * Returns this task's user-defined grouping mapper or
     * <i>null</i> if none defined.
     **/
    public final Reference getLabelsMapper()
    {
        return m_withLabels;
    }


    /**
     * Returns this task' effective GroupingMapper object. Will
     * return <i>null</i> if no custom mapper defined.
     **/
    protected GroupingMapper getGroupingMapper()
    {
        if (getLabelsMapper()!=null) {
            return (GroupingMapper)getReferencedObject
                (getProject(), getLabelsMapper().getRefId(), GroupingMapper.class);
        }
        return null;
    }



    /**
     * Sets a grouping hierarchy filter for this task. This filter
     * only affects the standard main fields: project, target, and
     * task. Indicator zones are always defined.
     **/
    public void setIncludes(String filterstring)
    {
        require_(filterstring!=null,"setInc- nonzro mask");
        m_groupingFilter = filterstring;
    }


    /**
     * Returns this task's grouping hierarchy filter. Will return
     * "default" if never explicitly defined.
     **/
    public String getIncludes()
    {
        return m_groupingFilter;
    }


    /**
     * Returns the effective EmitConfiguration used by this
     * task. Never returns <i>null</i>.
     * @see #setWith
     * @see #getDefaults
     * @.safety single
     **/
    protected EmitConfiguration getConfiguration()
    {
        if (m_ECInstance!=null) {
            return m_ECInstance;
        }
        if (getWith()!=null) {
            m_ECInstance = (EmitConfiguration)getReferencedObject
                (getProject(), getWith().getRefId(), EmitConfiguration.class);
        }
        else {
            m_ECInstance = getDefaults();
        }
        return m_ECInstance;
    }


    /**
     * Installs an event->emit conduit, performs nested tasks,
     * and removes conduit.
     **/
    protected void performNestedTasks()
    {
        Project P = getProject();
        EventEmitConduit conduit = getWorker();
        try {
            verify_(!m_isInstalled,"perform- not installed");
            verify_(!P.getBuildListeners().contains(conduit),"perform- not installed");
            conduit.setConfiguration(getConfiguration());
            conduit.setIncludes(getIncludes());
            conduit.setGroupingMapper(getGroupingMapper());
            P.addBuildListener(conduit);
            m_isInstalled = true;
            performTheTasksList();
        } finally {
            if (m_isInstalled) {
                m_isInstalled = false;
                P.removeBuildListener(conduit);
                conduit.setConfiguration(null);
            }
        }
    }


    /**
     * Returns the Ant logs -> Log4J logs transmorgifier helper.
     * Never returns <i>null</i>.
     **/
    protected final EventEmitConduit getWorker()
    {
        return m_emitWorker;
    }


    private Reference m_withEC;
    private EmitConfiguration m_ECInstance;
    private volatile boolean m_isInstalled;
    private final EventEmitConduit m_emitWorker = new EventEmitConduit();
    private String m_groupingFilter=Strings.DEFAULT;
    private Reference m_withLabels;
}

/* end-of-EmitLogsTask.java */
