/**
 * $Id: EmitContext.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureAdministrator;
import  com.idaremedia.antx.FixtureCore;
import  com.idaremedia.antx.FixtureOverlays;
import  com.idaremedia.antx.FixtureIds;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.KillMethodSkeleton;
import  com.idaremedia.antx.apis.ProblemHandler;

/**
 * Manages iteration-based EmitConfiguration for the active thread. All context-aware
 * emit configurables should use this class instead of directly accessing FixtureOverlays\
 * stacks. The default emit configurations are bound to the current iteration; if the
 * iteration is replaced, a new (empty) default configuration is installed.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,infra
 * @see      EmitTask
 * @see      EmitConfigureTask
 **/

public final class EmitContext implements FixtureCore, FixtureAdministrator
{
    /**
     * The fixture component id for all EmitContext information.
     **/
    public static final String FXID= FixtureIds.EMIT_CONFIGURATION;



    /**
     * Returns the current thread's EmitConfiguration instance if one exists.
     * Returns <i>null</i> if no configuration installed for current thread.
     **/
    public static EmitConfiguration getConfiguration()
    {
        return (EmitConfiguration)FixtureOverlays.getContextInstance().nearest(FXID);
    }



    /**
     * Helper to {@linkplain #getConfiguration getConfiguration} that returns
     * the {@linkplain #getDefaultConfiguration default configuration} if the
     * current thread has no explicit configuration. Never returns <i>null</i>.
     **/
    public static EmitConfiguration getConfigurationNoNull()
    {
        EmitConfiguration ec = getConfiguration();
        return (ec!=null) ? ec : getDefaultConfiguration();
    }



    /**
     * Installs a new EmitConfiguration instance for the current thread. This
     * configuration becomes the active (and only visible) configuration until
     * it is unwound or another configuration is installed.
     * @param ec the new configuration (non-null)
     * @param noInstallHandler [optional] used in event of error
     * @return previous configuration if any (can be null)
     * @throws BuildException if incoming configuration already on iteration stack
     **/
    public static EmitConfiguration installConfiguration(EmitConfiguration ec,
                                                   ProblemHandler noInstallHandler)
    {
        if (ec==null) {
            throw new IllegalArgumentException(AntX.uistrs().get("emit.null.ecfg"));
        }
        String whoami = AntX.uistrs().dget("emit.whoami","EmitConfiguration");
        return (EmitConfiguration)FixtureOverlays.installIfNot(FXID,ec,noInstallHandler,
                                                         whoami);
    }



    /**
     * Removes the most recently installed EmitConfiguration for the current
     * thread. The previous installation is reactivated, or if this was
     * the only configuration, the current thread's emit configuration is
     * undefined (should use default if necessary).
     * @param noUninstallHandler used in event of already unwound stack
     **/
    public static void unwindConfiguration(ProblemHandler noUninstallHandler)
    {
        String whoami = AntX.uistrs().dget("emit.whoami","EmitConfiguration");
        FixtureOverlays.uninstallIfIs(FXID,noUninstallHandler,whoami);
    }



    /**
     * Returns the default EmitConfiguration used by this context. Never
     * returns <i>null</i>.
     * @see DefaultEmitConfiguration#INSTANCE
     * @see #setDefaultConfiguration
     * @since JWare/AntX 0.3
     **/
    public static EmitConfiguration getDefaultConfiguration()
    {
        EmitConfiguration ec = (EmitConfiguration)Iteration.getProperty(FXID);
        return (ec==null) ? DefaultEmitConfiguration.INSTANCE : ec;
    }



    /**
     * Initializes the default EmitConfiguration returned by this context. Not
     * necessary since by default, is set to the predefined {@linkplain
     * DefaultEmitConfiguration#INSTANCE DefaultEmitConfiguration}. Should not
     * be set to a configuration that already exists on current thread's iteration
     * stack. This method <em>replaces</em> any pre-existing default
     * configuration-- the current object is lost unless the caller maintains it.
     * @since JWare/AntX 0.3
     * @.safety guarded
     **/
    public static EmitConfiguration setDefaultConfiguration(EmitConfiguration ec)
    {
        if (ec==null) {
            throw new IllegalArgumentException
                (AntX.uistrs().get("emit.null.ecfg"));
        }
        return (EmitConfiguration)Iteration.setProperty(FXID,ec);
    }



    /**
     * Returns <i>true</i> if the default EmitConfiguration is defined as
     * the empty placeholder.
     * @see #setDefaultConfiguration setDefaultConfiguration()
     * @since JWare/AntX 0.4
     **/
    public static final boolean isDefaultUndefined()
    {
        return getDefaultConfiguration()==DefaultEmitConfiguration.INSTANCE;
    }



    /**
     * Resets the default EmitConfiguration to the empty placeholder. Your
     * scripts and application should never need to do this. This method is
     * provided for test scripts and harnesses.
     * @.safety guarded
     * @since JWare/AntX 0.4
     **/
    static final void unsetDefaultConfiguration()
    {
        Iteration.removeProperty(FXID);
    }



    /**
     * Installs a test-harness helper that clears up the various EmitContext
     * fixture components. Your application should never use this helper
     * directly; it is provided so that test harnesses can reset the environment
     * to a known state.
     * @since JWare/AntX 0.4
     **/
    static {
        AntXFixture.setKillMethod
            (FXID,
             new String[]{"emitconfigurations"},
             new KillMethodSkeleton() {
                     protected boolean killDefault(ProblemHandler from) {
                         EmitContext.unsetDefaultConfiguration();
                         return true;
                     }
                     protected String getComponentId() {
                         return EmitContext.FXID;
                     }
                     protected String getComponentName() {
                         return AntX.uistrs().dget
                             ("emit.whoami", "EmitConfiguration");
                     }
                 }
             );
    }


    /** Disallow; only static helpers. **/
    private EmitContext()
    { }
}

/* end-of-EmitContext.java */
