/**
 * $Id: EmitConfigurationType.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2003 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  java.text.DateFormat;
import  java.text.FieldPosition;
import  java.text.SimpleDateFormat;
import  java.util.Date;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.apis.DiagnosticsEmitter;
import  com.idaremedia.apis.DiagnosticsEmitterFactory;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.helpers.DateTimeFormat;
import  com.idaremedia.antx.helpers.Setting;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Holder of a fixed set of EmitConfiguration bits; usually defined &lt;emitconfiguration&gt;.
 * This type allows users to create shareable configuration for various emit-aware tasks.
 * Any attributes not defined or defined as "inherited" return the values returned by the
 * configuration's default configuration (which is either the build-iteration fixture's
 * default configuration or the configuration referred-to by the <i>defaults</i> parameter).
 * The build-iteration default configuration is often the fixed {@linkplain
 * DefaultEmitConfiguration log4j configuration}.
 * <p>
 * Because configurations are referred-to by multiple executing tasks, a configuration "own"
 * APIs return only the explicit values assigned to their instance. These methods will
 * not look at the build-iteration context. However, the public {@linkplain EmitConfiguration}
 * APIs might read the build-iteration's context if told too (by using the "enclosing" value
 * for the <i>wrt</i> parameter). Users should be careful when setting a configuration to look
 * at the enclosing context because that context in turn cannot be setup to reference
 * that same configuration (an infinite loop will occur).
 * <p>
 * <b>Examples:</b><pre>
 *   &lt;<b>emitconfiguration</b> id="emit.defaults"
 *       from="AntX"
 *       wrt="longdate"
 *       noiselevel="info"
 *       echo="yes"
 *   /&gt;
 *   &lt;<b>emitconfiguration</b> id="emit.checkpoint"
 *       wrt="enclosing"
 *       echo="no"
 *       timestampFormat="short"
 *   /&gt;
 *   &lt;<b>emitconfiguration</b> id="goal.documents"
 *       wrt="enclosing"
 *       from="Documentation"
 *       timestamp="no"
 *       echo="no"
 *   /&gt;
 *   &lt;<b>emitconfiguration</b> id="emit.defaults.jdk14log"
 *       from="AntX/builds/nightly"
 *       noiselevel="info"
 *       emitterfactory="mycompany.antx.Jdk14Emitter"
 *       groupingseparator="/"
 *   /&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded (once fully configured)
 * @.group   api,helper
 * @see      EmitConfigureTask
 **/

public class EmitConfigurationType extends AssertableDataType
    implements EmitConfiguration, FixtureComponent, Cloneable
{
    /**
     * Initializes a new EmitConfigurationType instance.
     **/
    public EmitConfigurationType()
    {
        super(AntX.feedback+"configure");
    }


    /**
     * Initializes a new CV-labeled EmitConfigurationType instance.
     * @param iam CV-label (non-null)
     **/
    public EmitConfigurationType(String iam)
    {
        super(iam);
    }


    /**
     * Marks this type instance as acting on behalf of another
     * object (usually a task).
     * @param controller this type's controlling object (non-null)
     * @since JWare/AntX 0.4
     **/
    public final void setController(Object controller)
    {
        m_controller = controller;
    }


    /**
     * Returns the emit configuration referenced by this instance.
     * Only call if actually is a reference.
     * @throws BuildException if this object is not a reference
     **/
    protected final EmitConfigurationType getOtherEC()
    {
        return (EmitConfigurationType)getCheckedRef
            (EmitConfigurationType.class, "emitconfiguration");
    }


    /**
     * Returns a clone of this configuration. Supports copying
     * of configurations between parent and forked child projects.
     * @since JWare/AntX 0.3
     **/
    public synchronized Object clone()
    {
        if (isReference()) {
            return getOtherEC().clone();
        }
        try {
            EmitConfigurationType cloned = (EmitConfigurationType)super.clone();
            if (getOwnEmitterFactory()!=null) {
                cloned.m_emitterFactory = null;
                cloned.setEmitterFactory(getOwnEmitterFactory().getClass());
            }
            if (getOwnTimestampFormat()!=null) {
                if (m_df instanceof SimpleDateFormat) {
                    cloned.m_df = (DateFormat)((SimpleDateFormat)m_df).clone();
                } else {
                    cloned.m_timestampFormat = null;
                    cloned.setTimestampFormat(getOwnTimestampFormat());
                }
            }
            cloned.m_defaultsEC= null;//NB:force into own space
            return cloned;

        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }


    /**
     * Capture our identifier for feedback since types don't always
     * get correct location information.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }


    /**
     * Returns a unique identifier for this configuration. Either
     * returns a script-defined {@linkplain #setId id} or a
     * combination of this instance's system identity and its class
     * name. Never returns <i>null</i>.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        if (isReference()) {
            return getOtherEC().getId();
        }
        return super.getId();
    }


// ---------------------------------------------------------------------------------------
// Inheritance/Defaults Behavior:
// ---------------------------------------------------------------------------------------

    /**
     * Sets this configuration to refer to a another configuration
     * as its defaults. The referred-to object must exist and implement
     * the EmitConfiguration interface.
     * @since JWare/AntX 0.3
     **/
    public void setDefaults(Reference r)
    {
        require_(r!=null,"setDflt- nonzro ref");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_defaultsRef = r;
        edited();
    }


    /**
     * Returns this configuration's user-defined defaults. Will return
     * <i>null</i> if never set (meaning the fixture's default configuration
     * will be used).
     * @since JWare/AntX 0.3
     **/
    public final Reference getOwnDefaults()
    {
        if (isReference()) {
            return getOtherEC().getOwnDefaults();
        }
        return m_defaultsRef;
    }


    /**
     * Sets this configuration's defaults directly. Useful when a configuration
     * is being delegated to by another configuration-type implementation.
     * @.safety guarded
     * @param defaults new defaults instance (use <i>null</i> to reset)
     **/
    public synchronized void setDefaultConfiguration(EmitConfiguration defaults)
    {
        m_defaultsEC = defaults;
    }


    /**
     * Returns the effective default configuration used by this type.
     * Usually returns either the fixture's defaults or the nearest
     * installed emit configuration taskset.
     * @since JWare/AntX 0.3
     **/
    protected synchronized EmitConfiguration getDefaults()
    {
        if (m_defaultsEC!=null) {
            return m_defaultsEC;
        }
        if (m_defaultsRef!=null) {
            EmitConfiguration ec = (EmitConfiguration)getReferencedObject
                (getProject(),m_defaultsRef.getRefId(),EmitConfiguration.class);
            verifyNotCircular(ec);
            m_defaultsEC = ec;
            return m_defaultsEC;
        }
        return EmitContext.getDefaultConfiguration();
    }


// ---------------------------------------------------------------------------------------
// Emitter Grouping Path:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the default grouping or log4j category of messages emitted
     * using this configuration.
     * @param grpId this configuration's new default category
     * @see #getOwnFrom
     **/
    public void setFrom(String grpId)
    {
        require_(grpId!=null,"setFrom- nonzro grpId");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_grpId = grpId;
        edited();
    }


    /**
     * Returns this configuration's direct grouping or log4j category.
     * The returned value (which can be <i>null</i>) is <em>not</em>
     * necessarily the grouping used to determine this configuration's
     * default grouping. That setting is a combination of the result
     * of this method and the {@linkplain #getOwnWrt wrt} setting.
     * @since JWare/AntX 0.3
     **/
    public final String getOwnFrom()
    {
        if (isReference()) {
            return getOtherEC().getOwnFrom();
        }
        return m_grpId;
    }


    /**
     * Returns the default grouping or log4j category of messages
     * emitted using this configuration. The returned path includes
     * the effect an this configuration's '{@linkplain #setWrt wrt}'
     * option. Can return <i>null</i> if never set and no 'wrt'
     * option is specified.
     * @.safety guarded
     * @.impl Do not cache calculated paths (can be context-specific)
     * @.impl Uses defaults
     * @throws BuildException if detect circular reference to self
     **/
    public String getFrom()
    {
        if (isReference()) {
            return getOtherEC().getFrom();
        }

        String from = getOwnFrom();
        String base = null;
        String ISEP = getGroupingPathSeparator();

        if (getOwnWrt()!=null) {
            if (m_wrtImpl!=null) {
                base = m_wrtImpl;
            } else {
                String wrt = Tk.lowercaseFrom(getOwnWrt());
                if (Strings.DEFAULT.equals(wrt)) {
                    EmitConfiguration ec = getDefaults();
                    base = ec.getFrom();
                    ISEP = ec.getGroupingPathSeparator();
                }
                else if (wrt.length()==0 || "root".equals(wrt)) {
                    base = "";
                }
                else if (Strings.ENCLOSING.equals(wrt)) {
                    EmitConfiguration ec = EmitContext.getConfigurationNoNull();//danger!
                    verifyNotCircular(ec);
                    base = ec.getFrom();
                    ISEP = ec.getGroupingPathSeparator();
                }
                else {
                    boolean isIt=false;
                    if (getProject().getReference(wrt)!=null) {
                        Object o = getProject().getReference(wrt);
                        if (o instanceof EmitConfiguration) {
                            EmitConfiguration ec = (EmitConfiguration)o;
                            verifyNotCircular(ec);
                            base = ec.getFrom();
                            ISEP = ec.getGroupingPathSeparator();
                            isIt = true;
                        }
                    }
                    if (base==null && !isIt) {
                        String warning = uistrs().get("emit.invalid.wrt",getId(),wrt);
                        log(warning,Project.MSG_WARN);
                    }
                }//as-refid
            }
        }
        if (base!=null) {
            if (from==null || from.length()==0) {
                from = base;
            }
            else if (base.length()>0) {
                from = base + ISEP + from;
            }
        }
        return from;
    }


    /**
     * Defines this configuration's grouping as relative to
     * something else. If either of the symbolic date groupings are
     * specified, this type takes a date snapshot <em>now</em> and uses
     * that as the grouping until reset or cleared.
     * @param wrt the something else (use <i>null</i> to clear)
     * @since JWare/AntX 0.3
     **/
    public void setWrt(String wrt)
    {
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_wrt = wrt;
        m_wrtImpl = null;
        if (wrt!=null) {
            wrt = Tk.lowercaseFrom(wrt);
            if ("longdate".equals(wrt)) {
                synchronized(DateTimeFormat.STANDARD_DATE) {
                    m_wrtImpl = DateTimeFormat.STANDARD_DATE.format(new Date());
                }
            } else if ("shortdate".equals(wrt)) {
                synchronized(DateTimeFormat.ABBREV_DATE) {
                    m_wrtImpl = DateTimeFormat.ABBREV_DATE.format(new Date());
                }
            }
        }
        edited();
    }


    /**
     * Returns what this configuration's grouping is relative to.
     * Returns the value as given to {@linkplain #setWrt setWrt}-- does not
     * returned the process values. Returns <i>null</i> if never set or
     * been reset.
     * @since JWare/AntX 0.3
     **/
    public String getOwnWrt()
    {
        if (isReference()) {
            return getOtherEC().getOwnWrt();
        }
        return m_wrt;
    }


    /**
     * Sets a custom emitter factory's grouping path separator.
     * Only useful if the emitter factory is also customized.
     * @since JWare/AntX 0.3
     **/
    public void setGroupingSeparator(String separator)
    {
        require_(separator!=null,"setGrpSep- nonzro sep");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_emitterSeparator = separator;
        edited();
    }



    /**
     * Returns the default grouping category separator for this
     * configuration. If not defined, returns whatever this type's
     * default configuration's separator is.
     * @since JWare/AntX 0.3
     * @.impl Uses defaults
     **/
    public String getGroupingPathSeparator()
    {
        if (isReference()) {
            return getOtherEC().getGroupingPathSeparator();
        }
        return m_emitterSeparator!=null ? m_emitterSeparator
            : getDefaults().getGroupingPathSeparator();
    }


// ---------------------------------------------------------------------------------------
// Automatically Included Properties (Environment Snapshot):
// ---------------------------------------------------------------------------------------

    /**
     * Sets the list of property names to be automatically attached
     * to any emitted messages.
     * @param nameList comma-delimited list of property names
     **/
    public void setProperties(String nameList)
    {
        require_(nameList!=null,"setProps- nonzro names");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_propertyNames= nameList;
        edited();
    }


    /**
     * Returns name list of properties to be automatically included
     * in any emitted messages. Returns <i>null</i> (for none)
     * if never set.
     **/
    public final String getOwnProperties()
    {
        if (isReference()) {
            return getOtherEC().getOwnProperties();
        }
        return m_propertyNames;
    }


    /**
     * Adds any custom property names to given list. Does nothing
     * and returns <i>false</i> if additional properties never
     * defined.
     * @param list names buffer to update
     * @return <i>true</i> if list buffer modified
     * @see #setProperties
     * @.impl Uses defaults
     **/
    public boolean getPropertiesNameList(StringBuffer list)
    {
        require_(list!=null,"getNams- nonzro listbuf");
        if (isReference()) {
            return getOtherEC().getPropertiesNameList(list);
        }
        boolean dflts = getDefaults().getPropertiesNameList(list);

        String myList = getOwnProperties();
        if (myList!=null) {
            if (list.length()>0) {
                list.append(",");
            }
            list.append(myList);
        }
        return dflts || myList!=null;
    }


// ---------------------------------------------------------------------------------------
// Noise Making:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the default priority level for emitted messages.
     * @param nl default message noise level
     **/
    public void setNoiseLevel(NoiseLevel nl)
    {
        require_(nl!=null,"setNoiz- nonzro lvl");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_noiseLevel = nl;
        edited();
    }


    /**
     * Returns this type's noise level setting. Will return
     * <i>null</i> if never set.
     **/
    public NoiseLevel getOwnNoiseLevel()
    {
        if (isReference()) {
            return getOtherEC().getOwnNoiseLevel();
        }
        return m_noiseLevel;
    }


    /**
     * Returns this configuration's default message priority level.
     * Will return this type's default noise level if never set.
     * @.impl Uses defaults
     **/
    public NoiseLevel getNoiseLevel()
    {
        if (isReference()) {
            return getOtherEC().getNoiseLevel();
        }
        return m_noiseLevel!=null ? m_noiseLevel : getDefaults().getNoiseLevel();
    }


    /**
     * Sets whether emissions should also be echo to the standard
     * Ant logging system.
     * @param wantIt preference setting (yes,no,inherit)
     **/
    public void setEcho(String wantIt)
    {
        require_(wantIt!=null,"setEcho- nonzro setin");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_echoPref= Setting.from(wantIt, Setting.INHERITED);
        edited();
    }


    /**
     * Returns the echo-to-Ant-log preference for this configuration.
     * Will return <i>Setting.INHERITED</i> if never set.
     **/
    public final Setting getOwnEcho()
    {
        if (isReference()) {
            return getOtherEC().getOwnEcho();
        }
        return m_echoPref;
    }


    /**
     * Returns <i>true</i> if emitted messages should also echo to
     * the standard Ant logging system.
     * @.impl Uses defaults
     **/
    public boolean shouldEcho()
    {
        if (isReference()) {
            return getOtherEC().shouldEcho();
        }
        switch (getOwnEcho().getIndex()) {
            case Setting.ON_INDEX: {
                return true;
            }
            case Setting.OFF_INDEX:{
                return false;
            }
        }
        return getDefaults().shouldEcho();
    }

// ---------------------------------------------------------------------------------------
// Timestamping (Checkpoints):
// ---------------------------------------------------------------------------------------

    /**
     * Sets the preferred format of generated timestamps. Useful for
     * checkpoints where the only message can be a timestamp. The
     * incoming format can be one of three special format identifiers:
     * "short", "long", or "GMT". These identifiers select predefined
     * formats; otherwise the format string is considered the datetime
     * formatting specifier.
     * @param format new timestamp format (non-null)
     **/
    public void setTimestampFormat(String format)
    {
        require_(format!=null,"setDtTmFmt- nonzro fmt");
        if (isReference()) {
            throw tooManyAttributes();
        }
        String normal= Tk.lowercaseFrom(format);
        if ("short".equals(normal)) {
            m_df = DateTimeFormat.ABBREV;
        } else if ("long".equals(normal)) {
            m_df = DateTimeFormat.STANDARD;
        } else if ("gmt".equals(normal)) {
            m_df = DateTimeFormat.GMT;
        }
        m_timestampFormat = format;
        edited();
    }


    /**
     * Returns this configuration's preferred timestamp format.
     * Will return <i>null</i> if never set.
     **/
    public final String getOwnTimestampFormat()
    {
        if (isReference()) {
            return getOtherEC().getOwnTimestampFormat();
        }
        return m_timestampFormat;
    }


    /**
     * Returns this configuration's custom timestamp formatter. Returns
     * <i>null</i> if this configuration has no custom timestamp formatter.
     * @see #setTimestampFormat
     * @.safety guarded
     **/
    protected final synchronized DateFormat getDateTimeFormatter()
    {
        if (m_df!=null) {
            return m_df;
        }
        String fmt = getOwnTimestampFormat();
        if (!Tk.isWhitespace(fmt)) {
            m_df = new SimpleDateFormat(fmt);
            return m_df;
        }
        return null;
    }


    /**
     * Creates readable date-time string of given timestamp (milliseconds).
     * If this configuration's timestamp format has been set, the formatted
     * string will be in this format; otherwise, an AntX ABBREVIATED
     * date/time formatted string is returned.
     * @.safety guarded
     * @.impl Uses defaults
     **/
    public String stampify(long ms)
    {
        if (isReference()) {
            return getOtherEC().stampify(ms);
        }
        DateFormat df= getDateTimeFormatter();
        if (df!=null) {
            synchronized(df) {
                StringBuffer sb = new StringBuffer(40);
                df.format(new Date(ms),sb,new FieldPosition(0));
                return sb.substring(0);
            }
        }
        return getDefaults().stampify(ms);
    }


    /**
     * Sets the include-timestamp preference for this configuration.
     * @param wantIt preference setting (yes,no,inherit)
     **/
    public void setTimestamp(String wantIt)
    {
        require_(wantIt!=null,"setTimstmp- nonzro setin");
        if (isReference()) {
            throw tooManyAttributes();
        }
        m_timestampPref= Setting.from(wantIt, Setting.INHERITED);
        edited();
    }


    /**
     * Returns the include-timestamp preference for this configuration.
     * Returns <i>Setting.INHERITED</i> if never set.
     **/
    public final Setting getOwnTimestamp()
    {
        if (isReference()) {
            return getOtherEC().getOwnTimestamp();
        }
        return m_timestampPref;
    }


    /**
     * Returns <i>true</i> if timestamp information should be included
     * in any emitted messages.
     * @.impl Uses defaults
     **/
    public boolean wantTimestamp()
    {
        if (isReference()) {
            return getOtherEC().wantTimestamp();
        }
        switch (getOwnTimestamp().getIndex()) {
            case Setting.ON_INDEX: {
                return true;
            }
            case Setting.OFF_INDEX:{
                return false;
            }
        }
        return getDefaults().wantTimestamp();
    }

// ---------------------------------------------------------------------------------------
// Diagnostics Emitters:
// ---------------------------------------------------------------------------------------

    /**
     * Returns this configuration's default emitter. Never returns
     * <i>null</i>.
     * @see #setFrom
     * @.impl Do not cache calculated emiters (can be context-specific)
     * @.impl Uses defaults
     * @.safety guarded
     **/
    public DiagnosticsEmitter getEmitter()
    {
        if (isReference()) {
            return getOtherEC().getEmitter();
        }
        String from  = getFrom();
        boolean root = Tk.isWhitespace(from);

        if (getOwnEmitterFactory()!=null) {
            if (root) {
                return getOwnEmitterFactory().newEmitter();
            }
            return getOwnEmitterFactory().newEmitter(from);
        } else if (root) {
            return getDefaults().getEmitter();
        } else {
            return getDefaults().getCustomEmitter(from);
        }
    }


    /**
     * Returns a custom diagnostics emitter for specified grouping. If
     * this configuration has a custom emitter factory it is used to produce
     * the new emitter; otherwise, the default configuration's factory
     * is used.
     * @param grpId category's name (non-null)
     * @.impl Uses defaults
     **/
    public DiagnosticsEmitter getCustomEmitter(String grpId)
    {
        if (isReference()) {
            return getOtherEC().getCustomEmitter(grpId);
        }
        if (getOwnEmitterFactory()!=null) {
            return getOwnEmitterFactory().newEmitter(grpId);
        }
        return getDefaults().getCustomEmitter(grpId);
    }


    /**
     * Sets the custom emitter factory used by this configuration.
     * @param emitterFactoryClass custom emitter factory implementation (non-null)
     * @throws BuildException if unable to create a DiagnosticsEmitterFactory
     *         instance from given class instance
     **/
    public void setEmitterFactory(Class emitterFactoryClass)
    {
        require_(emitterFactoryClass!=null,"setEmitFctry- nonzro clas");
        if (isReference()) {
            throw tooManyAttributes();
        }
        try {
            m_emitterFactory = (DiagnosticsEmitterFactory)
                emitterFactoryClass.newInstance();
        } catch (Exception anyX) {
            log(AntX.uistrs().get("emit.invalid.factory",
                                  emitterFactoryClass.getName()),
                Project.MSG_ERR);
            throw new BuildException(anyX);
        }
        edited();
    }


    /**
     * Returns this configuration's custom emitter factory. Will return
     * <i>null</i> if a custom emitter factory has not be defined.
     * @see LJDiagnosticsEmitter#FACTORY
     **/
    public final DiagnosticsEmitterFactory getOwnEmitterFactory()
    {
        if (isReference()) {
            return getOtherEC().getOwnEmitterFactory();
        }
        return m_emitterFactory;
    }



    private void verifyNotCircular(EmitConfiguration ec)
    {
        //UNIMPL: needs to do real circular check (ssmc)
        if (ec==this || ec==m_controller) {
            String ermsg = uistrs().get("emit.circular.defaults");
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
    }


// ---------------------------------------------------------------------------------------

    private String m_Id;
    private Reference m_defaultsRef;
    private EmitConfiguration m_defaultsEC;//=> cached-1st-use

    private NoiseLevel m_noiseLevel;//NB:=> AntX|Project default-level (INFO)
    private String m_propertyNames;//NB:=>none
    private Setting m_echoPref = Setting.INHERITED;//NB:=> yes
    private Setting m_timestampPref=Setting.INHERITED;//NB:=> no
    private String m_timestampFormat;//NB:=>none
    private DateFormat m_df;//NB:=>none

    private String m_grpId;//NB:=>inherited default from
    private String m_wrt="", m_wrtImpl;//NB:=>inherited default
    private DiagnosticsEmitterFactory m_emitterFactory;//NB:inherited=>log4j
    private String m_emitterSeparator;//NB:=>inherited default
    private Object m_controller;//NB:=> acting on this thing's behalf
}

/* end-of-EmitConfigurationType.java */
