/**
 * $Id: ItemConstructorTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.construct;

import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.TaskContainer;
import  org.apache.tools.ant.UnknownElement;

import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.ownhelpers.UEContainerProxy;
import  com.idaremedia.antx.parameters.FeedbackLevel;


/**
 * Fixture configuration instruction that creates a new Ant item and installs it as
 * a top-level project reference. Unless told otherwise, a constructor will overwrite
 * an existing reference with its nested object's definition.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 **/

public abstract class ItemConstructorTask extends AssertableTask 
    implements Nameable, TaskContainer
{
    /**
     * Initializes a new item constructor instance.
     **/
    protected ItemConstructorTask(String iam)
    {
        super(iam);
    }


//  ---------------------------------------------------------------------------------------
//  Script-facing Parameters:
//  ---------------------------------------------------------------------------------------

    /**
     * Set the target reference's identifier (non-null)
     * @param dstRefId reference id to install item under (non-null)
     * @see #setHaltIfExists setHaltIfExists(&#8230;)
     **/
    public void setName(String dstRefId)
    {
        require_(dstRefId!=null, "setName- nonzro refid");
        m_toRefId = dstRefId;
    }




    /**
     * Returns the (new) target reference's identifer.
     * Returns <i>null</i> if never set and source never
     * set.
     **/
    public final String getName()
    {
        return m_toRefId;
    }



    /**
     * Sets if this task will generate a build error if the
     * destination reference id already exists within its project.
     * Trumps the 'overwrite' option.
     * @param halt <i>true</i> to throw build exception
     * @see #setName setName
     * @see #setOverwrite setOverwrite
     **/
    public void setHaltIfExists(boolean halt)
    {
        m_haltIfExists = halt;
    }



    /**
     * Returns <i>true</i> if this task will generate a build error
     * if the destination reference id already exists within its
     * project. Defaults <i>false</i> (supports overwrites).
     **/
    public final boolean isHaltIfExists()
    {
        return m_haltIfExists;
    }



    /**
     * Sets if this task will overwrite an existing reference.
     * This option is ignored if the 'haltiferror' option
     * is turned on.
     * @see #setHaltIfExists
     * @param allowOverwrite <i>true</i> if can overwrite old reference
     **/
    public void setOverwrite(boolean allowOverwrite)
    {
        m_allowOverwrite = allowOverwrite;
    }



    /**
     * Returns <i>true</i> if this task will overwrite an existing
     * reference. Defaults <i>true</i>. Ignored if 'haltiferror'
     * option is turned on.
     **/
    public final boolean willAllowOverwrite()
    {
        return m_allowOverwrite;
    }



    /**
     * Tells this task how much non-diagnostic feedback to generate.
     * Really only has "loud" vs. "quiet-ish" interpretation. If
     * set quiet, this task will not issue a warning if it overwrites
     * an existing reference.
     * @param level feedback level (non-null)
     **/
    public void setFeedback(String level)
    {
        require_(level!=null,"setFeedback- nonzro level");
        FeedbackLevel fbl = FeedbackLevel.from(level);
        if (fbl==null) {
            String e = getAntXMsg("task.illegal.param.value",
                           getTaskName(), level, "feedback");
            log(e, Project.MSG_ERR);
            throw new BuildException(e, getLocation());
        }
        m_fbLevel = fbl;
    }



    /**
     * Returns this task's assigned feedback level. Will return
     * <i>null</i> by default.
     **/
    public final FeedbackLevel getFeedbackLevel()
    {
        return m_fbLevel;
    }


//  ---------------------------------------------------------------------------------------
//  Script-facing Nested Elements:
//  ---------------------------------------------------------------------------------------


    /**
     * Stashes the named UE proxy for subsequent installation.
     * @param task the unknown element task proxy (non-null)
     **/
    public void addTask(Task task)
    {
        require_(task instanceof UnknownElement, "addTask- is UE proxy");
        UnknownElement ue = (UnknownElement)task;
        if (m_itemUE!=null) {
            String ermsg = uistrs().get("taskset.only.one.specialtask",
                                        ue.getQName(), m_itemQName);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        m_itemQName = ue.getQName();
        m_itemUE = new UEContainerProxy(ue);
    }


//  ---------------------------------------------------------------------------------------
//  Execution:
//  ---------------------------------------------------------------------------------------


    /**
     * Installs the nested item declaration as a top-level reference
     * in this task's parent project.
     * @throws BuildException if verification fails.
     */
    public void execute()
    {
        verifyCanExecute_("exec");
        
        String refid = getName();
        final Project P = getProject();
        boolean quiet = FeedbackLevel.isQuietish(m_fbLevel,true,true);

        Object there = FixtureExaminer.trueReference(P,refid);
        if (there==FixtureExaminer.PENDING_REFERENCE) {
            quiet = true;
            there = null;
        }

        Map refs = P.getReferences();
        if (there!=null) {
            String msg = uistrs().get("task.warn.refid.exists",refid);
            if (isHaltIfExists()) {
                log(msg,Project.MSG_ERR);
                throw new BuildException(msg,getLocation());
            }
            if (!willAllowOverwrite()) {
                if (!quiet) {
                    log(msg,Project.MSG_VERBOSE);
                }
                m_itemUE = null;
                return;
            }
            if (!quiet) {
                log(msg,Project.MSG_WARN);
            }
        }

        Object actual = beforeInstall(m_itemUE);
        verify_(actual!=null,"exec- nonzro item");

        if (quiet) {
            refs.put(refid,actual);
        } else {
            P.addReference(refid,actual);
        }

        m_itemUE = null;
    }



    /**
     * Verifies that this task has at least its source reference
     * identifier set.
     * @throws BuildException if missing any required bits.
     **/
    public void verifyIsDefined()
    {
        if (getName()==null) {
            String ermsg = uistrs().get("task.needs.this.attr",
                getTaskName(),"name");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        if (m_itemUE==null) {
            String ermsg = uistrs().get("task.needs.oneof.these.nested",
                getTaskName(),"<any ant component>");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }



    /**
     * Verifies this task is completely defined with a target refid
     * and definition sub-element.
     * @throws BuildException if not in project or not defined properly.
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);
        verifyIsDefined();
    }



    /**
     * Called just before the new item's proxy is inserted into the
     * project's reference table. Subclasses can perform the item (if
     * it is a task) or do further adjustments. The returned object
     * can be proxy or the underlying real object; it cannot be <i>null</i>.
     * @param ue wrapper for the item to be installed (non-null)
     * @return the object to be installed (cannot be <i>null</i>)
     **/
    protected Object beforeInstall(UEContainerProxy ue)
    {
        return ue;
    }


    private String m_toRefId;
    private boolean m_haltIfExists;//NB:false allow overwrites!
    private boolean m_allowOverwrite=true;//NB:true allow overwrites!
    private FeedbackLevel m_fbLevel=FeedbackLevel.NORMAL;//NB:=>overwrite noise
    private UEContainerProxy m_itemUE;
    private String m_itemQName;
}

/* end-of-ItemConstructorTask.java */