/**
 * $Id: AlterTaskTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.construct;

import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.DynamicAttribute;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.TaskContainer;
import  org.apache.tools.ant.UnknownElement;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;
import  com.idaremedia.antx.ownhelpers.UEContainerProxy;

/**
 * Modifies a previously cached task instance. Note that you can only <em>add</em> 
 * attributes and sub-components to an existing item; you cannot remove previously 
 * installed bits. The "<span class="src">name</span>" attribute is reserved by
 * the AlterTask instance; all other parameters are treated as input to the task
 * under construction. You can prefix each parameter's name with a "addparam-"
 * string to mark it as a parameter (this allows you set a parameter called "name"
 * also).
 * <p/>
 * If you use the &lt;altertask&gt; task within an AntX looping construct, you need
 * to be careful about using the loop cursor as an input value to any added parameter
 * or nested element. If the cursor property is not evaluated <em>before</em> the
 * loop exists, the property cannot be determined (because it lives only as long
 * as the executing loop block does).
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;createtask name="${$var:_junitinstance}"&gt;
 *       &lt;junit printsummary="yes" haltonfailure="yes"&gt;
 *          &#8230;
 *       &lt;/junit&gt;
 *    &lt;/createtask&gt;
 *    &lt;do true="${$notwhitespace:@{classpathref}}"&gt;
 *       &lt;<b>altertask</b> name="${$var:_junitinstance}"&gt;
 *          &lt;classpath refid="@{classpathref}"/&gt;
 *       &lt;/altertask&gt;
 *    &lt;/do&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 * @see       CreateTaskTask
 **/

public final class AlterTaskTask extends AssertableTask
    implements TaskContainer, DynamicAttribute
{
    /**
     * Initializes a new altertask task instance. 
     **/
    public AlterTaskTask()
    {
        super(AntX.utilities+"AlterTaskTask:");
    }


    
    /**
     * Tells this editor the reference id of the task under construction.
     * This task should've been installed by the AntX task creator method.
     * @param refId reference id of task under construction (non-null)
     **/
    public void setName(String refId)
    {
        require_(refId!=null,"setName- nonzro refid");
        m_itemUC = (UEContainerProxy)FixtureExaminer.getReferencedObject(getProject(), 
            new Requester.ForComponent(this), refId, UEContainerProxy.class);
    }



    /**
     * Tells this editor to immediately resolve properties in
     * proxy objects. 
     * @param inLoop <i>true</i> to resolve.
     **/
    public void setResolveProperties(boolean inLoop)
    {
        m_tryResolveProperties = inLoop;
    }



    /**
     * Tells this editor its modifications apply to a nested element
     * one level inside the target component. Only a single level (to
     * grandchild) is supported.
     * @param childElementName local name of nested element
     **/
    public void setChild(String childElementName)
    {
        m_childElementName = childElementName;
    }



    /**
     * Gives this wrapper an override for the named parameter. Only
     * allowed if this task's "fixed" flag was cleared.
     * @param param ARM parameter (non-null)
     * @param value ARM parameter value (non-null)
     * @throws BuildException the named attribute is invalid
     */
    public void setDynamicAttribute(String param, String value)
    {
        require_(param!=null,"setDynAttr- nonzro param");
        if (param.startsWith("addparam-")) {
            param = param.substring("addparam-".length());
            if (param.length()==0) {
                String err = getAntXMsg("task.unsupported.attr",getTaskName(),"");
                log(err,Project.MSG_ERR);
                throw new BuildException(err,getLocation());
            }
        }
        if (m_tryResolveProperties) {
            value = tryResolve(getProject(),value);
        }
        m_itemAtrs.put(param,value);
    }



    /**
     * Stashes the nested element's UE proxy for subsequent installation 
     * inside the main item under construction.
     * @param task the unknown element task proxy (non-null)
     **/
    public void addTask(Task task)
    {
        require_(task instanceof UnknownElement, "addTask- is UE proxy");
        m_nestedElems.add(task);
    }



    /**
     * Installs all named attributes and nested elements into main item
     * under construction.
     */
    public void execute()
    {
        verifyCanExecute_("exec");
        final boolean grandchild = m_childElementName!=null;

        if (!m_nestedElems.isEmpty()) {
            for (int i=0,N=m_nestedElems.size();i<N;i++) {
                UnknownElement ue = (UnknownElement)m_nestedElems.get(i);
                if (m_tryResolveProperties) {
                    ue = TaskExaminer.copyUEProxy
                            (ue,ue.getProject(),ue.getOwningTarget(),true);
                }
                if (grandchild) {
                    m_itemUC.addGrandchildTask(ue,m_childElementName);
                } else {
                    m_itemUC.addTask(ue);
                }
            }
        }

        if (!m_itemAtrs.isEmpty()) {
            if (grandchild) {
                m_itemUC.setGrandchildAttributes(m_itemAtrs,m_childElementName);
            } else {
                m_itemUC.setAttributes(m_itemAtrs);
            }
        }

        m_itemUC = null;
        m_nestedElems = null;
        m_itemAtrs = null;
    }



    /**
     * Ensures we've been given a reference to a task to perform.
     */
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);

        if (m_itemUC==null) {
            String ermsg = uistrs().get("task.needs.this.attr", getTaskName(), "name");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * Tries to immediately determine the value of an attribute resolving
     * any context-dependent properties.
     * @param P project (non-null)
     * @param string string to resolve (non-null)
     * @return resolved string or original if could not resolve now.
     **/
    private static String tryResolve(final Project P, final String string)
    {
        try {
            return Tk.resolveString(P,string);
        } catch(BuildException tooSoonX) {
            return string;
        }
    }



    private UEContainerProxy m_itemUC;//item under construction (non-null after name known)
    private List m_nestedElems = AntXFixture.newList();
    private Map m_itemAtrs = AntXFixture.newMap();
    private boolean m_tryResolveProperties;//NB:off-by-default (only in loops)
    private String m_childElementName;
}

/* end-of-AlterTaskTask.java */