/**
 * $Id: VerifyFixture.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.text.MessageFormat;
import  java.util.Stack;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.BuildAssertionException;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.Handling;
import  com.idaremedia.antx.parameters.IsA;

/**
 * Helper that verifies the expected state of fixture information. Basically a simplified
 * target-independent assertion that verifies the most basic build iteration constraints.
 * Usually defined as &lt;fixturecheck&gt; or &lt;verifyfixture&gt;. This task
 * is important when defining rules that expect certain properties to exist.
 * <p>
 * <b>Examples:</b><pre>
 *    &lt;fixturecheck isset="module_version"/&gt;
 *    &lt;fixturecheck isnotwhitespace="ANTPATTERN"/&gt;
 *    &lt;fixturecheck antIs="1.6+"/&gt;
 *
 * -OR-
 *
 *    &lt;rule id="required.subbuild.fixture"&gt;
 *       &lt;fixturecheck isset="ANTPATTERN"&gt;
 *       &lt;require msgid="err.misin.filters"&gt;
 *         &lt;isreference name="copyfilters.sources" class="${ANTPATTERN}"/&gt;
 *         &lt;isreference name="copyfilters.manifests" class="${ANTPATTERN}"/&gt;
 *       &lt;/require&gt;
 *    &lt;/rule&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      AssertTask
 **/

public final class VerifyFixture extends BooleanRule
{
    /**
     * Initializes a new standalone VerifyFixture instance.
     **/
    public VerifyFixture()
    {
        super(AntX.rules+"fixturecheck",false);
    }


    /**
     * Initializes a new standalone custom-labeled
     * VerifyFixture instance.
     * @param iam CV-label (non-null)
     **/
    public VerifyFixture(String iam)
    {
        super(iam,false);
    }


    /**
     * Initializes a new embedded custom-labeled
     * VerifyFixture instance.
     * @param embed <i>true</i> if this is an embedded task
     * @since JWare/AntX 0.4
     **/
    public VerifyFixture(boolean embed)
    {
        super(AntX.rules+"fixturecheck",embed);
    }


    /**
     * Initializes the enclosing project for this task. Also
     * updates internal project-component helpers.
     **/
    public void setProject(Project project)
    {
        super.setProject(project);
        m_shh.setProject(project);
    }


    /**
     * Any short-hand condition setup triggers an immediate
     * evaluation.
     **/
    private void evalIfEmbedded()
    {
        if (isEmbedded()) {
            eval();
        }
    }

// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets an inlined message to be displayed if check fails.
     * Diagnostics aid.
     **/
    public final void setMsg(String msg)
    {
        if (!Tk.isWhitespace(msg)) {
            m_defaultMsg = uistrs().get("brul.assert.failed.lead",msg);
        } else {
            m_defaultMsg = msg;
        }
    }


    /**
     * Returns this task's inlined default message. Returns
     * <i>null</i> if never set.
     */
    public final String getDefaultMsg()
    {
        return m_defaultMsg;
    }


    /**
     * Will verify the named property is defined in fixture.
     * @param property the property to check (non-null)
     **/
    public void setIsSet(String property)
    {
        m_shh.setIsSet(property);
        m_shh.setMalformed(Handling.REJECT);
        evalIfEmbedded();
    }


    /**
     * Will verify the named property is defined to a positive
     * boolean string in fixture. Synonyms like "on" are allowed.
     * @param property the property to check (non-null)
     **/
    public void setIsSetTrue(String property)
    {
        m_shh.setIsSetTrue(property);
        evalIfEmbedded();
    }


    /**
     * Will verify the named property is not defined in fixture.
     * @param property the property to check (non-null)
     **/
    public void setIsNotSet(String property)
    {
        m_shh.setIsNotSet(property);
        evalIfEmbedded();
    }

    /**
     * Will verify the named variable is defined in fixture.
     * @param variable the variable to check (non-null)
     **/
    public void setVarSet(String variable)
    {
        m_shh.setVarSet(variable);
        evalIfEmbedded();
    }


    /**
     * Will verify the named variable is defined to a positive
     * boolean string in fixture. Synonyms like "yes" are allowed.
     * @param variable the variable to check (non-null)
     **/
    public void setVarSetTrue(String variable)
    {
        m_shh.setVarSetTrue(variable);
        evalIfEmbedded();
    }


    /**
     * Will verify the named variable is not defined in fixture.
     * @param variable the variable to check (non-null)
     **/
    public void setVarNotSet(String variable)
    {
        m_shh.setVarNotSet(variable);
        evalIfEmbedded();
    }


    /**
     * Will verify the named reference is defined in fixture.
     * @param refid the reference to check (non-null)
     **/
    public void setIsRef(String refid)
    {
        m_shh.setIsRef(refid);
        evalIfEmbedded();
    }


    /**
     * Will verify the named reference is not defined in fixture.
     * @param refid the reference to check (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setIsNotRef(String refid)
    {
        m_shh.setIsNotRef(refid);
        evalIfEmbedded();
    }


    /**
     * Will verify that at least one of the named properties
     * exists in the fixture.
     * @param properties the comma-delimited list of property
     *                   names (non-null)
     **/
    public void setAnySet(String properties)
    {
        m_shh.setAnySet(properties);
        m_shh.setMalformed(Handling.REJECT);
        evalIfEmbedded();
    }


    /**
     * Will verify that all of the named properties exist in
     * the fixture.
     * @param properties the comma-delimited list of property
     *                   names (non-null)
     **/
    public void setAllSet(String properties)
    {
        m_shh.setAllSet(properties);
        m_shh.setMalformed(Handling.REJECT);
        evalIfEmbedded();
    }


    /**
     * Will verify that all of the matching properties exist
     * and are resolvable in the fixture.
     * @param pattern the filtering pattern for all script
     *        properties
     * @since JWare/AntX 0.4
     **/
    public void setAllSetLike(String pattern)
    {
        m_shh.setAllSetLike(pattern);
        evalIfEmbedded();
    }


    /**
     * Will verify that none of the named properties exist in
     * the fixture.
     * @param properties the comma-delimited list of property
     *                   names (non-null)
     **/
    public void setNoneSet(String properties)
    {
        m_shh.setNoneSet(properties);
        evalIfEmbedded();
    }


    /**
     * Will verify that the named property is defined as a
     * boolean string in fixture. Both positive and negative boolean
     * strings allowed (including synonyms like "on" and "no").
     * @param property the property to check (non-null)
     **/
    public void setIsBoolean(String property)
    {
        m_shh.setIsBoolean(property);
        m_shh.setIsA(IsA.PROPERTY);
        evalIfEmbedded();
    }


    /**
     * Will verify that the named property is defined as a
     * parseable integral value (short, int, long) in fixture. Both
     * positive and negative values allowed.
     * @param property the property to check (non-null)
     **/
    public void setIsNumeric(String property)
    {
        m_shh.setIsNumeric(property);
        m_shh.setIsA(IsA.PROPERTY);
        evalIfEmbedded();
    }


    /**
     * Will verify that the named property is defined to non-
     * whitespace characters. Failed property substitution is
     * automatically checked.
     * @param property the property to check (non-null)
     **/
    public void setIsNotWhitespace(String property)
    {
        m_shh.setIsNotWhitespace(property);
        m_shh.setIsA(IsA.PROPERTY);
        m_shh.setMalformed(Handling.REJECT);
        evalIfEmbedded();
    }

// ---------------------------------------------------------------------------------------

    /**
     * Defines short hand &lt;antversion is="&#8230;"/&gt; condition.
     * @since JWare/AntX 0.4
     **/
    public void setAntIs(String version)
    {
        m_shh.setAntIs(version);
        evalIfEmbedded();
    }


    /**
     * Defines short hand &lt;antversion pattern="&#8230;"/&gt;
     * condition.
     * @since JWare/AntX 0.4
     **/
    public void setAntLike(String pattern)
    {
        m_shh.setAntLike(pattern);
        evalIfEmbedded();
    }



    /**
     * Defines short hand check for specific os configuration.
     * @since JWare/AntX 0.4
     **/
    public void setOS(String selector)
    {
        m_shh.setOS(selector);
        evalIfEmbedded();
    }


// ---------------------------------------------------------------------------------------
// Rule Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Evaluates this fixture verification. If verification fails,
     * this evaluation throws a build exception immediately so
     * never returns <i>false</i>.
     * @throws BuildAssertionException if assertion <i>false</i>
     **/
    public boolean eval() throws BuildException
    {
        verifyInProject_("eval");

        Condition c = getRootCondition();

        if (c==null) {
            String msgid= isEmpty()
                ? "brul.err.atleast.one.condition"
                : "brul.err.only.one.condition";
            String ermsg = uistrs().get(msgid);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }

        boolean istrue = c.eval();
        setEvalResult(istrue,getConditionNames(1));
        return istrue;
    }


    /**
     * Called to record this verification's evaluation results. For
     * fixture verifications a <i>false</i> result is very bad.
     *<i>true</i> results are basically ignored.
     **/
    protected void setEvalResult(boolean istrue, final String listing)
        throws BuildException
    {
        if (!istrue) {
            String error;

            if (getMsgId()==null) {
                String inlined = getDefaultMsg();
                if (inlined==null) {
                    error = uistrs().get("brul.assert.failed",listing);
                } else {
                    error = MessageFormat.format(inlined,new Object[]{listing});
                }
            } else {
                error = getMsg(newMsgGetter(listing));
            }

            log(error, Project.MSG_ERR);
            throw new BuildAssertionException(error, getLocation());

        }
    }


    /**
     * No-op.
     **/
    public void verifyNoCircularDependency(Stack stk, Requester clnt)
    {
    }

// ---------------------------------------------------------------------------------------

    private String m_defaultMsg;
    private final ShortHandHelper m_shh = new ShortHandHelper(this);
}

/* end-of-VerifyFixture.java */
