/**
 * $Id: RuleTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.util.Iterator;
import  java.util.List;
import  java.util.Stack;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.Requester;

/**
 * Rule that functions as a task such as an assertion or a tally. RuleTasks aren't
 * usually embedded.  This class implements support for the following:<ul>
 *   <li>Rule referrals (and all the pain that entails)
 *   <li>Template evaluation methods (single &amp; multiple conditions)
 * </ul>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded (for evaluation <em>after</em> fully configured)
 * @.group   api,infra
 * @see      com.idaremedia.antx.condition.ShortHandConditions ShortHandConditions
 **/

public abstract class RuleTask extends BooleanRule
{
    /**
     * Initializes a new RuleTask.
     **/
    protected RuleTask()
    {
        super(AntX.rules,false);
    }


    /**
     * Initializes a new CV-labeled RuleTask.
     * @param iam CV-label (non-null)
     **/
    protected RuleTask(String iam)
    {
        super(iam,false);
    }


    /**
     * Initializes a new RuleTask subclass with custom embedded setting.
     * Allows strict inner types (like Requirement) to reuse rule task's
     * internals while being defined as a rule type.
     * @param iam CV-label
     * @param embed <i>true</i> if embedded
     **/
    protected RuleTask(String iam, boolean embed)
    {
        super(iam,embed);
    }


// ---------------------------------------------------------------------------------------
// Managing Rule's Conditions:
// ---------------------------------------------------------------------------------------

    /**
     * Appends new condition to this rule's conditions sequence.
     * @param c new condition (non-null)
     * @.safety single
     * @throws BuildException if this rule has a referral defined
     *         or is frozen
     **/
    public final void xaddCondition(Condition c)
    {
        if (isReferral()) {
            throw new BuildException
                (uistrs().get("task.too.many.attrs"),getLocation());
        }
        super.xaddCondition(c);
        edited("addCond");
    }


    /**
     * Adds a single and only condition to this rule. If at least
     * one other rule exists, this method returns <i>false</i>;
     * otherwise, returns <i>true</i>. Caller is expected to react
     * with appropriate exception if unable to add root condition.
     * @param c new root conditon (non-null)
     * @return <i>true</i> if could add as root condition
     * @.safety single
     * @throws BuildException if this rule has a referral defined
     *         or is frozen
     **/
    public final boolean xaddRootCondition(Condition c)
    {
        if (isReferral()) {
            throw new BuildException
                (uistrs().get("task.too.many.attrs"),getLocation());
        }
        boolean ok = super.xaddRootCondition(c);
        if (ok) {
            edited("addCond");
        }
        return ok;
    }

// ---------------------------------------------------------------------------------------
// Evaluation Template Methods:
// ---------------------------------------------------------------------------------------

    /**
     * Default evaluation result handler method. Assigns this rule's
     * update-property value to this rule's update-property iff this
     * property isn't already defined.
     **/
    protected void setEvalResult(boolean istrue, String listing)
    {
        String up = getUpdateProperty();
        if (up!=null) {
            if (istrue) {
                String upv = getUpdatePropertyValue();
                log("RuleTask's condition true; setting '"+up+ "' to "+upv,
                    Project.MSG_DEBUG);

                checkIfProperty_(up,true);
                getProject().setNewProperty(up, upv);

            } else {
                log("RuleTask's condition false; not setting '"+up+"'",
                    Project.MSG_DEBUG);
            }
        }
    }


    /**
     * Ensure this rule task does not contain any circular references
     * in its definition.
     * @param stk stack of referred-to (or used) rules (non-null)
     * @param clnt call controls (non-null)
     * @throws BuildException if circular dependency discovered
     **/
    public void verifyNoCircularDependency(Stack stk, Requester clnt)
    {
        if (!m_circularityChecked) {
            if (isReferral()) {
                ShareableCondition referral= getReferral();
                if (stk.contains(referral)) {
                    String refId = getReferralReference().getRefId();
                    String ermsg = uistrs().get("brul.circular.referals",refId);
                    clnt.problem(ermsg,Project.MSG_ERR);
                    throw new BuildException(ermsg,clnt.getLocation());
                }
                stk.push(referral);
                if (referral instanceof BooleanRule) {
                    ((BooleanRule)referral).verifyNoCircularDependency(stk,clnt);
                }
                stk.pop();
            } else if (this instanceof ShareableCondition) {//NB:!yikes! an-error?
                if (stk.contains(this)) {
                    String ermsg = uistrs().get("brul.circular.referals",getId());
                    clnt.problem(ermsg,Project.MSG_ERR);
                    throw new BuildException(ermsg,clnt.getLocation());
                }
            }
            m_circularityChecked = true;
        }
    }


    /**
     * This rule task's default evaluation strategy for a single root
     * condition. If this rule contains multiple or no conditions
     * a build exception is thrown. Expects all execution
     * pre-conditions have been evaluated by caller. An empty rule
     * is considered a problem state.
     * @see #getRootCondition
     * @return condition evaluated value
     * @throws BuildException if this rule is empty or has more than
     *         one condition
     **/
    protected final boolean defaultSingleConditionEval()
        throws BuildException
    {
        Condition c = getRootCondition();

        if (c==null) {
            String msgid= isEmpty()
                ? "brul.err.atleast.one.condition"
                : "brul.err.only.one.condition";
            String ermsg = uistrs().get(msgid);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }

        boolean istrue = c.eval();
        setEvalResult(istrue,getConditionNames(1));
        return istrue;
    }


    /**
     * This rule task's default evaluation strategy for multiple conditions
     * with a single combining logic operator ('and' or 'or'). Expects
     * all execution pre-conditions have been evaluated by caller. An empty
     * rule is considered a problem state.
     * @see #getDefaultCombineOperator
     * @return condition evaluated value
     * @throws BuildException if this rule is empty
     **/
    protected final boolean defaultMultipleConditionEval()
        throws BuildException
    {
        List l= getConditions();

        if (!l.isEmpty()) {

            int N=0;
            boolean stopASAP= isStopQuickEvaluation(); //@since AntX-0.3
            Boolean answer  = null;
            boolean isOR    = Operator.OR.equals(getDefaultCombineOperator());
            boolean isAND   = !isOR;

            Iterator itr= l.iterator();
            while (itr.hasNext()) {
                boolean t = ((Condition)itr.next()).eval();
                N++;
                if (t) {
                    if (isOR) {
                        answer=Boolean.TRUE;
                        if (stopASAP) {
                            break;//already ok
                        }
                    }
                } else if (isAND) {
                    answer= Boolean.FALSE;
                    if (stopASAP) {
                        break;//already busted
                    }
                }
            }
            boolean istrue;
            if (answer!=null) {
                istrue= answer.booleanValue();
            } else if (isOR) {
                istrue= N!=l.size();
            } else {
                istrue= N==l.size();
            }
            setEvalResult(istrue,getConditionNames(N));
            return istrue;
        }

        String ermsg= uistrs().get("brul.err.atleast.one.condition");
        log(ermsg,Project.MSG_ERR);
        throw new BuildException(ermsg, getLocation());
    }


    /**
     * Helper to determine if this task should run as a referral.
     * @.sideeffect if this rule is a referral this method ensures the referred-to
     *              rule is compatible with this task.
     * @.sideeffect verify that there are no circular dependencies in
     *              this task's rule-chain
     **/
    protected final synchronized boolean runReferral()
    {
        if (isReferral()) {
            if (!m_circularityChecked) {
                verifyNoCircularDependency(new Stack(), 
                    new Requester.ForComponent(this));
            }
            return true;

        } else if (!m_circularityChecked) {
            if (!isEmpty()) {
                Stack stk= new Stack();
                Iterator itr= getConditions().iterator();
                while (itr.hasNext()) {
                    Object oc = itr.next();
                    if (oc instanceof BooleanRule) {//=>We allow nested rules!
                        ((BooleanRule)oc).verifyNoCircularDependency
                            (stk, new Requester.ForComponent(this));
                    }
                    stk.clear();
                }
            }
            m_circularityChecked = true;
        }
        return false;
    }

// ---------------------------------------------------------------------------------------
// Shared Rule Usage/Definition Support:
// ---------------------------------------------------------------------------------------

    /**
     * Converts a rule referral to a living-breathing-twitching rule.
     * The reference must be a {@linkplain ShareableCondition shareable}
     * rule.
     * @param r the referenced rule
     * @throws BuildException if reference is bogus
     * @see #isIncompatibleReferral
     **/
    protected final ShareableCondition getReferencedReferral(Reference r)
    {
        require_(r!=null,"refToObj- nonzro refid");

        Object object = r.getReferencedObject(getProject());
        if (object instanceof ShareableCondition) {
            return ((ShareableCondition)object);
        }
        String ermsg = uistrs().get("brul.bad.ruleid",r.getRefId());
        log(ermsg,Project.MSG_ERR);
        throw new BuildException(ermsg);
    }



    /**
     * Determine if the given referral is compatible with this rule.
     * @return <i>true</i> if the referred-to-rule is incompatible
     **/
    protected abstract boolean isIncompatibleReferral(ShareableCondition rule);



    /**
     * Setup this rule to evaluate as its own definition another
     * shareable rule. Saves the referral's reference for conversion
     * to a livin' rule object when needed.
     * @param r referral's reference (non-null)
     * @throws BuildException if this rule already has conditions
     **/
    protected final void setReferralReference(Reference r)
    {
        require_(r!=null,"setReferalId- nonzro ref");
        checkModify("setReferalId");

        if (isIndependentlyEdited() || !isEmpty()) {
            throw new BuildException
                (uistrs().get("task.too.many.attrs"),getLocation());
        }
        m_referralReference = r;
    }



    /**
     * Returns this task's referral's original reference. Returns
     * <i>null</i> if no reference set.
     **/
    protected final Reference getReferralReference()
    {
        return m_referralReference;
    }



    /**
     * Returns <i>true</i> if this rule should evaluate another rule
     * as its own definition.
     **/
    protected boolean isReferral()
    {
        return getReferralReference()!=null;
    }



    /**
     * Returns this rule's refer-to build rule reference. Returns
     * <i>null</i> if this task does not have a referral.
     * @see #isReferral
     * @throws BuildException if live referral is incompatible with this task
     **/
    protected final synchronized ShareableCondition getReferral()
    {
        if (m_referral==null && m_referralReference!=null) {
            ShareableCondition rule = getReferencedReferral(m_referralReference);
            if (isIncompatibleReferral(rule)) {
                String ermsg = uistrs().get("brul.referal.mismatch",
                                            m_referralReference.getRefId(),getTaskName());
                log(ermsg, Project.MSG_ERR);
                throw new BuildException(ermsg,getLocation());
            }
            m_referral = rule;
        }
        return m_referral;
    }



    /**
     * Returns <i>true</i> if attributes other than the referral have
     * been defined.
     **/
    protected final boolean isIndependentlyEdited()
    {
        return m_attrsEdited>0;
    }



    /**
     * Increments the non-referral attribute modified count; if user tries
     * to subsequently set this item to a referral, it veill die
     * very big.
     **/
    protected void edited(String what)
    {
        m_attrsEdited++;
    }


// ---------------------------------------------------------------------------------------

    /**
     * Adapter for a RuleTask that is itself a user of a shareable condition.
     *
     * @since    JWare/AntX 0.2
     * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  multiple
     * @.group   impl,infra
     * @see      RuleTask#setReferralReference
     **/
    public class Referee implements ShareableConditionUser
    {
        /**
         * Initializes RuleTask that uses shared rules.
         **/
        public Referee()
        { }

        /**
         * Returns the source rule task's msgid.
         **/
        public String getMsgId()
        {
            return RuleTask.this.getMsgId();
        }

        /**
         * Returns the source rule task's location.
         **/
        public Location getLocation()
        {
            return RuleTask.this.getLocation();
        }

        /**
         * Returns the source rule task's update property.
         **/
        public String getUpdateProperty()
        {
            return RuleTask.this.getUpdateProperty();
        }

        /**
         * Returns the source rule task's update variable.
         **/
        public String getUpdateVariable()
        {
            return RuleTask.this.getUpdateVariable();
        }

        /**
         * Returns the source rule task's update value.
         **/
        public String getUpdateValue()
        {
            return RuleTask.this.getUpdatePropertyValue();
        }
    }

// ---------------------------------------------------------------------------------------
// Seekrit Members:
// ---------------------------------------------------------------------------------------

    private Reference m_referralReference;//NB:none
    private ShareableCondition m_referral;//NB:none
    private int m_attrsEdited;//NB: to zap refid+others

    /** Latch that is <i>true</i> if already verified against circularity checks. **/
    private boolean m_circularityChecked;//NB:nope
}

/* end-of-RuleTask.java */
