/**
 * $Id: PreferTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.io.File;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.Path;
import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.condition.ShortHandConditions;
import  com.idaremedia.antx.parameters.Handling;
import  com.idaremedia.antx.parameters.IsA;

/**
 * Helper task that allows builder to specify a property's and/or variable's default value
 * if a conditon has not been met. Usually that condition is a simple <i>is-set</i> check,
 * but it doesn't have to be. Warning: the target property's value is overridden (that's
 * one of the main points of this tasks). This task is also useful as a way to document
 * a preference/warning to the builder that something's not as optimal as it could be:
 * for example, using an older version of javadoc that generates gobs of errors because
 * the current source is written expecting javadoc 1.4 or better.
 * <p>
 * Example usage:<pre>
 *   &lt;prefer istrue="${jdk14.present}" msgid="warn.old.javadoc"/&gt;
 *
 *   &lt;prefer isfalse="${scrub.disable}" msgid="warn.notclean.build"/&gt;
 *
 *   &lt;prefer varset="loop.isdebug"/&gt;
 *
 *   &lt;prefer varnotset="__loopcount" msgid="warn.baditeration"/&gt;
 *
 *   &lt;prefer varsettrue="happyfuntimes"/&gt;
 *
 *   &lt;prefer anyset="junit.present,pet.present" falseproperty="tests.disable"
 *         default="true"/&gt;
 *
 *   &lt;prefer isnotwhitespace="${module.title}" falseproperty="module.title"
 *         defaultproperty="default.untitled"/&gt;
 *
 *   &lt;prefer isfile="${build.logfile}" falseproperty="build.logfile"
 *         default="/tmp/build-${TSTAMP}.log"/&gt;
 *
 *   &lt;prefer httpalive="http://localhost:9090/"
 *         falseproperty="localhost9090.disable" default="true"/&gt;
 * </pre>
 * <p>
 * Note if both the property and variable attributes are specified, <em>both</em> are
 * updated on a false evaluation; for example both the 'myProp" and 'myVar' items
 * will be set to "bogus" in the following setup:<pre>
 *    &lt;prefer isfalse="true" falseproperty="myProp" falsevar="myVar" default="bogus"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1 (was PreferRule)
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public class PreferTask extends RuleTask implements ShortHandConditions
{
    /**
     * Initializes a new prefer task.
     **/
    public PreferTask()
    {
        super(AntX.rules+"prefer");
    }


    /**
     * Initializes a new CV-labeled prefer task.
     * @param iam CV-label (non-null)
     **/
    public PreferTask(String iam)
    {
        super(iam);
    }


    /**
     * Initializes a new CV-labeled optionally embedded prefer task.
     * @param iam CV-label (non-null)
     * @param embedded <i>true</i> to quiet 'execute' method
     **/
    protected PreferTask(String iam, boolean embedded)
    {
        super(iam, embedded);
    }


    /**
     * Initializes the enclosing project for this task. Also
     * updates internal project-component helpers.
     **/
    public void setProject(Project project)
    {
        super.setProject(project);
        m_shh.setProject(project);
    }


// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the property to be (over)written on a negative
     * evaluation. Set once.
     **/
    public final void setFalseProperty(String propertyName)
    {
        setUpdateProperty(propertyName);
    }


    /**
     * Returns the property to be (over)written on a negative
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getFalseProperty()
    {
        return getUpdateProperty();
    }


    /**
     * Sets the exported property to be (over)written on a
     * negative evaluation. Set once.
     * @since JWare/AntX 0.2
     **/
    public final void setFalseVariable(String variable)
    {
        setUpdateVariable(variable);
    }


    /**
     * Synonym for {@linkplain #setFalseVariable setFalseVariable}.
     **/
    public final void setFalseVar(String variable)
    {
        setFalseVariable(variable);
    }


    /**
     * Returns the exported property to be (over)written on
     * a negative evaluation. Returns <i>null</i> if never set.
     **/
    public final String getFalseVariable()
    {
        return getUpdateVariable();
    }


    /**
     * Sets the value used be set specified properties on a
     * negative evaluation.
     * @param value the default value used to (over)write properties
     **/
    public final void setDefault(String value)
    {
        setUpdatePropertyValue(value);
    }


    /**
     * Sets the property from which default value should be read on
     * negative evaluation.
     * @param property the property from which value will be read
     * @since JWare/AntX 0.3
     **/
    public final void setDefaultProperty(String property)
    {
        require_(property!=null && property.length()>0,"setDfltProp- nonwspc name");
        checkModify("setDfltProp");

        FlexString upv = getUpdatePropertyFlexValue();
        upv.set(property);
        upv.setIsProperty(true);
    }


    /**
     * Returns the value to be used to (over)write properties
     * on a negative evaluation. Returns string "<i>true</i>" if
     * never set.
     **/
    public final String getDefault()
    {
        return getUpdatePropertyValue();
    }


    /**
     * Sets this prefer task to use an existing build rule as its
     * definition. The reference will be evaluated (and verified)
     * the first time this task is evaluated.
     * @param ruleId build rule reference id (non-null)
     * @see #isIncompatibleReferral
     * @since JWare/AntX 0.2
     **/
    public final void setRuleId(Reference ruleId)
    {
        setReferralReference(ruleId);
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if the given referral's effect is worse
     * than a warning.
     * @since JWare/AntX 0.2
     **/
    protected boolean isIncompatibleReferral(ShareableCondition rule)
    {
        NoiseLevel effect = rule.getFailureEffect();
        if (effect==null || effect.isAsBadAs(NoiseLevel.ERROR)) {
            return true;
        }
        return false;
    }


    /**
     * Returns this task's preferred message noise level.
     * Calls {@linkplain BooleanRule#getPreferenceMsgLevel(Project)
     * getPreferredMsgLevel} with this task's enclosing project.
     **/
    public NoiseLevel getPreferredMsgLevel()
    {
        return BooleanRule.getPreferenceMsgLevel(getProject());
    }


    /**
     * Triggers preferred overrides if this task's conditions weren't
     * met. Generates a log entry as a minimal reaction to a negative
     * evaluation. If an update property is specified, a negative
     * evaluation will (re)set its value.
     * @see #getPreferredMsgLevel
     **/
    protected void setEvalResult(boolean istrue, final String listing)
    {
        if (!istrue) {
            String msg = getMsg(newMsgGetter(listing));
            if (msg.length()>0) {
                log(msg, getPreferredMsgLevel().getNativeIndex());
            }

            boolean told = false;
            String what = getUpdatePropertyValue();
            String prop = getUpdateProperty();

            if (prop!=null) {
                log("Prefer condition false; setting '"+prop+
                    "' property to "+what, Project.MSG_DEBUG);

                getProject().setInheritedProperty(prop, what);
                told=true;

            }
            prop = getUpdateVariable();
            if (prop!=null) {
                log("Prefer condition false; setting '"+prop+
                    "' variable to "+what, Project.MSG_DEBUG);

                ExportedProperties.set(prop,what);
                told=true;
            }

            if (!told) {
                log("Prefer condition false; logged '"+msg+"'",
                    Project.MSG_DEBUG);
            }
        } else {
            log("Prefer condition true so ignored", Project.MSG_DEBUG);
        }
    }


    /**
     * Evaluates this prefer task. If condition not met,
     * this evaluation logs a predefined message and optionally
     * (re)sets a project property.
     **/
    public boolean eval() throws BuildException
    {
        if (runReferral()) {
            return getReferral().eval(new Referee());
        }

        verifyCanExecute_("eval");

        return defaultSingleConditionEval();
    }


// ---------------------------------------------------------------------------------------
// Condition Definitions:
// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for a single 'istrue' condition; for
     * example: <code>&lt;prefer istrue="${jdk14.present}" msgid="warn.old.javadoc"/&gt;</code>
     **/
    public void setIsTrue(String value)
    {
        m_shh.setIsTrue(value);
    }


    /**
     * Attribute-shortcut for a single 'isfalse' condition; for
     * example: <code>&lt;prefer isfalse="${scrub.disable}" msgid="warn.notclean.build"/&gt;</code>
     **/
    public void setIsFalse(String value)
    {
        m_shh.setIsFalse(value);
    }


    /**
     * Attribute-shortcut for a single 'isset' condition; for
     * example: <code>&lt;prefer isset="stem" default="com.idaremedia"/&gt;</code>
     **/
    public void setIsSet(String property)
    {
        m_shh.setIsSet(property);
    }


    /**
     * Attribute-shortcut for a single 'issettrue' condition; for
     * example: <code>&lt;prefer issettrue="test.engine.present"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setIsSetTrue(String property)
    {
        m_shh.setIsSetTrue(property);
    }


    /**
     * Attribute-shortcut for a single 'isnotset' condition; for
     * example: <code>&lt;prefer isnotset="build.number" msgid="warn.notclean.buildnum"/&gt;</code>
     **/
    public void setIsNotSet(String property)
    {
        m_shh.setIsNotSet(property);
    }


    /**
     * Set the type-cast modifier for this task's short-hand flex condition.
     * @param isa the value source for short-hand condition
     * @since JWare/AntX 0.2
     **/
    public void setIsA(IsA isa)
    {
        m_shh.setIsA(isa);
    }


    /**
     * Attribute-shortcut for a single 'varset' condition; for
     * example: <code>&lt;prefer varset="__loop.debug"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setVarSet(String variable)
    {
        m_shh.setVarSet(variable);
    }


    /**
     * Attribute-shortcut for a single 'varnotset' condition; for
     * example: <code>&lt;prefer varnotset="__loopcount" msgid="warn.baditeration"/&gt;</code>
     **/
    public void setVarNotSet(String variable)
    {
        m_shh.setVarNotSet(variable);
    }


    /**
     * Attribute-shortcut for a single 'varsettrue' condition; for
     * example: <code>&lt;prefer varsettrue="happyfuntimes"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setVarSetTrue(String variable)
    {
        m_shh.setVarSetTrue(variable);
    }


    /**
     * Attribute-shortcut for a single 'anyset' condition; for
     * example: <code>&lt;prefer anyset="junit.present,pet.present"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setAnySet(String properties)
    {
        m_shh.setAnySet(properties);
    }


    /**
     * Attribute-shortcut for a single 'allset' condition; for
     * example: <code>&lt;prefer allset="module.id,module.longname"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setAllSet(String properties)
    {
        m_shh.setAllSet(properties);
    }


    /**
     * Attribute-shortcut for a single 'noneset' condition; for
     * example: <code>&lt;prefer noneset="build.number,build.time"
     * falseproperty="not.clean"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setNoneSet(String properties)
    {
        m_shh.setNoneSet(properties);
    }

// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for a single 'isboolean' condition; for
     * example: <code>&lt;prefer isboolean="${loop.isdebug}"/&gt;</code>
     * @since JWare/AntX 0.2
     **/
    public void setIsBoolean(String value)
    {
        m_shh.setIsBoolean(value);
    }


    /**
     * Attribute-shortcut for a single 'isnumeric' condition; for
     * example: <code>&lt;prefer isnumeric="${build.number}"/&gt;</code>
     **/
    public void setIsNumeric(String value)
    {
        m_shh.setIsNumeric(value);
    }


    /**
     * Attribute-shortcut for checking existence of a file;
     * for example: <code>&lt;prefer isfile="${build.logfile}"
     * falseproperty="build.logfile" default="/tmp/build-${TSTAMP}.log"/&gt;</code>
     **/
    public void setIsFile(File file)
    {
        m_shh.setIsFile(file);
    }


    /**
     * Attribute-shortcut for checking existence of a directory.
     **/
    public void setIsDirectory(File directory)
    {
        m_shh.setIsDirectory(directory);
    }


    /**
     * Attribute-shortcut for a single 'isresource' condition; for
     * example: <code>&lt;prefer isresource="${build.uistrs}"
     * falseproperty="build.uistrs"
     * default="common/resources/Messages.properties"/&gt;</code>
     **/
    public void setIsResource(String resource)
    {
        m_shh.setIsResource(resource);
    }


    /**
     * Attribute-shortcut for a single 'isclass' condition; for
     * example: <code>&lt;prefer isclass="junit.framework.Assert"
     * falseproperty="junit.disable" default="true"/&gt;</code>
     **/
    public void setIsClass(String classname)
    {
        m_shh.setIsClass(classname);
    }


    /**
     * Specifies a lookup classpath for a shorthand 'available'
     * condition.
     * @since JWare/AntX 0.2
     **/
    public void setClasspath(Path clsp)
    {
        m_shh.setClasspath(clsp);
    }


    /**
     * Specifies a lookup filepath for a shorthand 'available'
     * condition.
     * @since JWare/AntX 0.2
     **/
    public void setFilepath(Path filp)
    {
        m_shh.setFilepath(filp);
    }


    /**
     * Specifies a lookup classpath-ref for a shorthand 'available'
     * condition.
     * @since JWare/AntX 0.2
     **/
    public void setClasspathRef(Reference cpr)
    {
        m_shh.setClasspathRef(cpr);
    }


    /**
     * Specifies Ant's runtime classes should be included for
     * a shorthand 'isclass' condition; like <code>&lt;prefer isclass="foo"
     * systemclasses="false"/&gt;</code>.
     * @param included <i>false</i> if system classes excluded
     * @since JWare/AntX 0.2
     **/
    public void setSystemClasses(boolean included)
    {
        m_shh.setSystemClasses(included);
    }

// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for checking whether a web server is alive or
     * not; for example: <code>&lt;prefer httpalive="http://localhost:9090/"
     * falseproperty="localhost9090.disable" default="true"/&gt;</code>
     **/
    public void setHttpAlive(String url)
    {
        m_shh.setHttpAlive(url);
    }


    /**
     * Attribute-shortcut for checking the OS's family name; for example:
     * <code>&lt;prefer osfamily="unix" msgid="chmod.ignored"/&gt;</code>
     **/
    public void setOSFamily(String family)
    {
        m_shh.setOSFamily(family);
    }


    public void setOS(String selector)
    {
        m_shh.setOS(selector);
    }


// ---------------------------------------------------------------------------------------

    /**
     * Attribute-shortcut for checking whether a property's value is
     * all whitespace; for example:
     * <code>&lt;prefer isnotwhitespace="${module.title}" falseproperty="module.title"
     * default="Untitled"/&gt;</code>
     **/
    public void setIsNotWhitespace(String value)
    {
        m_shh.setIsNotWhitespace(value);
    }


    /**
     * Attribute-shortcut for checking a '<i>matches</i>' condition; for
     * example, <code>&lt;prefer matches="(internal)|(dist)" value="${build.type}"/&gt;</code>
     * @param pattern pattern against which other arg ('value') matched
     * @since JWare/AntX 0.2
     **/
    public void setMatches(String pattern)
    {
        m_shh.setMatches(pattern);
    }


    /**
     * Sets the unknown value being compared in a inlined equality condition.
     * @param value the second (unknown) value to match
     * @since JWare/AntX 0.2
     **/
    public void setValue(String value)
    {
        m_shh.setValue(value);
    }


    /**
     * Sets the unknown value in an inlined equality condition as a variable
     * property's name. The exported property's value will be read at comparison
     * time.
     * @param variable the variable whose contents will be matched
     * @since JWare/AntX 0.2
     **/
    public void setVariable(String variable)
    {
        m_shh.setVariable(variable);
    }


    /**
     * Synonym for {@linkplain #setVariable setVariable}.
     **/
    public final void setVar(String variable)
    {
        setVariable(variable);
    }


    /**
     * Sets the unknown value in an inlined equality condition as a property's
     * name. The property's value will be read at comparison time.
     * @param property the property whose contents will be matched
     * @since JWare/AntX 0.2
     **/
    public void setProperty(String property)
    {
        m_shh.setProperty(property);
    }


    /**
     * Sets the unknown value in an inlined equality condition as a reference's
     * name. The reference's (stringified) value will be read at comparison time.
     * @param refid the reference whose contents will be matched
     * @since JWare/AntX 0.2
     **/
    public void setReference(String refid)
    {
        m_shh.setReference(refid);
    }


// ---------------------------------------------------------------------------------------

    /**
     * Set the case-sensitivity of this task's short-hand condition.
     * @param ignore <i>true</i> if case insensitive
     **/
    public void setIgnoreCase(boolean ignore)
    {
        m_shh.setIgnoreCase(ignore);
    }


    /**
     * Returns <i>true</i> if task's short-hand condition's ignore case
     * option is activated.
     * @since JWare/AntX 0.2
     **/
    public boolean isIgnoreCase()
    {
        return m_shh.isIgnoreCase();
    }


    /**
     * Set the will-trim option of this task's short-hand condition
     * @param trim <i>true</i> if condition' unknown strings will be trimmed
     **/
    public void setTrim(boolean trim)
    {
        m_shh.setTrim(trim);
    }


    /**
     * Returns <i>true</i> if task's short-hand condition's trim strings
     * option is activated.
     * @since JWare/AntX 0.2
     **/
    public boolean willTrim()
    {
        return m_shh.willTrim();
    }


    /**
     * Set the ignore-whitespace of a inlined string comparison condition.
     * @param response <i>ignore</i> if all-whitespace should be ignored
     * @since JWare/AntX 0.2
     **/
    public void setWhitespace(Handling response)
    {
        m_shh.setWhitespace(response);
    }


    /**
     * Returns <i>true</i> if task's short-hand condition's ignore-
     * whitespace option is activated.
     * @since JWare/AntX 0.2
     **/
    public boolean ignoreWhitespace()
    {
        return m_shh.ignoreWhitespace();
    }


    /**
     * Set the allow-synonyms modifier for this task's short-hand condition.
     * @param allowAll <i>true</i> if boolean synonyms are allowed
     * @since JWare/AntX 0.2
     **/
    public void setSynonyms(boolean allowAll)
    {
        m_shh.setSynonyms(allowAll);
    }


    /**
     * Specifies how a property-dependent condition should treat
     * unresolved property values.
     * @since JWare/AntX 0.4
     **/
    public void setMalformed(Handling response)
    {
        m_shh.setMalformed(response);
    }


    /**
     * Returns how property-dependent conditions will handle
     * unresolved properties.
     * @since JWare/AntX 0.4
     **/
    public Handling getMalformedHandling()
    {
        return m_shh.getMalformedHandling();
    }


    /**
     * Specifies that a short-hand 'isnotwhitspace' should
     * check for failed variable substitution by Ant.
     * @since JWare/AntX 0.2
     * @deprecated Use {@linkplain #setMalformed setMalformed}.
     **/
    public final void setCheckBadSubst(boolean check)
    {
        log(uistrs().get("cv.deprecated", getTaskName(),
                         "checkbadsubst"), Project.MSG_WARN);
        m_shh.setMalformed(check ? Handling.REJECT : Handling.ACCEPT);
    }


// ---------------------------------------------------------------------------------------

    private final ShortHandHelper m_shh = new ShortHandHelper(this);
}

/* end-of-PreferTask.java */
