/**
 * $Id: MatchesTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.condition.Matches;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.parameters.FlexValueSupport;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.TrimEnabled;
import  com.idaremedia.antx.parameters.TrueFalsePropertySetter;

/**
 * Task version of a {@linkplain Matches} condition that lets you set another property
 * based on the evaluation result. Usually defined &lt;matches&gt; like:<pre>
 *  &lt;<b>matches</b> trueproperty="loop.DIE"
 *      pattern="(true)|(yes)|(okidoki)|(on)" variable="_loop.haltiferror"
 *      ignorecase="true"/&gt;
 *
 * -OR-
 *
 *  &lt;target name="--verify.buildenv"/&gt;
 *    &lt;<b>matches</b> falseproperty="build.type.broken"
 *        pattern="(internal)|(distribution)|(local)" value="${build.type}"/&gt;
 *    &lt;stop msgid="msg.unknown.buildtype" msgarg1="${build.type}"
 *        if="build.type.broken"/&gt;
 *  &lt;/target&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @see      Matches
 **/

public final class MatchesTask extends AssertableTask
    implements FlexValueSupport, IgnoreCaseEnabled, TrimEnabled,
               TrueFalsePropertySetter, Condition
{
    /**
     * Initializes a new MatchesTask instance.
     **/
    public MatchesTask()
    {
        super(AntX.conditions);
    }


    /**
     * Initializes the enclosing project of this task.
     * Also updates this task's helper bits.
     **/
    public void setProject(Project P)
    {
        super.setProject(P);
        m_impl.setProject(P);
    }


// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets this task's regular expression pattern.
     * @param pattern pattern against which value matched (non-null)
     **/
    public void setPattern(String pattern)
    {
        m_impl.setPattern(pattern);
    }


    /**
     * Returns pattern against which values matched. Returns
     * <i>null</i> if never set.
     **/
    public String getPattern()
    {
        return m_impl.getPattern();
    }


    /**
     * Sets this task's to-be-matched literal value.
     * @param value value to be matched (non-null)
     **/
    public void setValue(String value)
    {
        m_impl.setValue(value);
    }


    /**
     * Returns this task's to-be-matched value. Returns
     * <i>null</i> if never set or not setup as a literal string.
     **/
    public String getValue()
    {
        return m_impl.getLiteral();
    }


    /**
     * Sets a property whose value is to be evaluated by this
     * task. Property read when this task is executed.
     * @param property the property's name (non-null)
     **/
    public void setProperty(String property)
    {
        m_impl.setProperty(property);
    }


    /**
     * Returns property's name used by this task. Returns
     * <i>null</i> if never set or not setup for a property.
     **/
    public String getProperty()
    {
        return m_impl.getProperty();
    }


    /**
     * Sets an exported property whose value is to be evaluated
     * by this task. Variable read when this task is executed.
     * @param variable the exported property's name (non-null)
     **/
    public void setVariable(String variable)
    {
        m_impl.setVariable(variable);
    }


    /**
     * Synonym for {@linkplain #setVariable setVariable}.
     **/
    public void setVar(String variable)
    {
        setVariable(variable);
    }


    /**
     * Returns the exported property's name used this task. Returns
     * <i>null</i> if never set or not setup for a variable.
     **/
    public final String getVariable()
    {
        return m_impl.getVariable();
    }


    /**
     * Sets a reference whose value is to be evaluated by this
     * task. Reference read wen this task is executed.
     * @param refid the reference's name (non-null)
     **/
    public void setReference(String refid)
    {
        m_impl.setReference(refid);
    }


    /**
     * Returns the reference's name used by this task. Returns
     * <i>null</i> if never set or not setup for a reference.
     **/
    public String getReference()
    {
        return m_impl.getReference();
    }


    /**
     * Set whether the value should be trimmed of whitespace
     * before it's compared.
     **/
    public void setTrim(boolean trim)
    {
        m_impl.setTrim(trim);
    }


    /**
     * Returns <i>true</i> if the value will be trimmed before
     * it's compared.
     **/
    public final boolean willTrim()
    {
        return m_impl.willTrim();
    }



    /**
     * Set whether the value should be lower-cased before
     * it's compared.
     **/
    public void setIgnoreCase(boolean ignore)
    {
        m_impl.setIgnoreCase(ignore);
    }


    /**
     * Returns <i>true</i> if the value will be lower-cased
     * before it's compared.
     **/
    public final boolean isIgnoreCase()
    {
        return m_impl.isIgnoreCase();
    }


    /**
     * Sets the property to be created on a negative evaluation.
     * Property will be set to the string "<i>true</i>."
     * @param property the property to create (non-null)
     **/
    public void setFalseProperty(String property)
    {
        require_(property!=null,"setFalsP- nonzro nam");
        m_falseProperty = property;
    }


    /**
     * Returns the property to be created/set on a negative
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getFalseProperty()
    {
        return m_falseProperty;
    }


    /**
     * Sets the property to be created on a positive evaluation.
     * Property will be set to the string "<i>true</i>."
     * @param property the property to create (non-null)
     **/
    public void setTrueProperty(String property)
    {
        require_(property!=null,"setTrueP- nonzro nam");
        m_trueProperty = property;
    }


    /**
     * Returns the property to be created/set on a positive
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getTrueProperty()
    {
        return m_trueProperty;
    }


    /**
     * Returns <i>true</i> if the value matches this task's pattern.
     * @throws BuildException if condition definition is incomplete
     * @since JWare/AntX 0.3
     **/
    public boolean eval() throws BuildException
    {
        verifyCanExecute_("eval");

        boolean istrue= m_impl.eval();

        if (istrue && getTrueProperty()!=null) {
            String prop = getTrueProperty();
            log("Match was true; setting true-property '"+prop+"' property",
                Project.MSG_DEBUG);
            getProject().setNewProperty(prop,Strings.TRUE);
        }
        if (!istrue && getFalseProperty()!=null) {
            String prop = getFalseProperty();
            log("Match was false; setting false-property '"+prop+"' property",
                Project.MSG_DEBUG);
            getProject().setNewProperty(prop,Strings.TRUE);
        }

        return istrue;
    }


    /**
     * Checks if the value-under-test matches this task's pattern
     * and updates properties based on results.
     * @throws BuildException if condition definition is incomplete
     **/
    public void execute() throws BuildException
    {
        eval();
    }

    private final Matches m_impl= new Matches();
    private String m_falseProperty;
    private String m_trueProperty;
}

/* end-of-MatchesTask.java */
