/**
 * $Id: ConditionValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.TaskAdapter;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that records the result of evaluating a referenced condition or
 * rule type. Because most Ant conditions are not designed for reuse and are not
 * thread safe, the effect of executing a general Ant condition from a value URI
 * is undefined. However, AntX conditions and rule types work fine. Executing an
 * AntX rule <em>task</em> (like a &lt;tally&gt;) is also possible; however, you
 * must carefully define the task's contents to avoid explicitly causing any
 * premature property evaluation.
 * <p/>
 * This handler is not installed automatically by the AntX runtime, you must install
 * it explicitly like the example below. Condition value URIs are often linked to
 * either the <span class="src">$test:</span> or the <span class="src">$criteria:</span>
 * scheme.
 * <p/>
 * If a called condition signals an error, this handler will return the string
 * "<span class="src">error</span>" <em>unless</em> the current iteration's haltiferror
 * flag has been turned on for value URIs. If so, this handler lets the error
 * propagate.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;tallyset id="failquick"&gt;
 *       &lt;matches pattern="youbetcha|yup|yessirree" var="haltiferror"/&gt;
 *    &lt;/tallyset&gt;
 *    ...
 *    &lt;assign var="haltiferror" value="${$default:haltiferror}"/&gt;
 *    &lt;do true="${<b>$test:</b>failquick}"&gt;
 *       ...
 *    &lt;/do&gt;
 *
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="test"
 *             value="com.idaremedia.antx.condition.solo.ConditionValueURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   special (as safe as called condition which is usually single-threaded)
 * @.group    api,helper
 * @see       com.idaremedia.antx.condition.ShortHandValueURIHandler
 **/

public final class ConditionValueURIHandler extends ValueURIHandlerSkeleton
{
    private static final String T= Strings.TRUE;
    private static final String F= Strings.FALSE;
    private static final String E= "error";


    /**
     * Initializes a new condition value handler.
     **/
    public ConditionValueURIHandler()
    {
    }


    /**
     * Looks for the named condition reference and evaluates it. The
     * result is returned as a string. If an exception was thrown by
     * the condition and the current iteration is not set to halt for
     * valueuris problems, this method returns the string
     * "<span class="src">error</span>".
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        final Project project = clnt.getProject();
        Object value = project.getReference(uriFragment);

        if (value instanceof TaskAdapter) {
            value = ((TaskAdapter)value).getProxy();
        }

        // a. safest (mt-safe also)
        if (value instanceof ShareableCondition) {
            ShareableCondition c = (ShareableCondition)value;
            NoiseLevel effect = c.getFailureEffect();
            if (effect!=null && effect.isAsBadAs(NoiseLevel.ERROR)) {
                String e = Iteration.uistrs().get
                    ("brul.referal.valueuri.mismatch",fullUri);
                clnt.problem(e, Project.MSG_WARN);
            }
            try {
                return c.eval(new Adapter(clnt)) ? T : F;
            } catch(RuntimeException anyX) {
                if (Iteration.defaultdefaults().isHaltIfError("valueuris")) {
                    throw anyX;
                }
                return E;
            }
        }
        // b. standard (unsafe for most Ant stuff)
        else if (value instanceof Condition) {
            Condition c = (Condition)value;
            try {
                return c.eval() ? T : F;
            } catch(RuntimeException anyX) {
                if (Iteration.defaultdefaults().isHaltIfError("valueuris")) {
                    throw anyX;
                }
                return E;
            }
        }
        return null;
    }


    /**
     * Adapter to let a shared rule access call-specifics for value URI handler.
     *
     * @since    JWare/AntX 0.5
     * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  multiple
     * @.group   impl,infra
     **/
    static class Adapter implements ShareableConditionUser
    {
        Adapter(Requester clnt)
        {
            this.clnt = clnt;
        }
        public String getMsgId()
        {
            return null;
        }
        public Location getLocation()
        {
            return clnt.getLocation();
        }
        public String getUpdateProperty()
        {
            return null;
        }
        public String getUpdateVariable()
        {
            return null;
        }
        public String getUpdateValue()
        {
            return null;
        }
        private final Requester clnt;
    }
}

/* end-of-ConditionValueURIHandler.java */