/**
 * $Id: BooleanRule.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.util.ArrayList;
import  java.util.Collections;
import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.EnumSkeleton;
import  com.idaremedia.antx.starters.Chainable;

/**
 * Sequence of one or more true|false conditions. A boolean's conditions are usually
 * evaluated in the order in which they're added. BooleanRule is modeled after the
 * standard <i>ConditionBase</i>, but is more amenable to extension "the AntX-way"
 * than the standard Ant task. A boolean rule when exposed as a type should be placed
 * at a project's top level or within a fixture definition container.
 * <p>
 * <b>Implementation Notes:</b><br>
 * Changes to a boolean rule are not guarded for concurrent modification. We expect
 * a rule to be fully configured (and optionally frozen) before it is exposed for
 * use (optionally from multiple threads).
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple for evaluation <em>after</em> fully configured
 * @.group   api,infra
 **/

public abstract class BooleanRule extends AssertableTask 
    implements Condition, Chainable, AntLibFriendly
{
    /**
     * A rule's combining logic selector. Only two options: or'ed or and'ed.
     * @since JWare/AntX 0.1
     * @author ssmc
     * @.safety multiple
     * @.group api,infra
     **/
    public static final class Operator extends EnumSkeleton
    {
        /** Index of {@linkplain #OR OR}. **/
        public static final int OR_INDEX  = 0;

        /** VM-shareable instance of 'or' operator. **/
        public static final Operator OR  = new Operator("or");

        /** Index of {@linkplain #AND AND}. **/
        public static final int AND_INDEX  = OR_INDEX+1;

        /** VM-shareable instance of 'and' operator. **/
        public static final Operator AND = new Operator("and");


        /** Required bean void constructor for Ant's introspector. **/
        public Operator() {
            super();
        }

        /** Private constructor used to create our singletons. **/
        private Operator(String v) {
            super(v);
        }

        /** Returns copy of all possible values as ordered string
            array. Must be in same order as public *_INDEX. **/
        public String[] getValues() {
            return new String[]{"or","and"};
        }

        /** Helper that converts a string to a known operator
            singleton. Returns <i>null</i> if string unrecognized. **/
        public static Operator from(String s) {
            if (s!=null) {
                s= Tk.lowercaseFrom(s);
                if (OR.value.equals(s))  { return OR; }
                if (AND.value.equals(s)) { return AND; }
            }
            return null;
        }

        /** Converts a string to a known operator, with a default
            operator value if string unrecognized. **/
        public static Operator from(String s, Operator dfltOp) {
            Operator op= from(s);
            return (op==null)  ? dfltOp : op;
        }
    }//clas:Operator

// ---------------------------------------------------------------------------------------
// Construction:
// ---------------------------------------------------------------------------------------

    /**
     * Initializes a new embedded boolean rule.
     **/
    protected BooleanRule()
    {
        super(AntX.rules);
        m_rules = new ArrayList(INITIAL_CAPACITY);
        m_rulesRO = Collections.unmodifiableList(m_rules);
    }


    /**
     * Initializes a new (embedded) CV-labeled boolean rule.
     * @param iam CV-label (non-null)
     **/
    protected BooleanRule(String iam)
    {
        super(iam);
        m_rules = new ArrayList(INITIAL_CAPACITY);
        m_rulesRO = Collections.unmodifiableList(m_rules);
    }


    /**
     * Initializes a new CV-labeled boolean rule with custom
     * embedded setting.
     * @param iam cv-label (non-null)
     * @param embedded <i>false</i> if this rule is not embedded
     **/
    protected BooleanRule(String iam, boolean embedded)
    {
        this(iam);
        setEmbedded(embedded);
    }


    /**
     * Initializes the enclosing project for this rule. Also
     * updates internal project-component helpers.
     **/
    public void setProject(Project project)
    {
        super.setProject(project);
        m_updateValue.setProject(project);
    }


    /**
     * Returns whether this rule is embedded (and acting more
     * like a component or type and not a standalone task).
     **/
    protected final boolean isEmbedded()
    {
        return m_embedded;
    }


    /**
     * Flags this rule as either embedded or not. Should be set
     * at construction.
     * @.safety single
     **/
    protected final void setEmbedded(boolean isE)
    {
        m_embedded= isE;
    }


// ---------------------------------------------------------------------------------------
// Common rule functionality helpers:
// ---------------------------------------------------------------------------------------

    /**
     * Sets this rule's default combining logic.
     * @param op new combining logic operator (non-null)
     **/
    protected final void setDefaultCombineOperator(Operator op)
    {
        require_(op!=null,"setDfltOp- nonzro op");
        checkModify("setDfltOp");
        m_defaultOperator = op;
    }


    /**
     * Returns this rule's default combining logic operator.
     * Never return <i>null</i>. Defaults to {@linkplain Operator#AND
     * AND}.
     **/
    protected final Operator getDefaultCombineOperator()
    {
        return m_defaultOperator;
    }


    /**
     * Sets whether this rule task will quick-stop its evaluation.
     * When quick-stop is <i>true</i> (the default) the default evaluation
     * methods should stop as soon as the evaluation result can be
     * determined (leaving unevaluated conditions if necessary). In the
     * case of "or" this means stopping on the first true-condition; in
     * the case of "and" this means stopping on the first false-condition.
     * If quick-stop is turned off, all conditions are evaluated. This
     * is useful for tallying tasks and rules that want all conditions
     * evaluated regardless of the outcome.
     * @since JWare/AntX 0.3
     **/
    protected final void setStopQuickEvaluation(boolean quikstop)
    {
        m_quickStop = quikstop;
    }


    /**
     * Returns <i>true</i> if default rule evaluation should stop as
     * soon as the final result can be determined. Defaults to
     * <i>true</i>.
     * @since JWare/AntX 0.3
     **/
    protected final boolean isStopQuickEvaluation()
    {
        return m_quickStop;
    }


    /**
     * Defines the property to be set on a negative evaluation.
     * @param pn property name (non-whitespace)
     **/
    protected void setUpdateProperty(String pn)
    {
        require_(pn!=null && pn.length()>0,"setUpdtProp- nonwspc nam");
        checkModify("setUpdtProp");
        m_updateProperty = pn;
    }


    /**
     * Returns name of property to be set if this rule evaluates
     * <i>false</i>. Returns <i>null</i> if property name never
     * set.
     **/
    protected String getUpdateProperty()
    {
        return m_updateProperty;
    }


    /**
     * Defines the value to which this rule's update-property will
     * be set. Defaults to string "<i>true</i>".
     * @param pv property value (non-whitespace)
     **/
    protected void setUpdatePropertyValue(String pv)
    {
        require_(pv!=null && pv.length()>0,"setUpdtValu- nonwspc val");
        checkModify("setUpdtValu");
        m_updateValue.set(pv);
        m_updateValue.setIsLiteral();
    }


    /**
     * Returns the underlying flexible value to update properties
     * and/or variables if this rule evaluates <i>false</i>. Never
     * returns <i>null</i>.
     * @since JWare/AntX 0.3
     **/
    protected final FlexString getUpdatePropertyFlexValue()
    {
        return m_updateValue;
    }


    /**
     * Returns value of update-property used if this rule evaluates
     * <i>false</i>. Returns <i>true</i> by default; never returns
     * <i>null</i>.
     **/
    protected String getUpdatePropertyValue()
    {
        return String.valueOf(m_updateValue.getValue());
    }


    /**
     * Sets name of modifiable property to be set on a negative
     * evaluation.
     * @param pn the property's name (non-null)
     **/
    protected void setUpdateVariable(String pn)
    {
        require_(pn!=null && pn.length()>0,"setUpdtVar- nonzro varnam");
        checkModify("setUpdtVar");
        m_updateVariable = pn;
    }


    /**
     * Returns name of modifiable property to be set if this
     * rule evaluates <i>false</i>. Returns <i>null</i> if
     * variable name never set.
     **/
    protected String getUpdateVariable()
    {
        return m_updateVariable;
    }


    /**
     * Tries to return an unique identifier for this rule. Never returns
     * <i>null</i>.
     **/
    public String getId()
    {
        if (getLocation()!=Location.UNKNOWN_LOCATION) {
            return getTaskName()+"@"+getLocation().toString();
        }
        return getTaskName()+"@"+String.valueOf(System.identityHashCode(this));
    }


    /**
     * Returns the preferred failed-preference noise level. Returns
     * <i>warning</i> if not defined in project property. Never returns
     * <i>null</i>.
     * @see AntX#PROPERTY_FALSE_PREFER_NOISELEVEL_PROP
     * @param P [optional] project from which noiselevel property read
     **/
    public final static NoiseLevel getPreferenceMsgLevel(Project P)
    {
        String s = Tk.getTheProperty
            (P, AntX.PROPERTY_FALSE_PREFER_NOISELEVEL_PROP);

        NoiseLevel nl= NoiseLevel.WARNING;
        if (s!=null) {
            nl = NoiseLevel.from(s,nl);
        }
        return nl;
    }


// ---------------------------------------------------------------------------------------
// Managing Rule's Conditions:
// ---------------------------------------------------------------------------------------

    /**
     * Returns the underlying conditions list. Never returns <i>null</i>.
     **/
    private List getConditionsList()
    {
        return m_rules;
    }


    /**
     * Appends new condition to this rule's conditions sequence.
     * @param c new condition (non-null)
     * @.safety single
     * @throws BuildException if this rule is frozen
     **/
    public void xaddCondition(Condition c)
    {
        require_(c!=null,"addInrRul- nonzro rul");
        checkModify("addRul");

        getConditionsList().add(c);
    }


    /**
     * Adds the first and only condition of this rule. If other
     * conditions already exist, this method returns <i>false</i>;
     * otherwise, it returns <i>true</i>. (Caller is expected to react
     * with appropriate exception if unable to add root condition.)
     * @param c new root conditon (non-null)
     * @return <i>true</i> if could add as root condition
     * @.safety single
     * @throws BuildException if this rule is frozen
     **/
    public boolean xaddRootCondition(Condition c)
    {
        require_(c!=null,"add1Rul- nonzro rul");
        checkModify("add1Rul");

        List l= getConditionsList();
        if (!l.isEmpty()) {
            return false;
        }
        l.add(c);
        return true;
    }


    /**
     * Returns a <em>readonly</em> view of this rule's underlying
     * conditions sequence. Never returns <i>null</i>. For use by
     * subclasses. While the list is readonly, its individual condition
     * elements are not.
     **/
    protected final List getConditions()
    {
        return m_rulesRO;
    }


    /**
     * Convenience method that returns <i>true</i> if this rule's
     * underlying conditions list is empty.
     **/
    protected final boolean isEmpty()
    {
        return m_rulesRO.isEmpty();
    }


    /**
     * Convenience method to return a supposedly single root condition.
     * Returns <i>null</i> if this rule is empty or contains more than one
     * condition (caller is expected to react with appropriate
     * exception).
     **/
    protected final Condition getRootCondition()
    {
        List list= getConditions();
        if (list.size()!=1) {
            return null;
        }
        return (Condition)list.get(0);
    }


    /**
     * Creates list of the first 'N' conditions in this rule. Returns
     *  the string "<i>none</i>" if this rule is empty or zero conditions
     * requested.
     * @param N number of conditions to include (1-based)
     **/
    protected String getConditionNames(final int N)
    {
        List cl= getConditionsList();
        if (cl.isEmpty() || N==0) {
            return Strings.NONE;
        }
        StringBuffer sb= new StringBuffer(300);
        int i=0;
        sb.append(getDefaultCombineOperator().getValue().toUpperCase());
        sb.append('[');
        Iterator itr= cl.iterator();
        while (itr.hasNext()) {
            if (i!=0) { sb.append(','); }
            sb.append(Tk.leafNameFrom(itr.next().getClass()));
            i++;
            if (i==N) {
                break;
            }
        }
        sb.append(']');
        return sb.toString();
    }


    /**
     * Freezes this rule against further modification. Caller should ensure
     * that this rule has been fully configured <em>before</em> freezing it.
     * Subclasses that override this method must call this inherited
     * implementation <em>first</em>. Because of how Ant configures its
     * objects, this method should never be called from a constructor.
     **/
    public void freezeDefinition()
    {
        setFrozen();
    }


    /**
     * Toggles this rule's frozen flag on. This step is extracted to ensure
     * a composite rule can first flag itself as frozen while it traverses
     * its sub-rules. Once frozen a rule cannot be unfrozen.
     **/
    protected final void setFrozen()
    {
        m_isFrozen = true;
    }


    /**
     * Returns <i>true</i> if this rule is already frozen.
     **/
    protected final boolean isFrozen()
    {
        return m_isFrozen;
    }


    /**
     * Verifies that this rule isn't frozen against further changes.
     * @throws IllegalStateException if this rule is frozen
     **/
    protected final void checkModify(String who)
    {
        require_(!m_isFrozen,who+"- not frozen");
    }


// ---------------------------------------------------------------------------------------
// Evaluation Template Methods:
// ---------------------------------------------------------------------------------------

    /**
     * Required evaluation result handler method. Should be called
     * post-evaluation before returning results.
     * @param istrue result of evaluation
     * @param listing listing of conditions executed to reach result
     **/
    protected abstract void setEvalResult(boolean istrue, String listing);



    /**
     * Hook to execute this rule as a standard Ant task. Exactly
     * same as evaluation if this is not an embedded rule.
     * @throws BuildException if this rule mis-configured
     **/
    public void execute() throws BuildException
    {
        if (!isEmbedded()) {
            eval();
        }
    }


// ---------------------------------------------------------------------------------------
// Support for Cloneable datatype derivatives:
// ---------------------------------------------------------------------------------------

    /**
     * Creates an independent list of sub-conditions for the new
     * cloned rule. The conditions are <em>not</em> themselves cloned.
     * @.safety single
     * @since JWare/AntX 0.3
     **/
    protected void cloneInternals(BooleanRule copy)
    {
        copy.m_rules = (ArrayList)m_rules.clone();
        copy.m_rulesRO = Collections.unmodifiableList(m_rules);
    }

// ---------------------------------------------------------------------------------------
// Seekrit Members:
// ---------------------------------------------------------------------------------------

    /** Shared debug message bits for all only-one-root condition checks. **/
    protected static final String ONLY_ONE= "- only one condition in rule";
    private static final int INITIAL_CAPACITY=5;

    private ArrayList m_rules;//NB:fixed by ctor|clone
    private List m_rulesRO;//NB:fixed by ctor|clone

    private Operator m_defaultOperator= Operator.AND;
    private String m_updateProperty;//NB:none
    private String m_updateVariable;//NB:none
    private FlexString m_updateValue= new FlexString(Strings.TRUE);
    private boolean m_embedded=true;//NB:set-once-at-ctor
    private volatile boolean m_isFrozen;//NB:false
    private boolean m_quickStop=true;//NB:stop-as-soon-as-can
}

/* end-of-BooleanRule.java */
