/**
 * $Id: IsSetTrue.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;
import  com.idaremedia.antx.parameters.IsA;
import  com.idaremedia.antx.parameters.SynonymsEnabled;

/**
 * Flexible Is-Set-True condition check. This condition helps get around the problem
 * of chained conditions where a later condition's evaluation depends on the
 * side-effects of a previous conditon's evaluation (like say its property flag) but
 * that previous condition hasn't been evaluated when Ant parses (and replaces)
 * attribute values. For instance, without this condition the following would never
 * evaluate <i>true</i> like you might think it would:<pre>
 *  &lt;tally trueproperty="tally.true"&gt;
 *    &lt;tally trueproperty="subtally.true"&gt;
 *      &lt;istrue value="true"/&gt;
 *    &lt;/tally&gt;
 *    &lt;istrue value="${subtally.true}"/&gt; <b>&lt;-- this is always false</b>
 *  &lt;/tally&gt;
 *  &lt;assert istrue="${tally.true}"/&gt; <b>&lt;-- this will always fail</b>
 * </pre>
 * Instead you have to write:<pre>
 *  &lt;tally trueproperty="tally.true"&gt;
 *    &lt;tally trueproperty="subtally.true"&gt;
 *      &lt;istrue value="true"/&gt;
 *    &lt;/tally&gt;
 *    &lt;issettrue property="subtally.true"/&gt; <b>&lt;-- this will be true</b>
 *  &lt;/tally&gt;
 *  &lt;assert issettrue="tally.true"/&gt; <b>&lt;-- this will pass</b>
 * </pre>
 * More examples:
 * <pre>
 *    &lt;issettrue property="jdk14.present" synonyms="off"/&gt;
 *    &lt;issettrue property="cvs.disable" ignorecase="true"/&gt;
 *    &lt;issettrue reference="tally.debug"/&gt;
 *
 * -OR, to just work like &lt;istrue&gt; does-
 *    &lt;issettrue value="${jdk14.present}"/&gt;
 *</pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public class IsSetTrue extends SimpleFlexCondition
    implements IgnoreCaseEnabled, SynonymsEnabled
{
    /**
     * Initializes new IsSetTrue condition. Will return <i>false</i>
     * if item-under-test is not set.
     **/
    public IsSetTrue()
    {
    }


    /**
     * Initializes pre-defined IsSetTrue condition.
     * @param property the property against which condition checks
     **/
    public IsSetTrue(String property)
    {
        setProperty(property);
    }


    /**
     * Initializes pre-defined IsSetTrue condition with variable or
     * property
     * @param value the property against which condition checks
     * @param isP <i>true</i> if this property otherwise is a variable
     **/
    public IsSetTrue(String value, boolean isP)
    {
        if (isP) {
            setProperty(value);
        } else {
            setVariable(value);
        }
    }


    /**
     * Sets whether property value should be lowercased before
     * comparision.
     **/
    public void setIgnoreCase(boolean ignore)
    {
        getValueHelper().setIgnoreCase(ignore);
    }


    /**
     * Returns <i>true</i> if this condition will ignore case
     * of property values when comparing to "<i>true</i>."
     **/
    public final boolean isIgnoreCase()
    {
        return getValueHelper().isIgnoreCase();
    }



    /**
     * Sets whether <i>true</i> synonyms like <i>yes</i> are
     * acceptable. Defaults <i>true</i>.
     **/
    public void setSynonyms(boolean allow)
    {
        m_allowAll = allow;
    }


    /**
     * Returns <i>true</i> if this condition will allow
     * "<i>true</i>" synonyms like "<i>on</i>".
     **/
    public final boolean allowSynonyms()
    {
        return m_allowAll;
    }



    /**
     * Sets this condition to evaluate a literal value as-is.
     * This parameter allows this condition work exactly like
     * standard &lt;istrue&gt; condition.
     * @param value the literal value to compare
     **/
    public void setValue(String value)
    {
        require_(value!=null,"setValu- nonzro");
        setLiteral(value);
    }



    /**
     * Marks this condition's property as the default field
     * set by a value URI.
     * @since JWare/AntX 0.5
     * @return IsA.PROPERTY always.
     */
    protected IsA getDefaultIsAForURI()
    {
        return IsA.PROPERTY;
    }



    /**
     * Returns <i>true</i> if the condition's property is
     * defined as the string "<i>true<i>".
     **/
    public boolean eval()
    {
        verifyCanEvaluate_("eval");

        String value = getValueHelper().getValue();

        return allowSynonyms() ? Tk.booleanFrom(value) :
            Strings.TRUE.equals(value);
    }


    private boolean m_allowAll=true;//NB:defaults yes
}

/* end-of-IsSetTrue.java */
