/**
 * $Id: IsAntVersion.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.go.IffAnt;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.TrueFalsePropertySetter;

/**
 * Condition that checks the current Ant runtime's version information
 * against a regular expression. If you can guarantee that the project property
 * "<span class="src">ant.version</span>" is always defined, you can get this
 * condition's basic test by using the standard AntX {@linkplain Matches}
 * condition. However, this condition adds the ability to set properties
 * and use short hand (not RE) version strings.
 * <p>
 * Example usage:<pre>
 *     &lt;antversion like="^.*version 1\.[5-7].*"/&gt;
 *     &lt;antversion is="1.6"/&gt;
 *     &lt;antversion is="1.5.4" trueproperty="old.ant.present"/&gt;
 *     &lt;antversion is="1.6+"/&gt;
 *     &lt;antversion like="1234" property="local.ant.version"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 * @.caveat  If this condition is part of a rule, you should not set either update
 *           property unless you only execute the rule once.
 **/

public class IsAntVersion extends AssertableProjectComponent
    implements Condition, TrueFalsePropertySetter, URIable
{
    /**
     * Initializes a new antversion condition.
     **/
    public IsAntVersion()
    {
        super(AntX.conditions);
    }


    /**
     * Initializes a new antversion condition bound to a
     * project and pattern.
     **/
    public IsAntVersion(Project P, String pattern)
    {
        super(AntX.conditions);
        setProject(P);
        setLike(pattern);
    }

// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    private static final String VERSION_RE_0= "^.*version ";
    private static final String VERSION_RE_N= " compiled .*$";


    /**
     * Sets this condition's regular expression pattern.
     * @param pattern pattern against which value matched (non-null)
     **/
    public void setLike(String pattern)
    {
        require_(pattern!=null,"setLike- nonzro pattern");
        m_pattern = pattern;
    }


    /**
     * Sets this condition's version string as a simple
     * <em>exact</em> number string like:
     * "<span class="src">1.6.1</span>". This method basically
     * builds the match pattern for caller.
     * @param version exact version to look for (non-null)
     **/
    public final void setIs(String version)
    {
        require_(!Tk.isWhitespace(version),"setIs- nonzro version");

        StringBuffer sb = AntXFixture.newStringBuffer();
        sb.append(VERSION_RE_0);

        int N= version.length();
        for (int i=0;i<N;i++) {
            char c = version.charAt(i);
            if (c=='.') {
                sb.append("\\.");
            } else if (c=='+' && i==N-1) {
                sb.append("(\\.[0-9]+)?(beta[1-9]+)?");
            } else {
                sb.append(c);
            }
        }

        sb.append(VERSION_RE_N);
        setLike(sb.substring(0));

        log("[antversion] looking for version pattern='"+m_pattern+"'",
            Project.MSG_DEBUG);
        sb = null;
    }


    /**
     * Returns pattern against which values matched. Returns
     * <i>null</i> if never set.
     **/
    public final String getPattern()
    {
        return m_pattern;
    }


    /**
     * Sets the property to be created on a negative evaluation.
     * Property will be set to the string "<i>true</i>."
     * @param property the property to create (non-null)
     **/
    public void setFalseProperty(String property)
    {
        require_(property!=null,"setFalsP- nonzro nam");
        m_falseProperty = property;
    }


    /**
     * Returns the property to be created/set on a negative
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getFalseProperty()
    {
        return m_falseProperty;
    }


    /**
     * Sets the property to be created on a positive evaluation.
     * Property will be set to the string "<i>true</i>."
     * @param property the property to create (non-null)
     **/
    public void setTrueProperty(String property)
    {
        require_(property!=null,"setTrueP- nonzro nam");
        m_trueProperty = property;
    }


    /**
     * Returns the property to be created/set on a positive
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getTrueProperty()
    {
        return m_trueProperty;
    }


    /**
     * Sets a custom source property for the version string. This
     * parameter is useful for test scripts and other situations
     * where you want to read version string from source other than
     * default Ant property.
     * @param property name of property containing version string
     **/
    public void setProperty(String property)
    {
        m_versionProperty = property;
    }



    /**
     * Sets this condition's pattern (like) as part of a value URI.
     * @param fragment the value uri bits (non-null)
     * @since JWare/AntX 0.5
     */
    public void xsetFromURI(String fragment)
    {
        setLike(fragment);
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if the Ant version matches the pattern.
     * @.sideeffect Will update true and/or false properties
     *              if defined.
     **/
    public boolean eval()
    {
        verifyCanEvaluate_("eval");

        boolean istrue;

        if (m_versionProperty==null) {
            istrue = IffAnt.pass(getPattern(),getProject(),false);
        } else {
            istrue = IffAnt.pass(getPattern(),m_versionProperty,getProject(),false);
        }

        if (m_trueProperty!=null && istrue) {
            log("[antversion] was true; setting true-property '"+
                m_trueProperty+
                "' property",  Project.MSG_DEBUG);
            getProject().setNewProperty(m_trueProperty,Strings.TRUE);
        }

        if (m_falseProperty!=null && !istrue) {
            log("[antversion] was false; setting false-property '"+
                m_falseProperty+
                "' property",  Project.MSG_DEBUG);
            getProject().setNewProperty(m_falseProperty,Strings.TRUE);
        }

        return istrue;
    }


    /**
     * Call to verify that this condition is in valid project
     * and has its pattern defined.
     * @param calr caller's identifier
     * @throws BuildException if not in project or all bits not defined
     **/
    protected void verifyCanEvaluate_(String calr)
    {
        super.verifyInProject_(calr);

        if (getPattern()==null) {
            String ermsg = uistrs().get("type.needs.this.attr",
                                        "antversion", "pattern|is");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
    }


    private String m_falseProperty;
    private String m_trueProperty;
    private String m_pattern;
    private String m_versionProperty;
}

/* end-of-IsAntVersion.java */
