/**
 * $Id: JavadocLogInterpreter.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  java.io.BufferedReader;
import  java.io.IOException;
import  java.io.Reader;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.types.RegularExpression;
import  org.apache.tools.ant.util.regexp.RegexpMatcher;

import  com.idaremedia.antx.helpers.Tk;

/**
 * Our default Ant &lt;javadoc&gt; output log interpreter. This interpreter does
 * not work for Maven generated javadoc logs (which have the tool's leading name
 * field removed). This interpreter only updates the count properties iff there was
 * at least one problem; it does not set the properties if the log was determined to
 * be clean.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 * @see       InterpretLoggedTask
 * @.impl     This interpreter is used to verify interpret task.
 **/

public final class JavadocLogInterpreter implements LogInterpreter 
{
    /**
     * Initializes a new javadoc log interpreter.
     **/
    public JavadocLogInterpreter()
    {
    }
 
 
    private static final String RE_ERRORS_    = "^  \\[javadoc\\] ([0-9]+) errors.*$";
    private static final String RE_WARNINGS_  = "^  \\[javadoc\\] ([0-9]+) warnings.*$";


    public String interpret(Reader inputr, InterpretParameters config)
        throws BuildException
    {
        String result = CLEAN;
        final Project P = config.getProject();
        int nErrs = 0;
        int nWarnings = 0;

        RegularExpression re = new RegularExpression();
        re.setProject(P);
        re.setPattern(RE_ERRORS_);
        RegexpMatcher errors = re.getRegexp(P);

        re = new RegularExpression();
        re.setProject(P);
        re.setPattern(RE_WARNINGS_);
        RegexpMatcher warnings = re.getRegexp(P);

        try {
            //NB: if we could read backwards this would be basically instantaneous
            BufferedReader in = new BufferedReader(inputr,512);
            String line;
            while ((line=in.readLine())!=null) {
                List gl = errors.getGroups(line);
                if (gl!=null && gl.size()>=2) {
                    int n = Tk.integerFrom(gl.get(1),Tk.NO_INT);
                    if (n!=Tk.NO_INT) {
                        nErrs = n;
                    }
                }
                gl = warnings.getGroups(line);
                if (gl!=null && gl.size()>=2) {
                    int n = Tk.integerFrom(gl.get(1),Tk.NO_INT);
                    if (n!=Tk.NO_INT) {
                        nWarnings = n;
                    }
                }
            }
            in.close();
        } catch(IOException ioX) {
            throw new BuildException(ioX);
        }

        if (nErrs!=0) {
            if (nErrs>config.getMaxErrors()) {
                result = FAILURE;
            } else {
                result = PROBLEM;
            }
        } else if (nWarnings!=0) {
            if (nWarnings>config.getMaxWarnings()) {
                result = FAILURE;
            } else {
                result = PROBLEM;
            }
        }

        ResultsHelper.set(config,nErrs,nWarnings,result);
        return result;
    }
}

/* end-of-JavadocLogInterpreter.java */