/**
 * $Id: AssertLoggedTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  java.util.ArrayList;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;
import  org.apache.tools.ant.types.RegularExpression;
import  org.apache.tools.ant.util.regexp.RegexpMatcher;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.BuildAssertionException;
import  com.idaremedia.antx.apis.TestScriptComponent;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.helpers.Empties;
import  com.idaremedia.antx.helpers.Strings;

/**
 * Assertion task whose conditions all involve checking contents of captured feedback
 * messages. No real use outside of testing other task implementations.
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;capturelogs&gt;
 *      &lt;echo level="warning" message="((Warning))"/&gt;
 *      &lt;assertlogged value="((Warning))" trueproperty="Was.warned"/&gt;
 *      &lt;assert issettrue="Was.warned"/&gt;
 *      &lt;echo level="verbose" message="((Verbose))"/&gt;
 *      &lt;assertlogged value="((Verbose))" occurances="0" msg="Broken importance check"/&gt;
 *      &lt;assertlogged important="no" value="((Verbose))" occurances="1"/&gt;
 *      &lt;assertlogged important="no"&gt;
 *          &lt;string value="((Warning))"/&gt;
 *          &lt;string value="((Verbose))"/&gt;
 *      &lt;/assertlogged&gt;
 *    &lt;/capturelogs&gt;
 * </pre>
 * If a regular expression is defined for the assertion (using
 * &lt;assertlogged like="&#46;&#46;&#46;"/&gt;) then no other match parameter or
 * nested element is allowed. The regular expression must contain any occurances specifiers.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,test,helper
 * @see      CaptureLogsTask
 * @see      CaptureStreamsTask
 **/

public final class AssertLoggedTask extends LogsUsingTask
    implements Condition, TestScriptComponent
{
    /**
     * Initializes a new AssertLoggedTask instance.
     **/
    public AssertLoggedTask()
    {
        super(AntX.capture+"AssertLoggedTask:");
    }


// ---------------------------------------------------------------------------------------
// Parameters, Nested Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Sets an inlined message to be displayed if assertion
     * fails.
     **/
    public final void setMsg(String msg)
    {
        m_defaultMsg = msg;
    }


    /**
     * Returns this task's inlined default message. Returns
     * <i>null</i> if never set.
     */
    public final String getDefaultMsg()
    {
        return m_defaultMsg;
    }


    /**
     * Defines the single value to look for in recorded logs. If
     * defined no other match parameter or nested elements are allowed.
     * @throws BuildException if this task already setup to look for an
     *         order list of values
     * @see #setLike
     **/
    public void setValue(String value)
    {
        require_(value!=null,"setValu- nonzro valu");
        checkLookupMechanism("value");
        m_requiredValue = value;
    }


    /**
     * Returns the single value to look for in recorded logs.
     * Will return <i>null</i> if never set.
     **/
    public final String getValue()
    {
        return m_requiredValue;
    }


    /**
     * Sets the expected number of occurances of the value under
     * test. Set to zero to indicate the value should <em>not</em>
     * exist in logs. Set to a negative value to indicate the value
     * can occur any number of times (will just check that exists).
     * @param Ntimes number of occurances
     **/
    public void setOccurances(int Ntimes)
    {
        m_Ntimes = Ntimes;
    }

    /**
     * Typo-friendly synonym for {@linkplain #setOccurances
     * setOccurances}.
     **/
    public final void setOccurance(int Ntimes)
    {
        setOccurances(Ntimes);
    }


    /**
     * Returns the expected number of occurances of the value
     * under test. Will return a negative number if never set (which
     * means the value can exist one or more times). Will return
     * zero if the value should <em>not</em> exist; otherwise
     * returns the number of expected occurance.
     **/
    public final int getOccurances()
    {
        return m_Ntimes;
    }


    /**
     * Defines the regular expression to look for in recorded logs.
     * If defined no other match parameter or nested elements are allowed.
     * @param pattern regular expression to match in logs (non-null)
     * @throws BuildException if this task already setup with any
     *         other type of lookup mechanim
     * @since JWare/AntX 0.3
     **/
    public void setLike(String pattern)
    {
        require_(pattern!=null,"setLike- nonzro pattern");
        checkLookupMechanism("like");
        m_RE = new RegularExpression();
        m_RE.setPattern(pattern);
    }


    /**
     * Returns the regular expression used to search recored logs.
     * Returns <i>null</i> if regular expression never defined.
     * @since JWare/AntX 0.3
     **/
    public final String getLike()
    {
        return (m_RE==null) ? null : m_RE.getPattern(getProject());
    }


    /**
     * Sets the property to be created on a positive evaluation.
     * Property will be set to the string "<i>true</i>."
     * @param property the property to create (non-null)
     **/
    public void setTrueProperty(String property)
    {
        require_(property!=null,"setTrueP- nonzro nam");
        m_trueProperty = property;
    }


    /**
     * Returns the property to be created/set on a positive
     * evaluation. Returns <i>null</i> if never set.
     **/
    public final String getTrueProperty()
    {
        return m_trueProperty;
    }


    /**
     * Adds next value item to this task's expected ordered value
     * sequence.
     * @throws BuildException if this task already setup to look for
     *         a single value
     * @see #setValue
     **/
    public void addConfiguredString(InnerString valueN)
    {
        require_(valueN!=null,"addString- nonzro valu");

        if (m_orderedValues.isEmpty()) {
            checkLookupMechanism("orderset");
        }
        m_orderedValues.add(valueN.toString(getProject()));
    }


    /**
     * Returns <i>true</i> if this task is already configured to
     * check an ordered sequence of values.
     **/
    public final boolean willCheckOrder()
    {
        return !m_orderedValues.isEmpty();
    }


// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Verifies that a value either exists or not within the given
     * log set.
     * @param log the logs against which value under test checked (non-null)
     * @param string value under test (non-null)
     * @param Nwant zero if the value should not exist otherwise value
     *              should exist at least once
     * @throws BuildAssertionException if value check fails
     **/
    protected final void verifyExists(String log, final String string, int Nwant)
    {
        String ermsg = null;
        int Nfound= -1;
        int i = log.indexOf(string);
        if (Nwant>0) {
            if (i == -1) {
                ermsg = "brul.logs.missing.value";
                Nfound = 0;
                Nwant = 1;
            }
        } else if (i != -1) {
            ermsg = "brul.logs.found.value";
            Nfound = 1;
            Nwant = 0;
        }
        if (ermsg!=null) {//NB:true means boo-boo found
            if (getDefaultMsg()!=null) {
                ermsg = getDefaultMsg();
            } else {
                ermsg = getAntXMsg(ermsg,String.valueOf(Nwant),
                                   String.valueOf(Nfound),string);
            }
            log(ermsg,Project.MSG_ERR);
            throw new BuildAssertionException(ermsg,getLocation());
        }
    }


    /**
     * Verifies that a value exists <i>N</i>-times in the given log set.
     * @param log the logs against which value under test checked (non-null)
     * @param string value under test (non-null)
     * @param Nwant number of times the 'string' should exist in the log
     * @throws BuildAssertionException if value check fails
     **/
    protected final void verifyOccurances(String log, String string, int Nwant)
    {
        int Nfound= 0;
        int i, from=0;
        while ((i=log.indexOf(string,from))>=0) {
            Nfound++;
            from = i+string.length();
        }
        if (Nfound!=Nwant) {
            String ermsg;
            if (getDefaultMsg()!=null) {
                ermsg = getDefaultMsg();
            } else {
                ermsg = getAntXMsg("brul.logs.mismatch.occurs",
                                   String.valueOf(Nwant),String.valueOf(Nfound),
                                   string);
            }
            log(ermsg,Project.MSG_ERR);
            throw new BuildAssertionException(ermsg,getLocation());
        }
    }


    /**
     * Verify the order in which certain marker strings were sent to the
     * captured logs.
     * @param log the logs against which string will be checked (non-null)
     * @param strings the sequence of strings as expected in log
     * @throws BuildAssertionException if value check fails
     **/
    protected final void verifyOutputInOrder(String log, String[] strings)
    {
        String ermsg=null;
        final int Nmax = log.length();
        int ilast=0;
        for (int i=0;i<strings.length;i++) {
            int at = log.indexOf(strings[i], ilast);
            if (at<0) {
                ermsg = getAntXMsg("brul.logs.outofplace",strings[i],String.valueOf(i));
                break;
            }
            ilast = at+strings[i].length();
            if ((ilast==Nmax) && (i!=strings.length-1)) {
                ermsg = getAntXMsg("brul.logs.eof",strings[i+1],strings[i],String.valueOf(i));
                break;
            }
        }
        if (ermsg!=null) {
            if (getDefaultMsg()!=null) {
                ermsg = getDefaultMsg();
            }
            log(ermsg,Project.MSG_ERR);
            throw new BuildAssertionException(ermsg,getLocation());
        }
    }


    /**
     * Verify that a regular expression matches against the given log set.
     * @param log the logs against which string will be checked (non-null)
     * @param re the Ant regular expression matches (non-null)
     * @throws BuildAssertionException if nothing matches
     * @since JWare/AntX 0.3
     **/
    protected final void verifyMatches(String log, RegexpMatcher re)
    {
        if (!re.matches(log)) {
            String ermsg = getAntXMsg("brul.logs.no.match",re.getPattern());
            log(ermsg,Project.MSG_ERR);
            throw new BuildAssertionException(ermsg,getLocation());
        }
    }


    /**
     * Evaluates this assertion as a condition. Will throw an assertion
     * exception if <i>false</i>.
     **/
    public boolean eval()
    {
        verifyCanExecute_("eval");

        boolean evaluated = false;

        String vutLog = getVUTLog();

        if (getLike()!=null) {
            verifyMatches(vutLog, m_RE.getRegexp(getProject()));
            evaluated = true;
        }
        else if (getValue()!=null) {
            int N = getOccurances();
            if (N>0) {
                verifyOccurances(vutLog,getValue(),N);
            } else {
                verifyExists(vutLog,getValue(),N==0 ? 0 : 1);
            }
            evaluated = true;
        }
        else if (!m_orderedValues.isEmpty()) {
            String[] strings = (String[])m_orderedValues.toArray(Empties.EMPTY_STRING_ARRAY);
            verifyOutputInOrder(vutLog,strings);
            strings = null;
            evaluated = true;
        }

        if (evaluated) {
            if (getTrueProperty()!=null) {
                String prop = getTrueProperty();
                log("AssertLogged was true; setting true-property '"+prop+"' property",
                    Project.MSG_DEBUG);
                getProject().setNewProperty(prop,Strings.TRUE);
            }
            if (willReset()) {
                LogsRecorder r = getRecorder(true);
                if (r!=null) {
                    r.clearLogs();
                }
            }
        }

        return true;
    }


    /**
     * Same as {@linkplain #eval evaluation} only ignoring
     * the result.
     **/
    public void execute()
    {
        eval();
    }


    /**
     * Verify this task is enclosed by at least one {@linkplain LogsRecorder
     * logs recording} taskset and has a fully defined condition.
     * @throws BuildException if neither a value or a sequence of values defined
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (m_numMechanisms==0) {
            String ermsg = getAntXMsg("brul.logs.nothing.to.check");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * Ensure there are no other lookup mechanisms defined.
     * @param from [optional] calling setter's name
     **/
    private void checkLookupMechanism(String from)
    {
        if (m_numMechanisms>0) {
            String ermsg = getAntXMsg("brul.logs.one.lookup.kind");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        m_numMechanisms++;
    }


    /** Flag to indicate looking just for existence. **/
    private static final int ATLEAST_ONCE= -1;

    private String m_requiredValue;//NB:this or nested ordered entries
    private int m_Ntimes= ATLEAST_ONCE;//NB:by default look any number
    private RegularExpression m_RE;//NB:cache the RE for 'like' lookups
    private String m_defaultMsg;//NB:inlined error message
    private String m_trueProperty;//NB:
    private ArrayList m_orderedValues= new ArrayList(5);
    private int m_numMechanisms;//NB:ensure only one defined
}

/* end-of-AssertLoggedTask.java */
