/**
 * $Id: Responses.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.apis;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.ProjectComponent;

/**
 * Collection of predefined problem handlers.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 **/

public final class Responses
{
    /**
     * Stops the build-iteration with a build error.
     * @version  0.5
     * @since    JWare/AntX 0.3
     **/
    public static final ProblemHandler DEATH=
        new ProblemHandler() {
                public void problem(Object nugget, int nl) {
                    throw new BuildError(String.valueOf(nugget));
                }
            };



    /**
     * Generates a build-iteration exception.
     * @version  0.5
     * @since    JWare/AntX 0.3
     **/
    public static final ProblemHandler ERROR=
        new ProblemHandler() {
                public void problem(Object nugget, int nl) {
                    throw new BuildException(String.valueOf(nugget));
                }
            };



    /**
     * Silently ignores the problem; no-op.
     * @version  0.5
     * @since    JWare/AntX 0.3
     **/
    public static final ProblemHandler SILENCE=
        new ProblemHandler() {
                public void problem(Object nugget, int nl) {
                    //nada
                }
            };



    /**
     * Simple problem handler that just records whether a problem was
     * reported back to the original caller as a boolean.
     * @since   JWare/AntX 0.4
     * @author  ssmc
     * @.safety single
     * @.group  impl,helper
     **/
    public static final class LitmusResult implements ProblemHandler {
        /** Reflects the result of the operation. True means a
            problem occured. **/
        public boolean hadProblem;

        /** Reflects the details of the encountered problem. **/
        public String what="";

        /** Initializes a new "no" problem result. **/
        public LitmusResult() {
        }

        /** Records operation has had some kind of problem. **/
        public void problem(Object nugget, int nl) {
            this.hadProblem = true;
            this.what = Tk.stringFrom(nugget,null);
        }
        /** Resets this response object for reuse. **/
        public void reset() {
            hadProblem = false;
            what = "";
        }
    }



    /**
     * Logs problem via its controlling log enabled component.
     * @version 0.5
     * @since   JWare/AntX 0.3
     * @.group  impl,helper
     **/
    public static class LogUsing implements ProblemHandler {
        /** Initializes new handler that logs to standard Ant component.
         *  @see com.idaremedia.antx.apis.LogEnabled.ForComponent
         **/
        public LogUsing(ProjectComponent pc) {
            logSpi = new LogEnabled.ForComponent(pc);
        }
        /** Initializes new handler that delegates to log enabled component.
         *  @since JWare/AntX 0.5
         **/
        public LogUsing(LogEnabled impl) {
            if (impl==null) {
                throw new IllegalArgumentException();
            }
            logSpi = impl;
        } 
        /** Logs problem via this response's component. **/
        public void problem(Object nugget, int nl) {
            logSpi.log(Tk.stringFrom(nugget,null),nl);
        }
        final LogEnabled logSpi;
    }



    /**
     * Logs problem via its controlling project component and
     * remember whether a problem was recorded.
     * @version 0.5
     * @since   JWare/AntX 0.4
     * @.group  impl,helper
     **/
    public static class LogAndRemember extends LogUsing {
        /** Reflects the result of the operation. True means a
            problem occured. **/
        public boolean hadProblem;

        /** Reflects the details of the encountered problem. **/
        public String what="";

        /** Initializes new handler that logs to standard Ant component and
         *  remembers whether had problem.
         **/
        public LogAndRemember(ProjectComponent pc) {
            super(pc);
        }
        /** Initializes new handler that delegates to log enabled component
         *  and remembers problem occured.
         *  @since JWare/AntX 0.5
         **/
        public LogAndRemember(LogEnabled impl) {
            super(impl);
        }
        /** Records there was a problem and logs problem via  this
         *  handler's log enabled component.
         **/
        public void problem(Object nugget, int nl) {
            this.hadProblem = true;
            this.what = Tk.stringFrom(nugget,null);
            logSpi.log(this.what,nl);
        }
        /** Resets this response object for reuse. **/
        public void reset() {
            hadProblem = false;
            what = "";
        }
    }


    /** Prevent; only static APIs and classes. **/
    private Responses()
    {}
}

/* end-of-Responses.java */
