/**
 * $Id: KillMethodSkeleton.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.Responses;
import  com.idaremedia.antx.helpers.Strings;

/**
 * Starter implementation of a fixture cleanup callback. Subclasses should fill in
 * the specific kill method (context,default,current) that applies.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public class KillMethodSkeleton implements KillMethod
{
    /**
     * Initializes a new silent killer of nothing and everything.
     **/
    protected KillMethodSkeleton()
    {
    }



    /**
     * Initializes a general fixture administrator kill method.
     * @param FXID the administrator's iteration identifier
     * @param name a script-facing label for administrator
     * @see FixtureIds
     **/
    public KillMethodSkeleton(String FXID, String name)
    {
        m_FXID = FXID;
        m_name = name;
    }



    /**
     * Return <i>true</i> if this kill method understands target.
     * @param target symbolic name of thing to be killed
     **/
    public boolean canKill(String target)
    {
        return true;
    }



    /**
     * Returns the fixture component's FixtureOverlays identifier.
     **/
    protected String getComponentId()
    {
        AntX.verify_(m_FXID!=null,"Fixture.KillMethod:","nonzro FXId");
        return m_FXID;
    }



    /**
     * Returns the fixture component's general classification.
     * By default just returns value of {@linkplain #getComponentId}.
     **/
    protected String getComponentName()
    {
        return m_name!=null ? m_name : getComponentId();
    }



    /**
     * Calls {@linkplain #kill(String,ProblemHandler) kill(String,Source)}
     * with symbolic "<span class="src">all</span>" target string.
     * @param from controlling task (non-null)
     **/
    public final boolean kill(ProblemHandler from)
    {
        return kill(Strings.ALL,from);
    }



    /**
     * Should unset the default (fall back) fixture component.
     * This version does nothing by default; subclasses can
     * customize.
     * @param from controlling task (non-null)
     **/
    protected boolean killDefault(ProblemHandler from)
    {
        return true;
    }



    /**
     * Removes all the fixture component's information from the
     * FixtureOverlays stack. Operation is thread-bound; only affects
     * the current thread's iteration information.
     * @param from controlling task (non-null)
     **/
    protected boolean killContext(ProblemHandler from)
    {
        FixtureOverlays overlays = FixtureOverlays.getContextInstance();
        overlays.clear(getComponentId());
        return true;
    }



    /**
     * Removes the frontmost component instance from the
     * FixtureOverlays stack. Operation is thread-bound; only affects
     * the current thread's iteration information.
     * @param from controlling task (non-null)
     **/
    protected boolean killCurrent(ProblemHandler from)
    {
        Responses.LitmusResult result= new Responses.LitmusResult();
        FixtureOverlays.uninstallIfIs(getComponentId(),result,getComponentName());
        if (result.hadProblem) {
            from.problem(result.what,Project.MSG_WARN);
        }
        return !result.hadProblem;
    }



    /**
     * Based on the requested target, removes or resets some aspect
     * of this method's associated fixture component.
     * @param target symbolic name of thing to be affected
     * @param from controlling task or test (non-null)
     **/
    public boolean kill(String target, ProblemHandler from)
    {
        AntX.require_(from!=null,"Fixture.KillMethod:","kill- nonzro src");
        if (Strings.ALL.equals(target)) {
            return killContext(from) && killDefault(from);
        } else if (Strings.CURRENT.equals(target)) {
            return killCurrent(from);
        } else if (Strings.CONTEXT.equals(target)) {
            return killContext(from);
        } else if (Strings.DEFAULT.equals(target)) {
            return killDefault(from);
        }
        return true;
    }


    private String m_FXID,m_name;
}

/* end-of-KillMethodSkeleton.java */
