/**
 * $Id: ExportedProperties.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.util.ArrayList;
import  java.util.Collection;
import  java.util.HashMap;
import  java.util.Iterator;
import  java.util.Map;
import  java.util.Properties;

import  com.idaremedia.antx.apis.ProblemHandler;

/**
 * Manager of a build-iteration's thread-local modifiable properties. This class is
 * used heavily by the core {@linkplain com.idaremedia.antx.solo.ExportTask ExportTask}
 * and {@linkplain com.idaremedia.antx.solo.ImportTask ImportTask} tasks.
 * This class is almost always used via the various {@linkplain Iteration singleton}
 * static APIs like {@linkplain #readstring readstring}, {@linkplain #set set},
 * and {@linkplain #delete delete}.
 * <p>
 * This class administers a <em>single</em> thread-independent collection of
 * thread-specific elements. In other words, for each property there is a single named
 * collection of thread-specific values. This is different from how other thread-based
 * iteration information is stored. See the {@linkplain FixtureOverlays} class for further
 * details.
 * <p>
 * This class's effectiveness relies on a single instance being used. Within Ant certain
 * tasks can create multiple class loaders hence muddying the singleton concept because
 * class loaders are a hidden (or forgotten) link in the definition of a unique entity
 * within a VM's object space. Soooo my advice is to use modifiable properties with care
 * when mucking with multiple, user controlled, class loaders within a single thread.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,helper
 * @see      com.idaremedia.antx.solo.ExportTask
 * @see      PropertyHandle
 * @see      Iteration#exportableProperties
 **/

public final class ExportedProperties implements FixtureCore, FixtureAdministrator
{
    private static final String IAM_ = "ExportableProperties:";



    /**
     * If you really want your own set of modifiable exportable
     * properties for something.
     * @see Iteration#exportableProperties
     **/
    public ExportedProperties()
    {
    }



    /**
     * Returns modifiable property handle for named item.
     * @param name the property's name (non-null)
     **/
    public PropertyHandle getProperty(String name)
    {
        AntX.require_(name!=null,IAM_,"get- nonzro name");

        synchronized(m_handles) {
            PropertyHandle ph = (PropertyHandle)m_handles.get(name);
            if (ph==null) {
                ph = new PropertyHandle();
                m_handles.put(name, ph);
            }
            return ph;
        }
    }



    /**
     * Convenience to get(create) and set a property handle in single
     * operation.
     * @param name the property's name (non-null)
     * @param valu the property's value
     **/
    public void putProperty(String name, Object valu)
    {
        PropertyHandle ph = getProperty(name);
        ph.set(valu);
    }



    /**
     * Returns <i>true</i> if a property handle already exists for
     * given name. This method does not check the property handle's
     * contents.
     * @param name the property's name  (non-null)
     **/
    public boolean knowsProperty(String name)
    {
        AntX.require_(name!=null,IAM_,"check- nonzro name");

        synchronized(m_handles) {
            return m_handles.containsKey(name);
        }
    }



    /**
     * Convenience to read a property's value iff it's defined or
     * return some default value. Returns <i>null</i> if the property
     * is defined and it's current value is <i>null</i>-- will not
     * return the default value.
     * @param name the property's name (non-null)
     * @param dflt default value return iff property not defined
     **/
    public Object readProperty(String name, Object dflt)
    {
        AntX.require_(name!=null,IAM_,"read- nonzro name");

        synchronized(m_handles) {
            PropertyHandle ph= (PropertyHandle)m_handles.get(name);
            return (ph!=null) ? ph.get() : dflt;
        }
    }



    /**
     * Convenience to clear the property handle's value in a single
     * operation. Any existing property handle is returned (can be
     * <i>null</i>).
     * @param name the property's name (non-null)
     **/
    public PropertyHandle removeProperty(String name)
    {
        AntX.require_(name!=null,IAM_,"remove- nonzro name");

        synchronized(m_handles) {
            return (PropertyHandle)m_handles.remove(name);
        }
    }



    /**
     * Convenience to clear a collection of properties in a single
     * operation.
     * @param names the collection of properties to delete (non-null)
     * @since JWare/AntX 0.5
     **/
    public void removeProperties(Collection names)
    {
        AntX.require_(names!=null,IAM_,"remove- nonzro names");

        synchronized(m_handles) {
            m_handles.keySet().removeAll(names);
        }
    }



    /**
     * Convenience to clear a collection of properties in a single
     * operation. Only properties with names matching those in the
     * given collection are retained.
     * @param names the collection of properties to retain (non-null)
     * @since JWare/AntX 0.5
     **/
    public void retainProperties(Collection names)
    {
        AntX.require_(names!=null,IAM_,"retain- nonzro names");

        synchronized(m_handles) {
            m_handles.keySet().retainAll(names);
        }
    }



    /**
     * Returns detached copy of all properties in current repository. The
     * returned Properties item is a regular key->value mapping. Useful
     * for capturing environment diagnostics.
     * @param existingP Properties to be updated (pass <i>null</i> for new)
     * @since JWare/AntX 0.2
     **/
    public Properties copyOfProperties(Properties existingP)
    {
        if (existingP==null) {
            existingP= new Properties();
        }
        synchronized(m_handles) {
            Iterator itr = m_handles.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry mE= (Map.Entry)itr.next();
                PropertyHandle ph= (PropertyHandle)mE.getValue();
                existingP.put(mE.getKey(),String.valueOf(ph.get()));//cvt null to "null"
            }
        }
        return existingP;
    }



    /**
     * Returns detached copy of names of all properties in current 
     * repository. The returns Collection contains property names.
     * This is used to capture environment diagnostics and support
     * execution bubbles.
     * @since JWare/AntX 0.5
     **/
    public Collection copyOfPropertyNames()
    {
        synchronized(m_handles) {
            return new ArrayList(m_handles.keySet());
        }
    }



    /**
     * Clears all properties for all threads in current repository. Your
     * application should never use this method directly; it is provided for
     * test harness to reset the environment to a known state.
     * @since JWare/AntX 0.4
     **/
    void clearProperties()
    {
        synchronized(m_handles) {
            m_handles.clear();
            m_handles = new HashMap(111,0.8f);
        }
    }


    /**
     * Instance-level member fields.
     **/
    private HashMap m_handles= new HashMap(111,0.8f);



//---------------------------------------------------------------------------------------------------------|
// Bunch of convenient utility APIs for the Iteration singleton:
//---------------------------------------------------------------------------------------------------------|


    /**
     * Returns a complete independent copy of the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     * @since JWare/AntX 0.5
     **/
    public static final Properties copy(Properties fillin)
    {
        return Iteration.exportableProperties().copyOfProperties(fillin);
    }


    /**
     * Returns <i>true</i> if the named property is defined in the
     * current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     * @since JWare/AntX 0.5
     **/
    public static final boolean has(String name)
    {
        return Iteration.exportableProperties().knowsProperty(name);
    }


    /**
     * Returns a property from the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     **/
    public static final PropertyHandle get(String name)
    {
        return Iteration.exportableProperties().getProperty(name);
    }


    /**
     * Updates a property value in the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     **/
    public static final void set(String name, Object valu)
    {
        Iteration.exportableProperties().putProperty(name,valu);
    }


    /**
     * Clears a property value in the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     * The property is still known to the iteration. Use {@linkplain #delete
     * delete(name)} to remove a property from the iteration.
     * @see #delete
     **/
    public static final void unset(String name)
    {
        Iteration.exportableProperties().putProperty(name,null);
    }


    /**
     * Reads a property value from the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}
     * returning a default value if property not defined.
     **/
    public static final Object read(String name, Object dflt)
    {
        return Iteration.exportableProperties().readProperty(name,dflt);
    }


    /**
     * Reads a property value from the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}
     * returning a default string if property not defined.
     **/
    public static final String readstring(String name, String dflt)
    {
        return (String)Iteration.exportableProperties().readProperty(name,dflt);
    }


    /**
     * Reads a property value from current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}
     * returning <i>null</i> if property not defined.
     **/
    public static final Object read(String name)
    {
        return Iteration.exportableProperties().readProperty(name,null);
    }


    /**
     * Reads a property value from current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}
     * returning <i>null</i> if property not defined.
     **/
    public static final String readstring(String name)
    {
        return (String)Iteration.exportableProperties().readProperty(name,null);
    }


    /**
     * Deletes a property from the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     **/
    public static final void delete(String name)
    {
        Iteration.exportableProperties().removeProperty(name);
    }


    /**
     * Deletes a collection of properties from the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     * @since JWare/AntX 0.5
     **/
    public static final void delete(Collection names)
    {
        Iteration.exportableProperties().removeProperties(names);
    }


    /**
     * Deletes a collection of properties from the current iteration's 
     * {@linkplain Iteration#exportableProperties exportable properties}.
     * Only properties with names matching those in the given collection
     * are retained.
     * @since JWare/AntX 0.5
     **/
    public static final void retain(Collection names)
    {
        Iteration.exportableProperties().retainProperties(names);
    }


//---------------------------------------------------------------------------------------------------------|
// Iteration reset for execution harnesses;
//---------------------------------------------------------------------------------------------------------|

    /**
     * Installs fixture component cleanup method.
     * @since JWare/AntX 0.4
     **/
    static {
        AntXFixture.setKillMethod
            (FixtureIds.VARIABLES_ADMINISTRATION,
             new String[]{"variables","exportedproperties"},
             new KillMethod() {
                     public boolean kill(ProblemHandler from) {
                         /*!*/
                         Iteration.exportableProperties().clearProperties();
                         return true;
                     }
                     public boolean kill(String target, ProblemHandler from) {
                         return kill(from);
                     }
                 }
             );
    }
}

/* end-of-ExportedProperties.java */
