/**
 * $Id: ExportScope.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.EnumSkeleton;

/**
 * An export's range of influence. Three options: <i>all</i>, <i>thread</i>, or
 * <i>project</i>.
 *
 * @since    JWare/AntX 0.2 (Moved out of ExportTask AntX 0.3)
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,helper
 * @see      ExportedProperties
 * @see      com.idaremedia.antx.solo.ExportTask
 * @see      com.idaremedia.antx.solo.ImportTask
 **/

public final class ExportScope extends EnumSkeleton
{
    /** Index of {@linkplain #ALL ALL} scope. **/
    public final int ALL_INDEX = 0;
    /** Index of {@linkplain #THREAD THREAD} scope. **/
    public final int THREAD_INDEX = ALL_INDEX+1;
    /** Index of {@linkplain #PROJECT PROJECT} scope. **/
    public final int PROJECT_INDEX = THREAD_INDEX+1;


    /**
     * VM-shareable instance of <i>thread</i> scope.
     **/
    public static final ExportScope THREAD = new ExportScope("thread");

    /**
     * VM-shareable instance of <i>project</i> scope.
     **/
    public static final ExportScope PROJECT = new ExportScope("project");

    /**
     * VM-shareable instance of <i>all</i> scope.
     **/
    public static final ExportScope ALL  = new ExportScope("all");



    /**
     * Required void constructor for Ant's introspector.
     **/
    public ExportScope()
    {
        super();
    }


    /**
       Private constructor to create our own singletons.
    **/
    private ExportScope(String v)
    {
        super(v);
    }


    /**
     * Returns copy of all possible values as ordered string
     * array. Ordering must be same as the _INDEX constants.
     **/
    public String[] getValues()
    {
        return new String[]{"all","thread","project"};
    }



    /**
     * The default scope used by all exporting tasks
     * within AntX.
     **/
    public static final ExportScope DEFAULT_SCOPE= ExportScope.THREAD;



    /**
     * Updates a modifiable property with given value
     * depending on the specified scope.
     * @param S the variable's scope (null => {@linkplain #DEFAULT_SCOPE})
     * @param P the active project (non-null if scope != THREAD)
     * @param name the variable's name (non-null)
     * @param value the variable's value (non-null)
     * @param erase <i>true</i> if an empty value should remove variable
     **/
    public static void setTheProperty(ExportScope S, Project P,
                                      String name, String value,
                                      boolean erase)
    {
        if (S==null) {
            S= ExportScope.DEFAULT_SCOPE;
        }
        if (erase) {
            erase = value.length()==0;//NB: only vars can be zapped (ssmc)
        }
        if (ExportScope.THREAD.equals(S)) {
            if (erase) {
                ExportedProperties.delete(name);
            } else {
                ExportedProperties.set(name,value);
            }
        }
        else {
            if (P==null) {
                throw new IllegalArgumentException
                    ("Export.setTheProperty- nonNULL project required");
            }
            if (ExportScope.PROJECT.equals(S)) {
                P.addReference(name,value);
            }
            else {
                P.setInheritedProperty(name,value);
            }
        }
    }


    /**
     * Returns a modifiable property's current value given its
     * declared scope. Will return <i>null</i> if property never
     * set. Uses a strict {@linkplain Stringifier} if a non-string
     * reference is specified.
     * @param S the variable's scope (<i>null</i> => {@linkplain #DEFAULT_SCOPE})
     * @param P the active project (non-null if scope != THREAD)
     * @param name the variable's name (non-null)
     **/
    public static String getTheProperty(ExportScope S, Project P, String name)
    {
        if (S==null) {
            S= ExportScope.DEFAULT_SCOPE;
        }
        if (ExportScope.THREAD.equals(S)) {
            return ExportedProperties.readstring(name);
        }
        if (P==null) {
            throw new IllegalArgumentException
                ("Export.getTheProperty- nonNULL project required");
        }
        if (ExportScope.PROJECT.equals(S)) {
            return Stringifier.get(false).stringFrom(P.getReference(name),P);
        }
        return P.getProperty(name);
    }




    /**
     * Helper that converts a scalar to a known scope. Returns 
     * <i>null</i> if value does not match any of expected indices.
     * @param i the index to be matched
     * @since JWare/AntX 0.5
     **/
    public static ExportScope from(int i)
    {
        if (i==ALL.index)     { return ALL; }
        if (i==PROJECT.index) { return PROJECT; }
        if (i==THREAD.index)  { return THREAD; }
        return null;
    }



    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known indices.
     * @param i the index to be matched
     * @param dflt the default scope if necessary
     * @since JWare/AntX 0.5
     **/
    public static ExportScope from(int i, ExportScope dflt)
    {
        ExportScope xs= from(i);
        return (xs==null) ? dflt : xs;
    }



    /**
     * Helper that converts a string to a known ExportScope singleton.
     * Returns <i>null</i> if string unrecognized. String can be
     * either ExportScope's symbolic name or its index.
     * @since JWare/AntX 0.5
     **/
    public static ExportScope from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if ("default".equals(s))     { return DEFAULT_SCOPE; }
                if (PROJECT.value.equals(s)) { return PROJECT; }
                if (THREAD.value.equals(s))  { return THREAD; }
                if (ALL.value.equals(s))     { return ALL;}
                if ("system".equals(s))      { return ALL;}
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(java.lang.String) from(String)} but
     * with a default value if value does not match any known
     * ExportScope's name.
     * @param s the symbolic name to be matched
     * @param dflt the default ExportScope if necessary
     * @since JWare/AntX 0.5
     **/
    public static ExportScope from(String s, ExportScope dflt)
    {
        ExportScope xs= from(s);
        return (xs==null) ? dflt : xs;
    }
}

/* end-of-ExportScope.java */
