/**
 * $Id: ExcludedFixture.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.util.Collection;
import  java.util.List;

/**
 * Default fixture exclusions for current application and/or iteration.  A fixture 
 * exclusion can apply to certain AntX fixture components like overlays and bubbles.
 * See each task's documentation for details on how/if if uses default fixture
 * exclusions.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   guarded
 * @.group    impl,helper
 * @see       Iteration#defaultFixtureExcludes
 **/

public class ExcludedFixture
{
    /**
     * Initializes a new excluded fixture instance.
     **/
    public ExcludedFixture()
    {
    }


    /**
     * Adds a property's name to list of excluded fixture.
     * @param property the name (non-null)
     **/
    public void addProperty(String property)
    {
        AntX.require_(property!=null,AntX.nopackage,"addProp- nonzro name");
        m_xp.add(property);
    }


    /**
     * Adds a set of property names en-masse to list of
     * excluded fixture.
     * @param properties the collection of properties (non-null)
     **/
    public void addProperties(Collection properties)
    {
        AntX.require_(properties!=null,AntX.nopackage,"addProps- nonzro lst");
        m_xp.addAll(properties);
    }


    /**
     * Adds a reference id to list of excluded fixture.
     * @param refid the reference id (non-null)
     **/
    public void addReference(String refid)
    {
        AntX.require_(refid!=null,AntX.nopackage,"addRef- nonzro refid");
        m_xr.add(refid);
    }


    /**
     * Adds a set of reference identifiers en-masse to list
     * of excluded fixture.
     * @param refidset the collection of reference ids (non-null)
     **/
    public void addReferences(Collection refidset)
    {
        AntX.require_(refidset!=null,AntX.nopackage,"addRefs- nonzro lst");
        m_xr.addAll(refidset);
    }


    /**
     * Returns a <em>copy</em> of the current set of excluded
     * properties. Returns a new list even if there are no
     * excluded properties.
     **/
    public List copyOfProperties()
    {
        return AntXFixture.newListCopy(m_xp);
    }



    /**
     * Returns a <em>copy</em> of the current set of excluded
     * properties if there are any. Otherwise returns <i>null</i>.
     **/
    public List copyOfPropertiesIfAny()
    {
        synchronized(m_xp) {
            if (m_xp.isEmpty()) {
                return null;
            }
            return AntXFixture.newListCopy(m_xp);
        }
    }



    /**
     * Copies the list of excluded properties directly into an
     * existing collection of property names.
     * @param properties collection of property names to update (non-null)
     **/
    public void addPropertiesTo(Collection properties)
    {
        AntX.require_(properties!=null,AntX.nopackage,"addPropsTo- nonzro lst");
        synchronized(m_xp) {
            if (!m_xp.isEmpty()) {
                properties.addAll(m_xp);
            }
        }
    }


    /**
     * Returns a <em>copy</em> of the current set of excluded
     * reference ids. Returns a new list even if there are no
     * excluded references.
     **/
    public List copyOfReferences()
    {
        return AntXFixture.newListCopy(m_xr);
    }



    /**
     * Returns a <em>copy</em> of the current set of excluded
     * references if there are any. Otherwise returns <i>null</i>.
     **/
    public List copyOfReferencesIfAny()
    {
        synchronized(m_xr) {
            if (m_xr.isEmpty()) {
                return null;
            }
            return AntXFixture.newListCopy(m_xr);
        }
    }



    /**
     * Copies the list of excluded reference ids directly into an
     * existing collection of reference names.
     * @param  refidset the collection of reference ids to update (non-null)
     **/
    public void addReferencesTo(Collection refidset)
    {
        AntX.require_(refidset!=null,AntX.nopackage,"addRefsTo- nonzro lst");
        synchronized(m_xr) {
            if (!m_xr.isEmpty()) {
                refidset.addAll(m_xr);
            }
        }
    }


    /** Thread-safe list of excluded property names. **/
    protected List m_xp= AntXFixture.newSynchronizedList();
    
    /** Thread-safe list of excluded reference names. **/
    protected List m_xr= AntXFixture.newSynchronizedList();
}


/* end-of-ExcludedFixture.java */