/**
 * $Id: AntXFixture.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2004-2005,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.apis.FixtureStarter;

import  com.idaremedia.antx.apis.BuildError;
import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.helpers.Empties;

/**
 * The AntX global fixture. The AntXFixture serves as a factory for standard Java and Ant
 * data types as well as a controller that resets (or recreates) various core AntX fixture
 * components in between execution iterations. Every administrator of fixture data must
 * define a iteration-bound clean-up method by supplying a {@linkplain KillMethod}
 * callback. 
 * <p/>
 * The AntXFixture differs from the {@linkplain Iteration} in that it is global,
 * always exists, and is <em>independent</em> of any particular execution iteration. 
 * For instance, kill methods are registered at the class level, not the iteration level.
 * Multiple iterations will use the same kill method to reset certain components.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004-2005,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  guarded
 * @.group   impl,infra
 * @.pattern GoF.Facade
 * @see      Iteration
 **/

public final class AntXFixture extends FixtureStarter implements FixtureCore
{
    private static final String IAM_= "AntX.Fixture:";



    /**
     * Returns a shareable <span class="src">FileUtils</span>
     * helper instance.
     * @since JWare/AntX 0.4
     **/
    public static FileUtils fileUtils()
    {
        return FileUtils.getFileUtils();
    }



    /**
     * Returns the kill method associated with target. Will return
     * <i>null</i> if nothing registered for target.
     **/
    static KillMethod getKillMethod(String target)
    {
        return (KillMethod)m_killMethods.get(target);
    }



    /**
     * Defines the fixture cleanup method for a specific administrator.
     * This method is used by the fixture component class when it is loaded.
     * @param component fixture administrator or component (non-null)
     * @param method cleanup method (non-null)
     * @throws BuildError if component already installed
     **/
    public static void setKillMethod(String component, KillMethod method)
    {
        setKillMethod(component, Empties.EMPTY_STRING_ARRAY, method);
    }



    /**
     * Defines the fixture cleanup method for a specific administrator
     * and its common script-facing aliases. This method is used by the
     * fixture component class when it is loaded.
     * @param component fixture administrator or component (non-null)
     * @param aliases component aliases (non-null)
     * @param method cleanup method (non-null)
     * @throws BuildError if component already installed
     * @throws IllegalArgumentException if any alias is already registered
     **/
    public static void setKillMethod(String component, String[] aliases,
                                     KillMethod method)
    {
        AntX.require_(component!=null,IAM_,"setKillMeth- nonzro id");
        AntX.require_(method!=null,IAM_,"setKillMeth- nonzro method");
        AntX.require_(aliases!=null,IAM_,"setKillMeth- nonzro aliases");

        KillMethod method0 = getKillMethod(component);
        if (method0!=null) {
            throw new BuildError
                (AntX.uistrs().get("fixture.cleanup.instald",component));
        }

        m_killMethods.put(component,method);

        for (int i=0;i<aliases.length;i++) {
            AntX.require_(m_killMethods.put(aliases[i],method)==null, IAM_,
                          "setKillMeth- unique component alias("+aliases[i]+")");
        }

        m_FXIDs.add(component);
    }



    /**
     * Resets a particular aspect of a fixture component. If there is
     * no cleanup for the component registered, this method will return
     * <i>true</i>.
     * @param component fixture component to be reset (non-null)
     * @param aspect aspect to be reset
     * @param from controlling task or test (non-null)
     * @return <i>false</i> if there was a problem with the cleanup
     **/
    public static boolean reset(String component, String aspect,
                                ProblemHandler from)
    {
        AntX.require_(from!=null,IAM_,"reset- nonzro source");
        AntX.require_(component!=null,IAM_,"reset- nonzro id");

        KillMethod method0 = getKillMethod(component);
        if (method0!=null) {
            boolean ok = method0.kill(aspect, from);
            if (!ok) {
                String warning = AntX.uistrs().get
                    ("fixture.errs.killin.comp.aspect", component,aspect);
                from.problem(warning, Project.MSG_WARN);
            }
            return ok;
        }
        return true;
    }



    /**
     * Resets a particular aspect of a fixture component. If there is
     * no cleanup for the component registered, this method will use the
     * supplied fall back cleanup method.
     * @param component fixture component to be reset (non-null)
     * @param aspect aspect to be reset
     * @param fallback fall back cleanup method (non-null)
     * @param from controlling task or test (non-null)
     * @return <i>false</i> if there was a problem with the cleanup
     **/
    public static boolean reset(String component, String aspect,
                                KillMethod fallback, ProblemHandler from)
    {
        AntX.require_(from!=null,IAM_,"reset- nonzro source");
        AntX.require_(component!=null,IAM_,"reset- nonzro id");
        AntX.require_(fallback!=null,IAM_,"reset- nonzro fallback");

        KillMethod method0 = getKillMethod(component);
        if (method0==null) {
            method0= fallback;
        }
        boolean ok = method0.kill(aspect, from);
        if (!ok) {
            String warning = AntX.uistrs().get
                ("fixture.errs.killin.comp.aspect", component,aspect);
            from.problem(warning, Project.MSG_WARN);
        }
        return ok;
    }



    /**
     * Resets <em>all</em> fixture components. Your application should
     * never use this method directly unless it is part of a execution
     * harness system. Call this method before resetting the iteration
     * to a new instance.
     * @param from controlling task or test (non-null)
     **/
    public static void reset(ProblemHandler from)
    {
        AntX.require_(from!=null,IAM_,"reset- nonzro source");

        List l= FixtureStarter.newListCopy(m_FXIDs);

        Iterator itr= l.iterator();
        while (itr.hasNext()) {
            String component = itr.next().toString();
            KillMethod method = getKillMethod(component);
            if (!method.kill(from)) {
                String warning = AntX.uistrs().get
                    ("fixture.errs.killin.comp", component);
                from.problem(warning, Project.MSG_WARN);
            }
        }

        l=null;
    }



    /**
     * Disallow; only public static utility methods.
     **/
    private AntXFixture()
    {
    }


    private static final List m_FXIDs= FixtureStarter.newSynchronizedList();
    private static final Map m_killMethods= FixtureStarter.newSynchronizedMap();
}

/* end-of-AntXFixture.java */
