<!--
  This stylesheet can be used to an convert XHTML document to the PSML format.

  The content must be using the XHTML namespace 'http://www.w3.org/1999/xhtml'.

  Fragments are represented using the 'body' tag.
  It can have the following attributes:
   - 'data-format' to specify the kind of fragment 'xref-fragment' or 'properties-fragment' (the PSML fragment element)
   - 'data-type' to specify the fragment type (the PSML type attribute on the fragment)

  The Fragment ID is supplied by PageSeeder to the transformer.

  @author Philip Rutherford
  @author Christophe Lauret
  @version 29 April 2019
-->
<xsl:transform version="2.0"
  xmlns:xhtml="http://www.w3.org/1999/xhtml"
  xmlns:xs="http://www.w3.org/2001/XMLSchema"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:psml="http://pageseeder.com/pmsl"
  xpath-default-namespace="http://www.w3.org/1999/xhtml"
  exclude-result-prefixes="xhtml xs psml">

<!-- Use UTF-8 as we can store the content in UTF-8 in the database -->
<xsl:output method="xml" indent="no" omit-xml-declaration="yes" encoding="utf-8" />

<!-- Fragment Identifier, sent by PageSeeder -->
<xsl:param name="fragment" select="''" />

<!-- Folder to the URI, sent by PageSeeder -->
<xsl:param name="urifolder" select="''" />

<!-- Site prefix (e.g. '/ps'), sent by PageSeeder -->
<xsl:param name="siteprefix" select="''" />

<xsl:template match="/">
  <xsl:apply-templates select="descendant::body"/>
</xsl:template>

<!-- ========================================================================================= -->
<!-- DEFAULT FRAGMENTS                                                                         -->
<!-- ========================================================================================= -->

<!--
  <fragment>
-->
<xsl:template match="body">
<fragment>
  <xsl:if test="$fragment != ''">
    <xsl:attribute name="id"><xsl:value-of select="$fragment"/></xsl:attribute>
  </xsl:if>
  <xsl:if test="@data-type">
    <xsl:attribute name="type" select="@data-type" />
  </xsl:if>
  <!-- Only select elements -->
  <xsl:apply-templates select="*"/>
</fragment>
</xsl:template>

<!-- ========================================================================================= -->
<!-- XREFS FRAGMENTS                                                                           -->
<!-- ========================================================================================= -->

<!--
  <xref-fragment>
-->
<xsl:template match="body[@data-format='xref-fragment']">
  <xref-fragment>
    <xsl:if test="$fragment != ''">
      <xsl:attribute name="id"><xsl:value-of select="$fragment"/></xsl:attribute>
    </xsl:if>
    <xsl:if test="@data-type">
      <xsl:attribute name="type" select="@data-type" />
    </xsl:if>
    <xsl:apply-templates/>
  </xref-fragment>
</xsl:template>

<!-- ========================================================================================= -->
<!-- PROPERTIES FRAGMENTS                                                                      -->
<!-- ========================================================================================= -->

<!--
  <properties-fragment>
-->
<xsl:template match="body[@data-format='properties-fragment']">
  <properties-fragment>
    <xsl:if test="$fragment != ''">
      <xsl:attribute name="id"><xsl:value-of select="$fragment"/></xsl:attribute>
    </xsl:if>
    <xsl:if test="@data-type">
      <xsl:attribute name="type" select="@data-type" />
    </xsl:if>
    <xsl:apply-templates select="descendant::tr" mode="properties"/>
  </properties-fragment>
</xsl:template>

<!--
  <metadata>
-->
<xsl:template match="body[@data-format='properties']">
  <properties>
    <xsl:apply-templates select="descendant::tr" mode="properties"/>
  </properties>
</xsl:template>

<!--
  Each property is in a table row as:

  <tr data-name="[name]" data-datatype="[datatype]" data-count="[count]" data-multiple="[true|false]">
    <th>[title]<th>
    <td>[value]</td>
  </tr>
-->
<xsl:template match="tr" mode="properties">
  <property name="{@data-name}" title="{th}">
    <xsl:variable name="datatype" select="@data-datatype" />
    <xsl:if test="@data-datatype"><xsl:attribute name="datatype" select="@data-datatype"/></xsl:if>
    <xsl:choose>
      <xsl:when test="@data-multiple"><xsl:attribute name="multiple" select="@data-multiple"/></xsl:when>
      <xsl:when test="@data-count"><xsl:attribute name="count" select="@data-count"/></xsl:when>
      <xsl:when test="descendant::li"><xsl:attribute name="count" select="'n'"/></xsl:when>
    </xsl:choose>
    <xsl:choose>
      <xsl:when test="$datatype='markdown'">
        <!-- trim leading and trailing new lines added by tidy -->
        <markdown><xsl:value-of select="td/pre"/></markdown>
      </xsl:when>
      <!-- Multiple values -->
      <xsl:when test="(@data-count and @data-count != '1') or @data-multiple = 'true' or (not(@data-count) and descendant::li)">
        <xsl:for-each select="descendant::li">
          <xsl:choose>
            <xsl:when test="$datatype='xref' or a"><xsl:apply-templates /></xsl:when>
            <xsl:otherwise><value><xsl:value-of select="." /></value></xsl:otherwise>
          </xsl:choose>
        </xsl:for-each>
      </xsl:when>
      <!-- Single values -->
      <xsl:when test="not($datatype='xref') and not(td/*)"><xsl:attribute name="value" select="td" /></xsl:when>
      <xsl:otherwise><xsl:apply-templates select="td/*" /></xsl:otherwise>
    </xsl:choose>
  </property>
</xsl:template>

<!-- ========================================================================================= -->
<!-- MEDIA FRAGMENTS                                                                           -->
<!-- ========================================================================================= -->

<!--
  <media-fragment>
-->
<xsl:template match="body[@data-format='media-fragment']">
  <!-- XXX: This may not be the correct way to handle media fragments -->
  <media-fragment>
    <xsl:if test="$fragment != ''">
      <xsl:attribute name="id"><xsl:value-of select="$fragment"/></xsl:attribute>
    </xsl:if>
    <xsl:copy-of select="*|text()"/>
  </media-fragment>
</xsl:template>

<!-- ========================================================================================= -->
<!-- BLOCK ELEMENTS                                                                            -->
<!-- ========================================================================================= -->

<!--
  All div elements will be converted to para elements by this template.

  However, div elements can sometimes be generated under li elements.
  Since para elements are not allowed as children of li elements, such divs must be ignored,
  and only their children should be copied.

  The only two elements that may contain a para element are body and cell.
-->
<xsl:template match="div">
<xsl:variable name="valid-div-parent" select="parent::body or parent::td"/>
<xsl:choose>
  <xsl:when test="self::div and (not($valid-div-parent))">
    <xsl:apply-templates/>
  </xsl:when>
  <xsl:otherwise>
    <xsl:for-each select="./node()">
      <xsl:variable name="previous" select="preceding-sibling::node()[1]" />
      <xsl:choose>
        <xsl:when test="psml:top-level(.)">
          <xsl:apply-templates select="."/>
        </xsl:when>
        <!-- when first non top level, non whitespace node start wrapping -->
        <xsl:when test="(not($previous) or psml:top-level($previous)) and
            not(self::text()[string-length(replace(., '\s', '')) = 0])">
          <!-- ignore leading <br/> elements -->
          <xsl:choose>
            <xsl:when test="not(self::br)">
              <xsl:element name="para">
                <xsl:call-template name="wrap-nodes">
                  <xsl:with-param name="node" select="." />
                </xsl:call-template>
              </xsl:element>
            </xsl:when>
            <!--
              all following nodes should be wrapped
             -->
            <xsl:otherwise>
              <xsl:for-each select="following-sibling::node()[not(self::br)][1]">
                <xsl:if test="not(psml:top-level(.))">
                  <xsl:element name="para">
                    <xsl:call-template name="wrap-nodes">
                      <xsl:with-param name="node" select="." />
                    </xsl:call-template>
                  </xsl:element>
                </xsl:if>
              </xsl:for-each>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
      </xsl:choose>
    </xsl:for-each>
  </xsl:otherwise>
</xsl:choose>
</xsl:template>

<!--
  Block labels
-->
<xsl:template match="div[@data-label or @label]">
  <block label="{if (@data-label) then @data-label else @label}">
    <xsl:apply-templates />
  </block>
</xsl:template>

<!--
  Block xrefs
-->
<xsl:template match="div[(@class='xref' or starts-with(@class, 'xref ')) and not(@label or @data-label)]">
  <xsl:if test="string-length(.) != 0">
    <xsl:variable name="manualtitle" select="psml:get-manualtitle(.)" />
    <blockxref>
      <xsl:if test="$manualtitle != ''"><xsl:attribute name="title" select="$manualtitle" /></xsl:if>
      <xsl:variable name="uriid"  select="psml:get-data(., 'uriid')"/>
      <xsl:variable name="docid"  select="psml:get-data(., 'docid')"/>
      <xsl:choose>
        <xsl:when test="$uriid != ''"><xsl:attribute name="uriid" select="$uriid" /></xsl:when>
        <xsl:when test="$docid != ''"><xsl:attribute name="docid" select="$docid" /></xsl:when>
        <xsl:otherwise><xsl:copy-of select="@href" /></xsl:otherwise>
      </xsl:choose>
      <!-- PSML uses lower case, but we need to capitalize for PS Standard -->
      <xsl:attribute name="type"         select="lower-case(psml:get-data(.,'type'))"/>
      <xsl:if test="not(@reverselink='false' or @data-reverselink='false')">
        <xsl:attribute name="reversetype"  select="lower-case(psml:get-data(., 'reversetype'))" />
      </xsl:if>
      <xsl:sequence select="psml:copy-data(., 'reverselink')" />
      <xsl:sequence select="psml:copy-data(., 'reversetitle')" />
      <xsl:sequence select="psml:copy-data(., 'frag')" />
      <xsl:sequence select="psml:copy-data(., 'display')" />
      <xsl:sequence select="psml:copy-data(., 'level')" />
      <xsl:sequence select="psml:copy-data(., 'config')" />
      <xsl:sequence select="psml:copy-data(., 'labels')" />
      <!-- for backward compatibility -->
      <xsl:if test="string(@labels) != ''"><xsl:copy-of select="@labels" /></xsl:if>
      <xsl:value-of select="psml:get-xref-content(psml:get-data(., 'display'), $manualtitle, psml:get-data(., 'urititle'), psml:get-data(., 'frag'), .)" />
    </blockxref>
  </xsl:if>
</xsl:template>


<!-- ========================================================================================= -->
<!-- INLINE ELEMENTS                                                                           -->
<!-- ========================================================================================= -->

<xsl:template match="b|strong">
  <bold><xsl:apply-templates /></bold>
</xsl:template>

<xsl:template match="em|i">
  <italic><xsl:apply-templates /></italic>
</xsl:template>

<xsl:template match="sup">
  <sup><xsl:apply-templates /></sup>
</xsl:template>

<xsl:template match="sub">
  <sub><xsl:apply-templates /></sub>
</xsl:template>

<xsl:template match="u">
  <underline><xsl:apply-templates /></underline>
</xsl:template>

<xsl:template match="var">
  <placeholder name="{if (@data-name) then @data-name else @name}"><xsl:value-of select="."/></placeholder>
</xsl:template>

<xsl:template match="span">
  <xsl:choose>
    <!-- XXX: Section titles?? -->
    <xsl:when test="@class='uneditable'">
      <title><xsl:apply-templates /></title>
    </xsl:when>
    <xsl:when test="@data-label or @label">
      <inline label="{if (@data-label) then @data-label else @label}">
        <xsl:apply-templates />
      </inline>
    </xsl:when>
    <xsl:when test="@style and contains(@style, 'text-decoration') and contains(@style, 'underline')">
      <underline><xsl:apply-templates /></underline>
    </xsl:when>
    <xsl:when test="@style and contains(@style, 'font-weight') and contains(@style, 'bold')">
      <bold><xsl:apply-templates /></bold>
    </xsl:when>
    <xsl:when test="@style and contains(@style, 'font-style') and contains(@style, 'italic')">
      <italic><xsl:apply-templates /></italic>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates />
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ========================================================================================= -->
<!-- HEADING ELEMENTS                                                                           -->
<!-- ========================================================================================= -->

<!--
  =================================================================
  Templates for mapping headings. The set of common parameter are:

  @param inline-content xs:boolean (tunnel)
          The current content is inside an inline tag such as
          <b> or <i>, and hence no block level elements should
          be created.
  @param level xs:string
          The level of the heading element to be created.
  =================================================================
-->
<xsl:template name="map-heading">
  <xsl:param name="inline-content" select="false()" as="xs:boolean" tunnel="yes"/>
  <xsl:param name="level" select="'1'" as="xs:string"/>
  <xsl:choose>
    <xsl:when test="$inline-content">
      <xsl:apply-templates
        select="*[not(self::img[psml:is-numbered(.)])] | text()"
        mode="heading" />
    </xsl:when>
    <xsl:otherwise>
      <heading level="{$level}">
        <xsl:choose>
          <!-- Auto-numbering -->
          <xsl:when test="psml:is-numbered(.) or img[psml:is-numbered(.)]">
            <xsl:attribute name="numbered" select="'true'" />
          </xsl:when>
          <!-- Manual prefixes (only considered when NOT auto-numbered to avoid validation errors) -->
          <xsl:when test="@data-prefix">
            <xsl:attribute name="prefix" select="@data-prefix" />
          </xsl:when>
        </xsl:choose>
        <xsl:apply-templates select="*[not(self::img[psml:is-numbered(.)])] | text()" mode="heading" />
      </heading>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="h1">
  <xsl:call-template name="map-heading">
    <xsl:with-param name="level" select="'1'" as="xs:string"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="h2">
  <xsl:call-template name="map-heading">
    <xsl:with-param name="level" select="'2'" as="xs:string"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="h3">
  <xsl:call-template name="map-heading">
    <xsl:with-param name="level" select="'3'" as="xs:string"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="h4">
  <xsl:call-template name="map-heading">
    <xsl:with-param name="level" select="'4'" as="xs:string"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="h5">
  <xsl:call-template name="map-heading">
    <xsl:with-param name="level" select="'5'" as="xs:string"/>
  </xsl:call-template>
</xsl:template>

<xsl:template match="h6">
  <xsl:call-template name="map-heading">
    <xsl:with-param name="level" select="'6'" as="xs:string"/>
  </xsl:call-template>
</xsl:template>

<!-- everything that is not allowed in a heading is ignored -->
<xsl:template match="*" mode="heading">
  <xsl:apply-templates mode="heading" />
</xsl:template>

<!-- everything that is allowed in a heading (a, br, span, bold, italic, underline, sup, sub, var and code) is carried through -->
<xsl:template match="a|br|span|b|strong|sup|sub|u|em|i|code|var" mode="heading">
  <xsl:apply-templates select="."  />
</xsl:template>

<xsl:template match="p">
  <xsl:param name="inline-content" select="false()" as="xs:boolean" tunnel="yes"/>

  <xsl:choose>
    <xsl:when test="$inline-content">
      <xsl:apply-templates />
      <xsl:if test="count(following-sibling::*) > 0"><br/></xsl:if>
    </xsl:when>
    <xsl:otherwise>
      <para>
        <xsl:variable name="indent" select="psml:get-indent(.)"/>
        <xsl:if test="$indent > 0">
          <xsl:attribute name="indent"><xsl:value-of select="$indent"/></xsl:attribute>
        </xsl:if>
        <xsl:if test="psml:is-numbered(.)">
          <xsl:attribute name="numbered" select="'true'" />
        </xsl:if>
        <!-- Manual prefixes (only considered when NOT auto-numbered to avoid validation errors) -->
        <xsl:if test="@data-prefix and not(psml:is-numbered(.))">
          <xsl:attribute name="prefix" select="@data-prefix" />
        </xsl:if>
        <xsl:apply-templates />
      </para>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="code">
  <xsl:param name="inline-content" select="false()" as="xs:boolean" tunnel="yes"/>
  <monospace><xsl:apply-templates /></monospace>
</xsl:template>

<xsl:template match="pre">
  <xsl:param name="inline-content" select="false()" as="xs:boolean" tunnel="yes"/>
  <preformat><xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if><xsl:apply-templates /></preformat>
</xsl:template>

<!-- ========================================================================================= -->
<!-- LISTS                                                                                     -->
<!-- ========================================================================================= -->

<!-- Ignore empty lists -->
<xsl:template match="ol|ul"/>

<xsl:template match="ol[li]">
  <xsl:variable name="type" select="psml:get-list-type(.)"/>
  <nlist>
    <xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if>
    <xsl:if test="$type"><xsl:attribute name="type" select="$type"/></xsl:if>
    <xsl:if test="@start"><xsl:attribute name="start"><xsl:value-of select="@start" /></xsl:attribute></xsl:if>
    <xsl:apply-templates select="li"/>
  </nlist>
</xsl:template>

<xsl:template match="ul[li]">
  <xsl:variable name="type" select="psml:get-list-type(.)"/>
  <list>
    <xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if>
    <xsl:if test="$type"><xsl:attribute name="type" select="$type"/></xsl:if>
    <xsl:apply-templates select="li"/>
  </list>
</xsl:template>

<xsl:template match="li">
  <item>
    <xsl:apply-templates />
  </item>
</xsl:template>

<!-- ========================================================================================= -->
<!-- TABLES                                                                                    -->
<!-- ========================================================================================= -->

<xsl:template match="table">
<table>
  <xsl:variable name="width"  select="psml:get-width-and-unit(.)"/>
  <xsl:variable name="height" select="if (@height) then @height else @data-height"/> <!-- ignore height from style attribute -->
  <xsl:if test="$width" ><xsl:attribute name="width"  select="$width"/></xsl:if>
  <xsl:if test="$height"><xsl:attribute name="height" select="$height"/></xsl:if>
  <xsl:copy-of select="@summary"/>
  <!-- Deprecated attributes -->
  <xsl:copy-of select="@cellspacing | @cellpadding"/>
  <xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if>
  <xsl:apply-templates select="caption"/>
  <xsl:apply-templates select="colgroup|col"/>
  <xsl:apply-templates select="thead"/>
  <xsl:apply-templates select="tbody"/>
  <xsl:apply-templates select="tfoot"/>
</table>
</xsl:template>

<xsl:template match="caption">
  <caption>
     <xsl:apply-templates />
  </caption>
</xsl:template>

<xsl:template match="tbody">
<xsl:apply-templates select="tr"/>
</xsl:template>

<xsl:template match="col">
  <col>
    <xsl:copy-of select="@span"/>
    <!-- Non HTML stored as data attributes -->
    <xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if>
    <xsl:if test="@data-part"><xsl:attribute name="part" select="@data-part"/></xsl:if>
    <xsl:if test="@data-align"><xsl:attribute name="align" select="@data-align"/></xsl:if>
    <xsl:choose>
      <xsl:when test="@width">
        <xsl:attribute name="width" select="@width"/>
      </xsl:when>
      <!-- Width stored as inline CSS property -->
      <xsl:when test="@style">
        <xsl:analyze-string regex="width:\s*([\d\.]+(px|%)?)" select="@style">
          <xsl:matching-substring><xsl:attribute name="width" select="regex-group(1)"/></xsl:matching-substring>
        </xsl:analyze-string>
      </xsl:when>
    </xsl:choose>
  </col>
</xsl:template>

<xsl:template match="td | th[ancestor::thead]">
  <cell>
    <xsl:variable name="width"  select="psml:get-width-and-unit(.)"/>
    <xsl:if test="$width" ><xsl:attribute name="width"  select="$width"/></xsl:if>
    <xsl:variable name="align"  select="psml:get-align(.)"/>
    <xsl:if test="$align" ><xsl:attribute name="align"  select="$align"/></xsl:if>
    <xsl:copy-of select="@colspan | @rowspan" />
    <!-- Deprecated attributes -->
    <xsl:copy-of select="@valign"/>
    <xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if>
    <xsl:choose>
      <xsl:when test="normalize-space(.)='&#160;'">
        <xsl:apply-templates select="*" />
      </xsl:when>
      <xsl:otherwise>
          <xsl:apply-templates />
      </xsl:otherwise>
    </xsl:choose>
  </cell>
</xsl:template>

<xsl:template match="th[not(ancestor::thead)]">
  <hcell>
    <xsl:variable name="width"  select="psml:get-width-and-unit(.)"/>
    <xsl:if test="$width" ><xsl:attribute name="width"  select="$width"/></xsl:if>
    <xsl:variable name="align"  select="psml:get-align(.)"/>
    <xsl:if test="$align" ><xsl:attribute name="align"  select="$align"/></xsl:if>
    <xsl:copy-of select="@colspan | @rowspan" />
    <!-- Deprecated attributes -->
    <xsl:copy-of select="@valign"/>
    <xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if>
    <xsl:choose>
      <xsl:when test="normalize-space(.)='&#160;'">
        <xsl:apply-templates select="*" />
      </xsl:when>
      <xsl:otherwise>
          <xsl:apply-templates />
      </xsl:otherwise>
    </xsl:choose>
  </hcell>
</xsl:template>

<xsl:template match="tr">
  <row>
    <xsl:if test="@data-align"><xsl:attribute name="align" select="@data-align"/></xsl:if>
    <xsl:if test="@data-role"><xsl:attribute name="role" select="@data-role"/></xsl:if>
    <xsl:choose>
      <xsl:when test="parent::thead"><xsl:attribute name="part" select="'header'"/></xsl:when>
      <xsl:when test="parent::tfoot"><xsl:attribute name="part" select="'footer'"/></xsl:when>
    </xsl:choose>
    <xsl:apply-templates select="td|th"/>
  </row>
</xsl:template>

<!-- ========================================================================================= -->
<!-- LINKS AND IMAGES                                                                          -->
<!-- ========================================================================================= -->

<xsl:template match="a">
  <xsl:choose>
    <xsl:when test="@class='xref' or starts-with(@class, 'xref ')">
      <xsl:if test="string-length(.) != 0">
        <xsl:variable name="manualtitle" select="psml:get-manualtitle(.)" />
        <xref>
          <xsl:if test="$manualtitle != ''"><xsl:attribute name="title" select="$manualtitle" /></xsl:if>
          <xsl:variable name="uriid"  select="psml:get-data(., 'uriid')"/>
          <xsl:variable name="docid"  select="psml:get-data(., 'docid')"/>
          <xsl:choose>
            <xsl:when test="$uriid != ''"><xsl:attribute name="uriid" select="$uriid" /></xsl:when>
            <xsl:when test="$docid != ''"><xsl:attribute name="docid" select="$docid" /></xsl:when>
            <xsl:otherwise><xsl:copy-of select="@href" /></xsl:otherwise>
          </xsl:choose>
          <!-- PSML uses lower case, but we need to capitalize for PS Standard -->
          <xsl:attribute name="type"         select="lower-case(psml:get-data(., 'type'))"/>
          <xsl:if test="not(@reverselink='false' or @data-reverselink='false')">
            <xsl:attribute name="reversetype"  select="lower-case(psml:get-data(., 'reversetype'))" />
          </xsl:if>
          <xsl:sequence select="psml:copy-data(., 'reverselink')" />
          <xsl:sequence select="psml:copy-data(., 'reversetitle')" />
          <xsl:sequence select="psml:copy-data(., 'frag')" />
          <xsl:sequence select="psml:copy-data(., 'display')" />
          <xsl:sequence select="psml:copy-data(., 'level')" />
          <xsl:sequence select="psml:copy-data(., 'config')" />
          <xsl:sequence select="psml:copy-data(., 'labels')" />
          <!-- for backward compatibility -->
          <xsl:if test="string(@labels) != ''"><xsl:copy-of select="@labels" /></xsl:if>
          <xsl:value-of select="psml:get-xref-content(psml:get-data(., 'display'), $manualtitle, psml:get-data(., 'urititle'), psml:get-data(., 'frag'), .)" />
        </xref>
      </xsl:if>
    </xsl:when>
    <xsl:when test="@href">
      <link href="{@href}">
        <xsl:sequence select="psml:copy-data(., 'uriid')" />
        <xsl:sequence select="psml:copy-data(., 'role')" />
        <xsl:sequence select="psml:copy-data(., 'labels')" />
        <xsl:apply-templates/>
      </link>
    </xsl:when>
    <xsl:when test="@name">
      <anchor name="{@name}">
        <xsl:apply-templates/>
      </anchor>
    </xsl:when>
    <xsl:when test="@id">
      <anchor name="{@id}">
        <xsl:apply-templates/>
      </anchor>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="img">
  <image>
    <xsl:variable name="uriid"  select="psml:get-data(., 'uriid')"/>
    <xsl:choose>
      <xsl:when test="$uriid != ''"><xsl:attribute name="uriid" select="$uriid" /></xsl:when>
      <xsl:otherwise><xsl:sequence select="psml:copy-data(., 'src')" /></xsl:otherwise>
    </xsl:choose>
    <xsl:sequence select="psml:copy-data(., 'labels')" />
    <!-- We need to support width and height sent using inline styles: e.g. 'style="width: 300px; height: 225px;"' -->
    <xsl:variable name="width"  select="psml:get-width(.)"/>
    <xsl:variable name="height" select="psml:get-height(.)"/>
    <xsl:if test="$width" ><xsl:attribute name="width"  select="$width"/></xsl:if>
    <xsl:if test="$height"><xsl:attribute name="height" select="$height"/></xsl:if>
    <!-- for backward compatibility -->
    <xsl:if test="@alt!=''">
      <xsl:copy-of select="@alt"/>
    </xsl:if>
  </image>
</xsl:template>

<xsl:template match="br">
  <xsl:choose>
    <!-- If the break is directly under 'body' ignore it -->
    <xsl:when test="parent::body"/>
    <!-- if the br is underneath pre, generate line feed instead of br -->
    <xsl:when test="ancestor::pre">
      <xsl:text>&#10;</xsl:text>
    </xsl:when>
    <xsl:otherwise>
      <br/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ========================================================================================= -->
<!-- MISC TEMPLATES AND FUNCTIONS                                                              -->
<!-- ========================================================================================= -->

<!-- applies templates to siblings until next top level node -->
<xsl:template name="wrap-nodes">
  <xsl:param name="node" />
  <xsl:apply-templates select="$node"/>
  <xsl:variable name="next" select="$node/following-sibling::node()[1]" />
  <xsl:if test="$next and not(psml:top-level($next))">
    <xsl:call-template name="wrap-nodes">
      <xsl:with-param name="node" select="$next" />
    </xsl:call-template>
  </xsl:if>
</xsl:template>

<!--
  Returns the name of a node if it is a valid top level element.

  @param node element()
          The element
  @return local name of the node if applicable as xs:string
-->
<xsl:function name="psml:top-level"  as="xs:boolean">
  <xsl:param name="node" />
  <xsl:sequence
    select="local-name($node)='h1' or
            local-name($node)='h2' or
            local-name($node)='h3' or
            local-name($node)='h4' or
            local-name($node)='h5' or
            local-name($node)='h6' or
            local-name($node)='p' or
            local-name($node)='ol' or
            local-name($node)='ul' or
            local-name($node)='div' or
            local-name($node)='table' or
            local-name($node)='img' or
            local-name($node)='dl' or
            local-name($node)='pre'"/>
</xsl:function>

<!--
  Get the correct xref or blockxref element content string.

  @param display     One of document|document+manual|document+fragment|manual
  @param manualtitle The manualtitle data attribute or attribute.
  @param urititle    The urititle data attribute or attribute.
  @param frag        The frag data attribute or attribute.
  @param content     The existing content of the element.

  @return the content string.
-->
<xsl:function name="psml:get-xref-content" as="xs:string">
<xsl:param name="display"        as="xs:string"/>
<xsl:param name="manualtitle"    as="xs:string"/>
<xsl:param name="urititle"       as="xs:string"/>
<xsl:param name="frag"           as="xs:string"/>
<xsl:param name="content"        as="xs:string"/>
<xsl:choose>
  <xsl:when test="$display='manual'">
    <xsl:value-of select="$manualtitle"/>
  </xsl:when>
  <xsl:when test="$display='template'">
    <xsl:value-of select="$content"/> <!-- to include exported interpublication xrefs -->
  </xsl:when>
  <xsl:when test="$urititle!=''">
    <xsl:choose>
      <xsl:when test="$display='document+fragment'">
        <xsl:value-of select="concat($urititle,': ',$frag)"/>
      </xsl:when>
      <xsl:when test="$display='document+manual' and $manualtitle != ''">
        <xsl:value-of select="concat($urititle,': ',$manualtitle)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$urititle"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:when>
  <xsl:otherwise>
    <xsl:value-of select="$content"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:function>

<!--
  Retrieve the manual xref title, falling back on the content
  if the data-title attribute does not exist and @data-display = 'manual'

  @param element The div or a element for the xref

  @return the title
-->
<xsl:function name="psml:get-manualtitle" as="xs:string">
<xsl:param name="element" as="element()"/>
<xsl:variable name="data1" select="$element/@data-title"/>
<xsl:variable name="disp" select="$element/@data-display"/>
<xsl:value-of select="if (string($data1) != '') then $data1
                      else if ($disp = 'manual') then $element
                      else ''"/>
</xsl:function>

<!--
  Retrieve the value of a data attribute, falling back on the attribute if the data attribute does not exist.

  @param element The element
  @param name    The name of the data attribute or attribute.
  @return the attribute value.
-->
<xsl:function name="psml:get-data" as="xs:string">
<xsl:param name="element" as="element()"/>
<xsl:param name="name"    as="xs:string"/>
<xsl:variable name="data" select="$element/@*[name() = concat('data-', $name)]"/>
<xsl:variable name="attr" select="$element/@*[name() = $name]"/>
<xsl:value-of select="if ($data) then $data else if ($attr) then $attr else ''"/>
</xsl:function>

<!--
  Copy the attribute and/or corresponding data attribute of the specified element.

  @param element The element node
  @param name    The name of the attribute or data attribute.
  @return The attribute node(s)
-->
<xsl:function name="psml:copy-data">
<xsl:param name="element" as="element()"/>
<xsl:param name="name"    as="xs:string"/>
<xsl:variable name="data" select="$element/@*[name() = concat('data-', $name)]"/>
<xsl:variable name="attr" select="$element/@*[name() = $name]"/>
<xsl:choose>
  <xsl:when test="$data"><xsl:attribute name="{$name}" select="$data"/></xsl:when>
  <xsl:when test="$attr"><xsl:attribute name="{$name}" select="$attr"/></xsl:when>
</xsl:choose>
</xsl:function>

<!--
  @return the indent level of a paragraph or 0 if no indent
-->
<xsl:function name="psml:get-indent" as="xs:integer">
<xsl:param name="p" as="element()"/>
<xsl:choose>
  <xsl:when test="$p/@data-indent and $p/@data-indent castable as xs:integer">
    <xsl:sequence select="$p/@data-indent"/>
  </xsl:when>
  <xsl:when test="matches($p/@class, 'indent\d')">
    <xsl:analyze-string regex="indent(\d+)" select="$p/@class">
      <xsl:matching-substring><xsl:sequence select="regex-group(1) cast as xs:integer"/></xsl:matching-substring>
    </xsl:analyze-string>
  </xsl:when>
  <xsl:when test="matches($p/@style, 'margin-left:\s*\d+px')">
    <xsl:analyze-string regex="margin-left:\s*(\d+)px" select="$p/@class">
      <xsl:matching-substring><xsl:sequence select="number(regex-group(1)) idiv 40"/></xsl:matching-substring>
    </xsl:analyze-string>
  </xsl:when>
  <xsl:otherwise><xsl:sequence select="0"/></xsl:otherwise>
</xsl:choose>
</xsl:function>

<!--
  @return the indent level of a paragraph or 0 if no indent
-->
<xsl:function name="psml:is-numbered" as="xs:boolean">
  <xsl:param name="element" as="element()"/>
  <xsl:sequence select="$element/@data-numbered = 'true'
             or matches($element/@class, 'numbered')"/>
</xsl:function>

<xsl:function name="psml:get-width" as="xs:string?">
  <xsl:param name="img" as="element(img)"/>
  <xsl:choose>
    <xsl:when test="$img/@width"><xsl:value-of select="$img/@width"/></xsl:when>
    <xsl:when test="$img/@style">
      <xsl:analyze-string regex="width:\s*(\d+)px" select="$img/@style">
        <xsl:matching-substring>
          <xsl:value-of select="regex-group(1)"/>
        </xsl:matching-substring>
      </xsl:analyze-string>
    </xsl:when>
  </xsl:choose>
</xsl:function>

<xsl:function name="psml:get-height" as="xs:string?">
  <xsl:param name="img" as="element(img)"/>
  <xsl:choose>
    <xsl:when test="$img/@height"><xsl:value-of select="$img/@height"/></xsl:when>
    <xsl:when test="$img/@style">
      <xsl:analyze-string regex="height:\s*(\d+)px" select="$img/@style">
        <xsl:matching-substring>
          <xsl:value-of select="regex-group(1)"/>
        </xsl:matching-substring>
      </xsl:analyze-string>
    </xsl:when>
  </xsl:choose>
</xsl:function>

<xsl:function name="psml:get-width-and-unit" as="xs:string?">
  <xsl:param name="el" as="element()"/>
  <xsl:choose>
    <xsl:when test="$el/@width"><xsl:value-of select="$el/@width"/></xsl:when>
    <xsl:when test="$el/@style">
      <xsl:analyze-string regex="width:\s*([\d\.]+(px|%)?)" select="$el/@style">
        <xsl:matching-substring>
          <xsl:value-of select="regex-group(1)"/>
        </xsl:matching-substring>
      </xsl:analyze-string>
    </xsl:when>
  </xsl:choose>
</xsl:function>

<xsl:function name="psml:get-height-and-unit" as="xs:string?">
  <xsl:param name="el" as="element()"/>
  <xsl:choose>
    <xsl:when test="$el/@height"><xsl:value-of select="$el/@height"/></xsl:when>
    <xsl:when test="$el/@style">
      <xsl:analyze-string regex="height:\s*(\d+(px)?)" select="$el/@style">
        <xsl:matching-substring>
          <xsl:value-of select="regex-group(1)"/>
        </xsl:matching-substring>
      </xsl:analyze-string>
    </xsl:when>
  </xsl:choose>
</xsl:function>

<xsl:function name="psml:get-align" as="xs:string?">
  <xsl:param name="el" as="element()"/>
  <xsl:choose>
    <xsl:when test="$el/@align"><xsl:value-of select="$el/@align"/></xsl:when>
    <xsl:when test="$el/@style">
      <xsl:analyze-string regex="text-align:\s*((left|center|right|justify))" select="$el/@style">
        <xsl:matching-substring>
          <xsl:value-of select="regex-group(1)"/>
        </xsl:matching-substring>
      </xsl:analyze-string>
    </xsl:when>
  </xsl:choose>
</xsl:function>

<xsl:function name="psml:get-list-type" as="xs:string?">
  <xsl:param name="list" as="element()"/>
  <xsl:choose>
    <xsl:when test="$list/@data-type"><xsl:value-of select="$list/@data-type"/></xsl:when>
    <xsl:when test="$list/@style">
      <xsl:analyze-string select="$list/@style"
                          regex="list-style-type:\s*(none|default|circle|square|decimal|lower-alpha|upper-alpha|lower-roman|upper-roman)">
        <xsl:matching-substring>
          <xsl:variable name="is-ordered" select="name($list) = 'ol'"/>
          <xsl:variable name="css-type" select="regex-group(1)"/>
          <xsl:choose>
            <xsl:when test="$css-type = 'none'">none</xsl:when>
            <!-- UL -->
            <xsl:when test="$css-type = 'default' and not($is-ordered)">disc</xsl:when>
            <xsl:when test="$css-type = 'circle'">circle</xsl:when>
            <xsl:when test="$css-type = 'square'">square</xsl:when>
            <!-- OL -->
            <xsl:when test="($css-type = 'default' or $css-type = 'decimal') and $is-ordered">arabic</xsl:when>
            <xsl:when test="$css-type = 'lower-alpha'">loweralpha</xsl:when>
            <xsl:when test="$css-type = 'upper-alpha'">upperalpha</xsl:when>
            <xsl:when test="$css-type = 'lower-roman'">lowerroman</xsl:when>
            <xsl:when test="$css-type = 'upper-roman'">upperroman</xsl:when>
          </xsl:choose>
        </xsl:matching-substring>
      </xsl:analyze-string>
    </xsl:when>
  </xsl:choose>
</xsl:function>

</xsl:transform>