/*
 * Decompiled with CFR 0.152.
 */
package org.pageseeder.berlioz.bundler;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.eclipse.jdt.annotation.Nullable;
import org.pageseeder.berlioz.bundler.BundleType;
import org.pageseeder.berlioz.bundler.CSSMin;
import org.pageseeder.berlioz.bundler.JSMin;
import org.pageseeder.berlioz.bundler.ParsingException;
import org.pageseeder.berlioz.bundler.WebBundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class WebBundleTool {
    private static final Logger LOGGER = LoggerFactory.getLogger(WebBundleTool.class);
    private static final Pattern CSS_URL = Pattern.compile("url\\(([^)]*)\\)");
    private static final Map<String, WebBundle> instances = new Hashtable<String, WebBundle>();
    private static final long DATA_URI_MAX_SIZE = 4096L;
    private final File _bundles;
    private File virtual;
    private long _dataURIThreshold = 4096L;

    public WebBundleTool(File bundles) {
        this._bundles = this.checkBundlesFile(bundles);
        this.virtual = bundles;
    }

    private File checkBundlesFile(File bundles) {
        if (!bundles.exists()) {
            throw new IllegalArgumentException("The location where bundles are saved must exist: " + bundles);
        }
        if (!bundles.isDirectory()) {
            throw new IllegalArgumentException("The location where bundles are saved must be a directory: " + bundles);
        }
        return bundles;
    }

    public void setVirtual(File virtual) {
        this.virtual = virtual;
    }

    public void setDataURIThreshold(long threshold) {
        this._dataURIThreshold = threshold;
    }

    public File getBundlesDir() {
        return this._bundles;
    }

    public @Nullable File getBundle(List<File> files, String prefix, boolean minimize) {
        if (files.isEmpty()) {
            return null;
        }
        String filename = new WebBundle(prefix, files, minimize).getFileName();
        return new File(this._bundles, filename);
    }

    public @Nullable File bundle(List<File> files, String name, boolean minimize) throws IOException {
        if (files.isEmpty()) {
            return null;
        }
        String ext = WebBundleTool.getExtension(files.get(0));
        for (BundleType t : BundleType.values()) {
            if (ext == null || !t.matches(ext)) continue;
            return this.bundle(files, name, t, minimize);
        }
        return null;
    }

    public @Nullable File bundle(List<File> files, String name, BundleType type, boolean minimize) throws IOException {
        switch (type) {
            case JS: {
                return this.bundleScripts(files, name, minimize);
            }
            case CSS: {
                return this.bundleStyles(files, name, minimize);
            }
        }
        return null;
    }

    public @Nullable File bundleScripts(List<File> files, String name, boolean minimize) throws IOException {
        if (files.isEmpty()) {
            return null;
        }
        File bundle = this.getBundle(files, name, minimize);
        if (bundle != null && !bundle.exists()) {
            LOGGER.debug("Generating bundle:{} with {} files", (Object)bundle.getName(), (Object)files.size());
            WebBundleTool.concatenate(files, bundle, minimize);
            bundle.deleteOnExit();
        }
        return bundle;
    }

    public @Nullable File bundleStyles(List<File> files, String name, boolean minimize) throws IOException {
        if (files.isEmpty()) {
            return null;
        }
        String key = WebBundle.id(files);
        WebBundle bundle = instances.get(key);
        boolean stale = false;
        if (bundle == null) {
            bundle = new WebBundle(name, files, minimize);
            stale = true;
        } else if (!bundle.isFresh()) {
            stale = true;
        }
        String filename = bundle.getFileName();
        File file = new File(this._bundles, filename);
        if (stale || !file.exists()) {
            LOGGER.debug("Generating bundle:{} with {} files", (Object)filename, (Object)files.size());
            bundle.clearImport();
            StringWriter writer = new StringWriter();
            WebBundleTool.expandStyles(bundle, writer, new File(this.virtual, file.getName()), minimize, this._dataURIThreshold);
            bundle.getETag(true);
            filename = bundle.getFileName();
            instances.put(key, bundle);
            StringReader reader = new StringReader(writer.toString());
            file = new File(this._bundles, filename);
            if (minimize && bundle.isCSSMinimizable()) {
                CSSMin.minimize((Reader)reader, (OutputStream)new FileOutputStream(file));
            } else {
                WebBundleTool.copyTo(reader, (OutputStream)new FileOutputStream(file));
            }
            file.deleteOnExit();
        }
        return file;
    }

    protected static void concatenate(List<File> files, File bundle, boolean minimize) throws IOException {
        try (FileOutputStream out = new FileOutputStream(bundle);){
            for (File f : files) {
                if (minimize && !f.getName().endsWith(".min.js")) {
                    WebBundleTool.minimizeAndCopyTo(f, out);
                    continue;
                }
                WebBundleTool.copyTo(f, (OutputStream)out);
            }
        }
    }

    protected static void expandStyles(WebBundle bundle, Writer writer, File virtual, boolean minimize, long threshold) throws IOException {
        IOException exception = null;
        ArrayList<File> processed = new ArrayList<File>();
        for (File f : bundle.files()) {
            exception = WebBundleTool.expandStylesTo(bundle, f, virtual, writer, processed, minimize, threshold);
            writer.write(10);
        }
        if (exception != null) {
            throw exception;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void minimizeAndCopyTo(File file, OutputStream out) throws IOException {
        FileInputStream input = new FileInputStream(file);
        try {
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            JSMin minimizer = new JSMin(input, buffer);
            minimizer.jsmin();
            byte[] min = buffer.toByteArray();
            out.write(min);
            out.write(10);
            out.flush();
        }
        catch (ParsingException ex) {
            LOGGER.warn("Unable to minimize {}: {}", (Object)file.getName(), (Object)ex.getMessage());
            WebBundleTool.closeQuietly(input);
            WebBundleTool.copyTo(file, out);
        }
        finally {
            WebBundleTool.closeQuietly(input);
        }
    }

    private static void copyTo(File file, OutputStream out) throws IOException {
        try (BufferedReader reader = WebBundleTool.newBufferedReader(file);){
            OutputStreamWriter writer = new OutputStreamWriter(out, StandardCharsets.UTF_8);
            String line = reader.readLine();
            while (line != null) {
                writer.append(line);
                writer.append('\n');
                line = reader.readLine();
            }
            writer.flush();
            writer = null;
        }
    }

    private static void copyTo(StringReader reader, OutputStream out) throws IOException {
        int length;
        OutputStreamWriter writer = new OutputStreamWriter(out, StandardCharsets.UTF_8);
        char[] buffer = new char[1024];
        while ((length = reader.read(buffer)) >= 0) {
            writer.write(buffer, 0, length);
        }
        writer.flush();
        writer = null;
    }

    private static @Nullable String getExtension(File file) {
        int dot = file.getName().lastIndexOf(46);
        return dot >= 0 ? file.getName().substring(dot) : null;
    }

    private static @Nullable IOException expandStylesTo(WebBundle bundle, File file, File virtual, Writer out, List<File> processed, boolean minimize, long threshold) throws IOException {
        IOException exception = null;
        if (processed.contains(file)) {
            return exception;
        }
        processed.add(file);
        try (BufferedReader reader = WebBundleTool.newBufferedReader(file);){
            String line = reader.readLine();
            while (line != null) {
                Matcher m = CSS_URL.matcher(line);
                if (m.find()) {
                    if (line.trim().toLowerCase().startsWith("@import")) {
                        String path = WebBundleTool.unquote(m.group(1));
                        if (WebBundleTool.isRelative(path)) {
                            File imported = new File(file.getParentFile(), path);
                            if (imported.exists()) {
                                if (minimize && path.endsWith("min.css")) {
                                    out.write("/*!nomin*/\n");
                                }
                                out.write("/* START import " + path + " */\n");
                                bundle.addImport(imported);
                                WebBundleTool.expandStylesTo(bundle, imported, virtual, out, processed, minimize, threshold);
                                out.write("/* END import " + path + " */\n");
                                if (minimize && path.endsWith("min.css")) {
                                    out.write("/*!min*/\n");
                                }
                            } else {
                                out.write("/* ERROR Unable to import */\n");
                                LOGGER.warn("Unable to find referenced CSS file: {}", (Object)path);
                                out.write(line);
                                out.write(10);
                            }
                        } else {
                            out.write(line);
                            out.write(10);
                        }
                    } else {
                        m.reset();
                        StringBuffer sb = new StringBuffer();
                        while (m.find()) {
                            String url = WebBundleTool.unquote(m.group(1));
                            String query = "";
                            int q = url.indexOf(63);
                            if (q > 0) {
                                query = url.substring(q);
                                url = url.substring(0, q);
                            }
                            m.appendReplacement(sb, "url(" + WebBundleTool.getLocation(file, virtual, url, threshold) + query + ")");
                        }
                        m.appendTail(sb);
                        out.write(sb.toString());
                    }
                } else {
                    out.write(line);
                }
                out.write(10);
                line = reader.readLine();
            }
        }
        catch (IOException ex) {
            exception = ex;
        }
        return exception;
    }

    protected static String getLocation(File source, File target, String path, long threshold) {
        if (!WebBundleTool.isRelative(path)) {
            return path;
        }
        StringBuilder location = new StringBuilder();
        try {
            boolean isImage;
            File ftarget = new File(source.getParentFile(), path);
            boolean bl = isImage = path.endsWith(".png") || path.endsWith(".jpg") || path.endsWith(".gif");
            if (isImage && ftarget.exists() && ftarget.length() < threshold) {
                location.append("data:image/").append(path.substring(path.lastIndexOf(46) + 1)).append(";base64,");
                location.append(WebBundleTool.encodeBase64(ftarget));
            } else {
                String csource = ftarget.getCanonicalPath();
                String ctarget = target.getCanonicalPath();
                int x = WebBundleTool.common(csource, ctarget);
                String rbundle = ctarget.substring(x);
                for (int i = 0; i < rbundle.length(); ++i) {
                    if (rbundle.charAt(i) != File.separatorChar) continue;
                    location.append("../");
                }
                location.append(csource.substring(x).replace('\\', '/'));
            }
        }
        catch (IOException ex) {
            LOGGER.warn("Error while calculating location", (Throwable)ex);
        }
        return location.toString();
    }

    private static int common(String a, String b) {
        int i;
        for (i = 0; i < a.length() && i < b.length() && a.charAt(i) == b.charAt(i); ++i) {
        }
        return i;
    }

    private static String unquote(String url) {
        if (url.length() < 2) {
            return url;
        }
        char first = url.charAt(0);
        char last = url.charAt(url.length() - 1);
        if (first == '\'' && last == '\'' || first == '\"' && last == '\"') {
            return url.substring(1, url.length() - 1);
        }
        return url;
    }

    private static boolean isRelative(String url) {
        return !url.startsWith("https://") && !url.startsWith("http://") && !url.startsWith("data:") && !url.startsWith("/") && !url.startsWith("<");
    }

    private static void closeQuietly(@Nullable Closeable closeable) {
        if (closeable == null) {
            return;
        }
        try {
            closeable.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static BufferedReader newBufferedReader(File f) throws FileNotFoundException {
        return new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(f), StandardCharsets.UTF_8));
    }

    private static String encodeBase64(File file) throws IOException {
        Base64.Encoder encoder = Base64.getEncoder();
        byte[] bytes = Files.readAllBytes(file.toPath());
        return encoder.encodeToString(bytes);
    }
}

