/*
 * Decompiled with CFR 0.152.
 */
package com.pageseeder.common.security;

import com.pageseeder.common.security.IntegrityException;
import com.pageseeder.common.util.Strings;
import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.util.Base64;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

public final class SafeStorage {
    private static final String KEYGEN_ALGORITHM = "PBKDF2WithHmacSHA1";
    private static final char DELIM = '.';
    private static final String DELIM_SPLIT_REGEX = "\\.";
    private static final String MAC_PREFIX = "S/1";
    private static final SecureRandom R = new SecureRandom();
    private static final int SALT_SIZE_BITS = 128;

    private SafeStorage() {
    }

    public static String encrypt(String data, char[] password) throws SecurityException {
        Algorithm encryption = Algorithm.AES_128_CBC;
        byte[] encryptionSalt = SafeStorage.randomBytes(128);
        byte[] encryptionIV = SafeStorage.randomBytes(encryption.ivBits);
        byte[] encryptedData = SafeStorage.doEncrypt(data.getBytes(StandardCharsets.UTF_8), encryption, password, encryptionSalt, encryptionIV);
        String hmacBaseString = SafeStorage.toHMACBaseString(encryptedData, encryptionIV, encryptionSalt);
        Algorithm integrity = Algorithm.SHA_256;
        byte[] integritySalt = SafeStorage.randomBytes(128);
        byte[] integrityHmac = SafeStorage.hmac(password, hmacBaseString, integritySalt, integrity);
        return SafeStorage.assemble(hmacBaseString, integritySalt, integrityHmac);
    }

    public static String decrypt(String stored, char[] password) throws SecurityException, IntegrityException {
        String[] parts = stored.split(DELIM_SPLIT_REGEX);
        if (parts.length != 6) {
            throw new IntegrityException(stored, "Unable to parse data: " + parts.length + " fields found");
        }
        String macPrefix = parts[0];
        String encryptionSaltString = parts[1];
        String encryptionIvBase64 = parts[2];
        String encryptedDataB64 = parts[3];
        String integrityHmacSaltB64 = parts[4];
        String integrityHmacB64 = parts[5];
        if (!macPrefix.equals(MAC_PREFIX)) {
            throw new IntegrityException(stored, "Data uses prefix " + macPrefix + " but this version requires S/1");
        }
        String hmacBaseString = macPrefix + "." + encryptionSaltString + "." + encryptionIvBase64 + "." + encryptedDataB64;
        Algorithm integrity = Algorithm.SHA_256;
        Base64.Decoder decoder = Base64.getDecoder();
        byte[] integrityByteSalt = decoder.decode(integrityHmacSaltB64);
        byte[] checkIntegrityHmac = SafeStorage.hmac(password, hmacBaseString, integrityByteSalt, integrity);
        String checkIntegrityHmacB64 = Base64.getEncoder().encodeToString(checkIntegrityHmac);
        if (!Strings.fixedTimeEquals(checkIntegrityHmacB64, integrityHmacB64)) {
            throw new IntegrityException(stored, "Invalid HMAC");
        }
        byte[] encryptedData = decoder.decode(encryptedDataB64);
        byte[] encryptionIv = decoder.decode(encryptionIvBase64);
        byte[] encryptionSalt = decoder.decode(encryptionSaltString);
        Algorithm encryption = Algorithm.AES_128_CBC;
        byte[] deencrypted = SafeStorage.doDecrypt(encryptedData, encryption, password, encryptionSalt, encryptionIv);
        return new String(deencrypted, StandardCharsets.UTF_8);
    }

    private static byte[] hmac(char[] password, String baseString, byte[] salt, Algorithm algorithm) throws SecurityException {
        Mac mac;
        SecretKey secretKey = SafeStorage.generateKey(password, salt, algorithm);
        try {
            mac = Mac.getInstance(algorithm.transformation);
            mac.init(secretKey);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new SecurityException("Unknown algorithm " + algorithm.transformation, ex);
        }
        catch (InvalidKeyException ex) {
            throw new SecurityException("Key " + String.valueOf(secretKey) + " is not valid", ex);
        }
        return mac.doFinal(baseString.getBytes(StandardCharsets.UTF_8));
    }

    private static byte[] doEncrypt(byte[] data, Algorithm algorithm, char[] password, byte[] salt, byte[] iv) throws SecurityException {
        SecretKey key = SafeStorage.generateKey(password, salt, algorithm);
        try {
            Cipher cipher = Cipher.getInstance(algorithm.transformation);
            cipher.init(1, (Key)key, new IvParameterSpec(iv));
            return cipher.doFinal(data);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new SecurityException("Encryption algorithm " + algorithm.transformation + " not found", ex);
        }
        catch (NoSuchPaddingException ex) {
            throw new SecurityException("Cannot work with padding given by " + algorithm.transformation, ex);
        }
        catch (InvalidKeyException ex) {
            throw new SecurityException("Key " + String.valueOf(key) + " is invalid", ex);
        }
        catch (InvalidAlgorithmParameterException ex) {
            throw new SecurityException("Initialization vector passed to cipher initialization", ex);
        }
        catch (IllegalBlockSizeException ex) {
            throw new SecurityException("Illegal block size when decrypting", ex);
        }
        catch (BadPaddingException ex) {
            throw new SecurityException("Bad padding when decrypting", ex);
        }
    }

    private static byte[] doDecrypt(byte[] encryptedData, Algorithm algorithm, char[] password, byte[] salt, byte[] iv) throws SecurityException {
        SecretKey secretKey = SafeStorage.generateKey(password, salt, algorithm);
        try {
            Cipher cipher = Cipher.getInstance(algorithm.transformation);
            cipher.init(2, (Key)secretKey, new IvParameterSpec(iv));
            return cipher.doFinal(encryptedData);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new SecurityException("Encryption algorithm " + algorithm.transformation + " not found", ex);
        }
        catch (NoSuchPaddingException ex) {
            throw new SecurityException("Cannot work with padding given by " + algorithm.transformation, ex);
        }
        catch (InvalidKeyException ex) {
            throw new SecurityException("Key " + String.valueOf(secretKey) + " is invalid", ex);
        }
        catch (InvalidAlgorithmParameterException ex) {
            throw new SecurityException("Initialization vector passed to cipher initialization seems to be invalid algorithm parameter", ex);
        }
        catch (IllegalBlockSizeException ex) {
            throw new SecurityException("Illegal block size when decrypting", ex);
        }
        catch (BadPaddingException ex) {
            throw new SecurityException("Bad padding when decrypting", ex);
        }
    }

    private static SecretKey generateKey(char[] password, byte[] salt, Algorithm algorithm) throws SecurityException {
        try {
            SecretKeyFactory keyFactory = SecretKeyFactory.getInstance(KEYGEN_ALGORITHM);
            PBEKeySpec keySpec = new PBEKeySpec(password, salt, 1, algorithm.keyBits);
            SecretKey key = keyFactory.generateSecret(keySpec);
            return new SecretKeySpec(key.getEncoded(), "AES");
        }
        catch (NoSuchAlgorithmException ex) {
            throw new SecurityException("Algorithm PBKDF2WithHmacSHA1 not found by SecretKeyFactory", ex);
        }
        catch (InvalidKeySpecException ex) {
            throw new SecurityException("KeySpec is invalid", ex);
        }
    }

    private static String toHMACBaseString(byte[] encryptedData, byte[] encryptionIV, byte[] encryptionSalt) {
        Base64.Encoder encoder = Base64.getEncoder();
        return "S/1." + encoder.encodeToString(encryptionSalt) + "." + encoder.encodeToString(encryptionIV) + "." + encoder.encodeToString(encryptedData);
    }

    private static String assemble(String hmacBaseString, byte[] integritySalt, byte[] integrityHmac) {
        Base64.Encoder encoder = Base64.getEncoder();
        String integritySaltString = encoder.encodeToString(integritySalt);
        String integrityHmacB64Url = encoder.encodeToString(integrityHmac);
        return hmacBaseString + "." + integritySaltString + "." + integrityHmacB64Url;
    }

    private static byte[] randomBytes(int nbits) {
        int nbytes = (int)Math.ceil((double)nbits / 8.0);
        byte[] iv = new byte[nbytes];
        R.nextBytes(iv);
        return iv;
    }

    public static enum Algorithm {
        AES_128_CBC("AES/CBC/PKCS5PADDING", 128, 128),
        SHA_256("HmacSHA256", 256, 0);

        private final String transformation;
        private final int keyBits;
        private final int ivBits;

        private Algorithm(String transformation, int keyBits, int ivBits) {
            this.transformation = transformation;
            this.keyBits = keyBits;
            this.ivBits = ivBits;
        }
    }
}

