<!--
  ~ Copyright (c) 1999-2019 Allette Systems Pty Ltd
  -->
<xsl:stylesheet version="3.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:json="http://www.w3.org/2005/xpath-functions"
                xmlns:f="http://www.pageseeder.com/function"
                xmlns:psml="http://pageseeder.com/PSML"
                exclude-result-prefixes="#all">

<xsl:template match="root[@service='api-help']" mode="json">
  <xsl:variable name="path" select="//help-file/@path"/>
  <json:map>
    <json:string key="path"><xsl:value-of select="$path"/></json:string>
    <json:string key="title"><xsl:value-of select="descendant::section[@id='title' or ends-with(@id, '-title')]/descendant::heading"/></json:string>
    <json:string key="page"><xsl:value-of select="descendant::property[@name='page_id']/@value"/></json:string>
    <json:string key="type"><xsl:value-of select="descendant::property[@name='document_type']/@value"/></json:string>
    <json:boolean key="isCustom">false</json:boolean>
    <json:string key="date"><xsl:value-of select="//help-file/@date"/></json:string>
    <json:string key="content">
      <xsl:variable name="html">
        <xsl:apply-templates select="descendant::section[not(@id='title' or ends-with(@id, '-title'))]/*" mode="psml-help"/>

        <!-- Notice if the document status is still `In Progress` -->
        <xsl:variable name="status" select="descendant::document/@status"/>
        <xsl:choose>
          <xsl:when test="$status = 'Initiated'">
            <callout-box type="tip" icon="draft">Sorry, it looks like this help article isn't ready yet.</callout-box>
          </xsl:when>
          <xsl:when test="$status = 'In Progress'">
            <callout-box type="tip" icon="draft">This help article is pending review by our editorial team.</callout-box>
          </xsl:when>
        </xsl:choose>

        <!-- Related help -->
        <xsl:call-template name="help-related" />

      </xsl:variable>
      <xsl:value-of select="serialize($html)"/>
    </json:string>
  </json:map>
</xsl:template>

<xsl:template match="root[@service='api-custom-help']" mode="json">
  <xsl:variable name="path" select="//documentinfo/uri/@path"/>
  <json:map>
    <json:string key="path"><xsl:value-of select="$path"/></json:string>
    <json:string key="title"><xsl:value-of select="descendant::section[@id='title' or ends-with(@id, '-title')]/descendant::heading"/></json:string>
    <json:string key="page"><xsl:value-of select="descendant::property[@name='page_id']/@value"/></json:string>
    <json:string key="type"><xsl:value-of select="descendant::property[@name='document_type']/@value"/></json:string>
    <json:boolean key="isCustom">true</json:boolean>
    <json:string key="date"><xsl:value-of select="content/document/@date"/></json:string>
    <json:string key="content">
      <xsl:variable name="html">
        <xsl:apply-templates select="descendant::section[not(@id='title' or ends-with(@id, '-title'))]/*" mode="psml-help">
          <xsl:with-param name="id-prefix" select="'help-'" tunnel="yes"/>
        </xsl:apply-templates>
        <xsl:call-template name="help-related" />
      </xsl:variable>
      <xsl:value-of select="serialize($html)"/>
    </json:string>
  </json:map>
</xsl:template>

<xsl:template match="fragment" mode="psml-help">
  <xsl:param name="id-prefix" tunnel="yes"/>
  <a id="help-{@id}" />
  <xsl:apply-templates mode="psml"/>
</xsl:template>

<xsl:template match="xref-fragment" mode="psml-help">
  <xsl:param name="id-prefix" tunnel="yes"/>
  <a id="help-{@id}" />
  <xsl:choose>
    <xsl:when test="f:should-alphabetize(.)">
      <xsl:for-each-group select="blockxref" group-by="upper-case(substring(@urititle, 1, 1))">
        <xsl:sort select="@urititle"/>
        <h2><xsl:value-of select="current-grouping-key()"/></h2>
        <ul class="help_xref-fragment">
          <xsl:apply-templates select="current-group()" mode="psml"/>
        </ul>
      </xsl:for-each-group>
    </xsl:when>
    <xsl:otherwise>
      <ul class="help_xref-fragment">
        <xsl:apply-templates mode="psml"/>
      </ul>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="media-fragment" mode="psml-help">
  <a id="help-{@id}" />
  <xsl:variable name="content" select="psml:serialize-media-content(*|text(), @mediatype)"/>
  <psml-media mediatype="{@mediatype}" content="{$content}"/>
</xsl:template>

<xsl:template match="properties-fragment" mode="psml-help">
  <a id="help-{@id}" />
  <callout-box type="warning" message="Properties fragments not supported in the built-in help."/>
</xsl:template>

<!-- Note, Tips and warning turned into a message -->
<xsl:template match="root[@service='api-help' or @service='api-custom-help']//block[@label='note' or @label='tip' or @label='warning']" mode="psml" priority="2">
  <callout-box type="{if (@label = 'note') then 'info' else @label}">
    <xsl:apply-templates mode="#current"/>
  </callout-box>
</xsl:template>

<xsl:template match="root[@service='api-help' or @service='api-custom-help']//block[@label='todo']" mode="psml" priority="2">
  <!-- Ignore content -->
</xsl:template>

<xsl:template match="root[@service='api-help' or @service='api-custom-help']//block[@label='keypoint']" mode="psml" priority="2">
  <div class="help_wideblock is-{@label}">
    <xsl:sequence select="f:help-block-prefix(@label, true(), true())"/>
    <xsl:choose>
      <xsl:when test="para|heading|block|preformat">
        <xsl:apply-templates select="*/*|*/text()" mode="psml" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates mode="psml" />
      </xsl:otherwise>
    </xsl:choose>
  </div>
</xsl:template>

<xsl:template match="root[@service='api-help' or @service='api-custom-help']//block[@label='developer' or @label='manager' or @label='administrator']" mode="psml" priority="2">
  <div class="help_wideblock is-{@label}">
    <xsl:variable name="first-element" select="*[1]"/>
    <xsl:choose>
      <!-- Single paragraph or just text: insert prefix inline -->
      <xsl:when test="(para and count(*) = 1)
                   or (text() and not(para|heading|preformat|block|list|nlist|table))">
        <p>
          <xsl:sequence select="f:help-block-prefix(@label, true(), true())"/>
          <xsl:apply-templates select="$first-element/text()|$first-element/*" mode="psml" />
        </p>
      </xsl:when>
      <!-- Start with a heading: insert icon -->
      <xsl:when test="heading[1] = $first-element">
        <xsl:element name="h{$first-element/@level}">
          <xsl:sequence select="f:help-block-prefix(@label, false(), false())"/>
          <xsl:text> </xsl:text>
          <xsl:apply-templates select="$first-element/text()|$first-element/*" mode="psml" />
        </xsl:element>
        <xsl:apply-templates select="*[position() gt 1]" mode="psml"/>
      </xsl:when>
      <!-- Otherwise: add paragraph before -->
      <xsl:otherwise>
        <p><xsl:sequence select="f:help-block-prefix(@label, true(), false())"/></p>
        <xsl:apply-templates mode="psml"/>
      </xsl:otherwise>
    </xsl:choose>
  </div>
</xsl:template>


<xsl:template match="root[@service='api-help' or @service='api-custom-help']//block[@label='location']" mode="psml" priority="2">
  <xsl:variable name="path" select="ancestor::help-file/@path" as="xs:string?"/>
  <div class="help_wideblock is-{@label}">
    <h3 class="help_location_title">How to find this<xsl:choose>
      <xsl:when test="contains($path, '/page/')"> page</xsl:when>
      <xsl:when test="contains($path, '/dialog/')"> dialog</xsl:when>
      <xsl:when test="contains($path, '/panel/')"> panel</xsl:when>
    </xsl:choose></h3>
    <xsl:apply-templates mode="psml"/>
  </div>
</xsl:template>

<xsl:function name="f:help-block-prefix">
  <xsl:param name="label"/>
  <xsl:param name="with-text"/>
  <xsl:param name="with-colon"/>
  <b class="help_wideblock-prefix" title="{$label}">
    <xsl:choose>
      <xsl:when test="$label = 'developer'"><svg-icon name="developer"/></xsl:when>
      <xsl:when test="$label = 'manager'"><svg-icon name="wrench"/></xsl:when>
      <xsl:when test="$label = 'administrator'"><svg-icon name="tools"/></xsl:when>
    </xsl:choose>
    <xsl:if test="$with-text">
      <xsl:choose>
        <xsl:when test="$label = 'keypoint'"> Key point</xsl:when>
        <xsl:when test="$label = 'developer'"> For developers</xsl:when>
        <xsl:when test="$label = 'manager'"> For managers</xsl:when>
        <xsl:when test="$label = 'administrator'"> For administrators</xsl:when>
      </xsl:choose>
      <xsl:if test="$with-colon">: </xsl:if>
    </xsl:if>
  </b>
</xsl:function>

<xsl:template match="root[@service='api-help' or @service='api-custom-help']//inline[@label='svg-icon']" mode="psml" priority="2">
  <xsl:variable name="icon" select="tokenize(normalize-space(text()[1]), ' ')"/>
  <svg-icon name="{$icon[1]}">
    <xsl:if test="$icon[2]">
      <xsl:attribute name="color" select="$icon[2]"/>
    </xsl:if>
  </svg-icon>
</xsl:template>

<xsl:template match="root[@service='api-help' or @service='api-custom-help']//xref" mode="psml" priority="2">
  <xsl:choose>
    <xsl:when test="@type = 'math'">
      <xsl:apply-templates mode="#current"/>
    </xsl:when>
    <xsl:when test="matches(@href, '.psml(#.*)?$') or matches(@href, '^#.*$')">
      <xsl:apply-templates select="." mode="psml-help"/>
    </xsl:when>
    <xsl:when test="@id">
      <a href="{f:help-image-src(ancestor::help-file/@path, @href)}"
         download="{$site-prefix}/uri/{@id}?behavior=download"
      ><xsl:apply-templates mode="#current"/>&#xa0;<svg-icon name="download" />
      </a>
    </xsl:when>
    <xsl:otherwise>
      <a href="{f:help-image-src(ancestor::help-file/@path, @href)}"
         download="{tokenize(@href, '/')[last()]}"
      ><xsl:apply-templates mode="#current"/>&#xa0;<svg-icon name="download" />
      </a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="root[@service='api-help' or @service='api-custom-help']//blockxref[@type='embed']" mode="psml" priority="2">
  <xsl:choose>
    <xsl:when test="matches(@href, '.psml$')">
      <li class="help_blockxref"><xsl:apply-templates select="." mode="psml-help"/></li>
    </xsl:when>
    <xsl:otherwise>
      <xsl:apply-templates mode="#current"/>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:template match="help-file//para[image][normalize-space(.) = '']" mode="psml" priority="2">
  <xsl:for-each select="image">
    <figure class="{for $label in tokenize(@labels, ',') return concat('label-', $label)}">
      <img src="{f:help-image-src(ancestor::help-file/@path, @src)}" alt="{@alt}">
        <xsl:copy-of select="@width|@height"/>
      </img>
      <xsl:if test="not(ends-with(@src, @alt))">
        <figcaption><xsl:value-of select="@alt"/></figcaption>
      </xsl:if>
    </figure>
  </xsl:for-each>
</xsl:template>

<xsl:template match="help-file//image" mode="psml" priority="2">
  <span class="{for $label in tokenize(@labels, ',') return concat('label-', $label)}">
    <img src="{f:help-image-src(ancestor::help-file/@path, @src)}" alt="{@alt}">
      <xsl:copy-of select="@width|@height"/>
    </img>
    <xsl:if test="not(ends-with(@src, @alt))">
      <span><xsl:value-of select="@alt"/></span>
    </xsl:if>
  </span>
</xsl:template>

<xsl:template match="xref|blockxref" mode="psml-help">
  <xsl:param name="id-prefix" tunnel="yes"/>
  <xsl:variable name="help-file" select="ancestor::help-file"/>
  <xsl:variable name="group" select="//uri-parameters/parameter[@name='group']"/>
  <xsl:variable name="ref" select="
  if ($help-file)
    then f:compute-help-path($help-file/@path, @href)
    else concat($group, ':', @uriid)
  "/>
  <xsl:variable name="is-self" select="
  if ($help-file)
    then starts-with(@href, '#') or ends-with(concat($help-file/@path, '.psml'), concat('/', @href))
    else @uriid = ancestor::document/@id
  "/>
  <xsl:choose>
    <xsl:when test="$is-self">
      <a href="#help-{if (starts-with(@href, '#')) then substring-after(@href, '#') else @frag}" class="xref">
        <xsl:apply-templates mode="psml"/>
      </a>
    </xsl:when>
    <xsl:otherwise>
      <xsl:variable name="options">{component:'help-panel', idref:'<xsl:value-of select="$ref"/>'}</xsl:variable>
      <a v-panel="{$options}" class="xref">
        <xsl:apply-templates mode="psml"/>
      </a>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<xsl:function name="f:compute-help-path" as="xs:string">
  <xsl:param name="source"/>
  <xsl:param name="target"/>
  <!-- We need a dummy host to use `resolve-uri` -->
  <xsl:variable name="base" select="concat('https://localhost', $source)"/>
  <xsl:variable name="resolved" select="substring-after(resolve-uri($target, $base), 'https://localhost')"/>
  <xsl:value-of select="replace($resolved, '.psml$', '')"/>
</xsl:function>

<xsl:function name="f:help-image-src" as="xs:string">
  <xsl:param name="source"/>
  <xsl:param name="target"/>
  <!-- We need a dummy host to use `resolve-uri` -->
  <xsl:variable name="base" select="concat('https://localhost', $source)"/>
  <xsl:variable name="resolved" select="substring-after(resolve-uri($target, $base), 'https://localhost')"/>
  <xsl:value-of select="concat($site-prefix, '/weborganic/help', $resolved)"/>
</xsl:function>

<xsl:function name="f:should-alphabetize" as="xs:boolean">
  <xsl:param name="xref-fragment"/>
  <xsl:variable name="xrefs" select="$xref-fragment/blockxref" />
  <xsl:variable name="initials" select="distinct-values(for $xref in $xrefs return upper-case(substring($xref/@urititle, 1, 1)))" />
  <xsl:sequence select="
       count($xrefs) gt 10
   and count($initials) gt 4
   and count($initials) lt (count($xrefs) idiv 2)"/>
</xsl:function>

<xsl:template name="help-related">
  <xsl:variable name="help-file" select="//help-file"/>
  <xsl:variable name="group" select="//uri-parameters/parameter[@name='group']"/>
  <xsl:if test="descendant::xref or descendant::reversexref">
    <aside class="help_related">
      <h3 class="help_related_title">Related</h3>
      <ul>
        <xsl:for-each-group select="descendant::xref|descendant::reversexref" group-by="@urititle">
          <xsl:sort select="lower-case(@urititle)"/>
          <li>
            <xsl:variable name="ref" select="if ($help-file) then f:compute-help-path($help-file/@path, @href) else concat($group, ':', @uriid)"/>
            <xsl:variable name="options">{component:'help-panel', idref:'<xsl:value-of select="$ref"/>'}</xsl:variable>
            <a v-panel="{$options}" class="xref"><xsl:value-of select="@urititle"/></a>
          </li>
        </xsl:for-each-group>
      </ul>
    </aside>
  </xsl:if>
</xsl:template>

</xsl:stylesheet>
