<!--
  ~ Copyright (c) 1999-2018 Allette Systems Pty Ltd

  Functions to convert some objects to JSON
-->
<xsl:stylesheet version="3.0"
                xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                xmlns:f="http://www.pageseeder.com/function"
                xmlns:err="http://www.w3.org/2005/xqt-errors"
                xmlns:json="http://www.w3.org/2005/xpath-functions"
                exclude-result-prefixes="#all">

<!--
  Converts JSON-XML to a JSON String
-->
<xsl:function name="f:xml-to-json" as="xs:string">
  <xsl:param name="json-as-xml" as="node()?"/>
  <xsl:try select="json:xml-to-json($json-as-xml)">
    <xsl:catch errors="*">
      <xsl:message>Warning: JSON String failed!
        Error code: <xsl:value-of select="$err:code"/>
        Reason: <xsl:value-of select="$err:description"/>
      </xsl:message>
      <xsl:sequence select="'null'"/>
    </xsl:catch>
  </xsl:try>
</xsl:function>

<!--
  Converts a single item to JSON

  This function will invoke the template on the template in mode "json" for elements,
  recognize numbers and default to string for other types of items

  @param The item to convert to JSON
  @returns The corresponding JSON or null as a string
-->
<xsl:function name="f:to-json" as="xs:string">
  <xsl:param name="item" as="item()?"/>
  <xsl:variable name="json-as-xml" as="element()?">
    <xsl:choose>
      <xsl:when test="$item instance of element()">
        <xsl:apply-templates select="$item" mode="json"/>
      </xsl:when>
      <xsl:when test="$item instance of xs:float or $item instance of xs:decimal or $item instance of xs:double">
        <json:number><xsl:sequence select="$item"/></json:number>
      </xsl:when>
      <xsl:when test="$item instance of xs:boolean">
        <json:boolean><xsl:sequence select="$item"/></json:boolean>
      </xsl:when>
      <xsl:when test="count($item) = 0">
        <json:null/>
      </xsl:when>
      <xsl:otherwise>
        <json:string><xsl:value-of select="$item"/></json:string>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:value-of select="json:xml-to-json($json-as-xml)"/>
</xsl:function>

<!--
  Converts a sequence of items to a JSON array

  This function will invoke the template on the template in mode "json" for elements,
  recognize numbers and default to string for other types of items

  @param The sequence of items to convert to JSON
  @returns The corresponding JSON array as a string
-->
<xsl:function name="f:to-json-array" as="xs:string">
  <xsl:param name="item" as="item()*"/>
  <xsl:variable name="json-as-xml" as="element()">
    <json:array>
      <xsl:for-each select="$item">
        <xsl:choose>
          <xsl:when test=". instance of element()">
            <xsl:apply-templates select="." mode="json"/>
          </xsl:when>
          <xsl:when test=". instance of xs:integer or $item instance of xs:decimal or $item instance of xs:double">
            <json:number><xsl:sequence select="."/></json:number>
          </xsl:when>
          <xsl:when test=". instance of xs:boolean">
            <json:boolean><xsl:sequence select="."/></json:boolean>
          </xsl:when>
          <xsl:otherwise>
            <json:string><xsl:value-of select="."/></json:string>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:for-each>
    </json:array>
  </xsl:variable>
  <xsl:value-of select="json:xml-to-json($json-as-xml)"/>
</xsl:function>

<xsl:function name="f:to-json-group-properties" as="xs:string">
  <xsl:param name="group-properties" as="element(properties)"/>
  <xsl:variable name="config" as="element(property-config)*">
    <property-config name="disableIndexing"              type="boolean" grouping="index"/>
    <property-config name="documentStatusApprover"       type="array"   grouping="task_workflow"/>
    <property-config name="documentStatusIcon"           type="array"   grouping="task_workflow"/>
    <property-config name="documentStatusContributor"    type="array"   grouping="task_workflow"/>
    <property-config name="documentStatusLocked"         type="array"   grouping="task_workflow"/>
    <property-config name="documentStatusManager"        type="array"   grouping="task_workflow"/>
    <property-config name="documentStatusReviewer"       type="array"   grouping="task_workflow"/>
    <property-config name="emailFooterStart"             type="string"  grouping="notification"/>
    <property-config name="ignoreEmbeddedAttachmentSize" type="string"  grouping="notification"/>
    <property-config name="indexIgnoreFolders"           type="string"  grouping="index"/>
    <property-config name="indexVersions"                type="boolean" grouping="index"/>
    <property-config name="indexXFields"                 type="boolean" grouping="index"/>
    <property-config name="priority"                     type="string"  grouping="task_workflow"/>
    <property-config name="statusActive"                 type="array"   grouping="task_workflow"/>
    <property-config name="statusApprover"               type="array"   grouping="task_workflow"/>
    <property-config name="statusContributor"            type="array"   grouping="task_workflow"/>
    <property-config name="statusManager"                type="array"   grouping="task_workflow"/>
    <property-config name="statusReviewer"               type="array"   grouping="task_workflow"/>
    <property-config name="statusBroadcastApprovers"     type="array"   grouping="task_workflow"/>
    <property-config name="statusIcon"                   type="array"   grouping="task_workflow"/>
    <property-config name="mailNewThreadByTitle"         type="boolean" grouping="notification"/>
    <property-config name="mailRejectedAddress"          type="string"  grouping="notification"/>
    <property-config name="mailReturnedAddress"          type="string"  grouping="notification"/>
    <property-config name="notifyFromAddress"            type="string"  grouping="notification"/>
    <property-config name="reminderNotification"         type="boolean" grouping="notification"/>
    <property-config name="stripHtmlEmailFooter"         type="boolean" grouping="notification" />
    <property-config name="errorReportAddress"           type="string"  grouping="notification" />
    <property-config name="acceptInvitationRequired"     type="boolean" grouping="other"/>
  </xsl:variable>
  <xsl:variable name="object">
    <json:map>
      <xsl:for-each select="$group-properties/property">
        <xsl:sort select="@name" />
        <xsl:variable name="name" select="@name"/>
        <json:map key="{@name}">
          <json:boolean key="useDefault"><xsl:value-of select="@source != 'local'"/></json:boolean>
          <json:boolean key="sourceProject"><xsl:value-of select="@source = 'project'"/></json:boolean>
          <xsl:choose>
            <xsl:when test="$config[@name=$name]/@type = 'boolean'">
              <json:boolean key="value"><xsl:value-of select="if (@value = 'true') then 'true' else 'false'"/></json:boolean>
              <json:boolean key="inherit"><xsl:value-of select="if (@inherit = 'true') then 'true' else 'false'"/></json:boolean>
            </xsl:when>
            <xsl:when test="$config[@name=$name]/@type = 'array'">
              <json:array key="value">
                <xsl:for-each select="tokenize(@value, ',')"><json:string><xsl:value-of select="."/></json:string></xsl:for-each>
              </json:array>
              <json:array key="inherit">
                <xsl:for-each select="tokenize(@inherit, ',')"><json:string><xsl:value-of select="."/></json:string></xsl:for-each>
              </json:array>
            </xsl:when>
            <xsl:otherwise>
              <json:string key="value"><xsl:value-of select="@value"/></json:string>
              <json:string key="inherit"><xsl:value-of select="@inherit"/></json:string>
            </xsl:otherwise>
          </xsl:choose>
          <json:string key="grouping"><xsl:value-of select="$config[@name=$name]/@grouping"/></json:string>
        </json:map>
      </xsl:for-each>
    </json:map>
  </xsl:variable>
  <xsl:value-of select="json:xml-to-json($object)"/>
</xsl:function>

<xsl:function name="f:to-json-filetree" as="xs:string">
  <xsl:param name="file" as="element()"/>
  <xsl:variable name="object">
    <xsl:apply-templates select="$file" mode="json-filetree"/>
  </xsl:variable>
  <xsl:value-of select="json:xml-to-json($object)"/>
</xsl:function>

<xsl:template match="file" mode="json-filetree">
<json:map>
  <json:string key="name"><xsl:value-of select="@name"/></json:string>
  <json:string key="type"><xsl:value-of select="if (@type) then @type else 'folder'"/></json:string>
  <xsl:if test="@path">
    <json:string key="path"><xsl:value-of select="@path"/></json:string>
  </xsl:if>
  <xsl:if test="parent::content">
    <json:boolean key="root">true</json:boolean>
  </xsl:if>
  <xsl:choose>
    <xsl:when test="@type = 'folder' or not(@length)">
      <json:array key="children">
        <xsl:apply-templates select="file" mode="json-filetree">
          <xsl:sort select="if (@type = 'folder') then 1 else 2"/>
          <xsl:sort select="lower-case(@name)"/>
        </xsl:apply-templates>
      </json:array>
    </xsl:when>
    <xsl:otherwise>
      <json:number key="length"><xsl:value-of select="@length"/></json:number>
      <json:string key="mediatype"><xsl:value-of select="@mediatype"/></json:string>
      <json:string key="modified"><xsl:value-of select="@modified"/></json:string>
      <json:boolean key="text"><xsl:value-of select="@text = 'true'"/></json:boolean>
    </xsl:otherwise>
  </xsl:choose>
</json:map>
</xsl:template>

<xsl:template match="text()" mode="json">
  <json:string><xsl:value-of select="."/></json:string>
</xsl:template>

<xsl:template match="session/permissions | session/group-permissions" mode="json">
  <xsl:param name="key" />
  <json:map>
    <xsl:if test="$key"><xsl:attribute name="key" select="$key" /></xsl:if>
    <xsl:for-each select="@*">
      <json:boolean key="{f:camelify(name())}"><xsl:value-of select=". = 'true'"/></json:boolean>
    </xsl:for-each>
  </json:map>
</xsl:template>

<xsl:function name="f:to-json-search-parameters" as="xs:string">
  <xsl:param name="parameters" as="element(parameter)*"/>
  <xsl:variable name="object">
    <json:map>
      <json:boolean key="isSearchPage">true</json:boolean>
      <json:string key="question"><xsl:value-of select="$parameters[@name='question']"/></json:string>
      <json:string key="scope"><xsl:value-of select="$parameters[@name='scope']"/></json:string>
      <json:string key="filters"><xsl:value-of select="$parameters[@name='filters']"/></json:string>
      <json:string key="ranges"><xsl:value-of select="$parameters[@name='ranges']"/></json:string>
      <json:string key="columns"><xsl:value-of select="$parameters[@name='columns']"/></json:string>
      <json:string key="format"><xsl:value-of select="$parameters[@name='format']"/></json:string>
      <xsl:if test="$parameters[@name='questionfields']">
        <json:string key="questionfields"><xsl:value-of select="$parameters[@name='questionfields']"/></json:string>
      </xsl:if>
      <xsl:if test="$parameters[@name='sortfields']">
        <json:string key="sortfields"><xsl:value-of select="$parameters[@name='sortfields']"/></json:string>
      </xsl:if>
      <xsl:if test="$parameters[@name='page'] and $parameters[@name='page'] castable as xs:int">
        <json:number key="page"><xsl:value-of select="$parameters[@name='page']"/></json:number>
      </xsl:if>
      <xsl:if test="$parameters[@name='pagesize'] and $parameters[@name='pagesize'] castable as xs:int">
        <json:number key="pageSize"><xsl:value-of select="$parameters[@name='pagesize']"/></json:number>
      </xsl:if>
      <xsl:if test="$parameters[@name='global'] = 'true'">
        <json:boolean key="global">true</json:boolean>
      </xsl:if>
    </json:map>
  </xsl:variable>
  <xsl:value-of select="json:xml-to-json($object)"/>
</xsl:function>

<!--
  Load all labels defined in the variables provided:
    - all labels from label-config
    - all labels in document-types <labeling> tag
    - all xref labels in xref-configs
    - all creation labels in document-types <creation> tag

  @param label-config the label config
  @param document-types the document type configs
  @param xref-configs the xref configs

  @returns An array of labels
-->
<xsl:function name="f:to-json-all-labels">
  <xsl:param name="label-config" />
  <xsl:param name="document-types" />
  <xsl:param name="xref-configs" />

  <xsl:variable name="labels">
    <json:array>
      <!-- first all labels -->
      <xsl:variable name="all-labels" select="$label-config/labels/label" />
      <xsl:for-each select="$all-labels">
        <json:map>
          <json:string key="name"><xsl:value-of select="@name" /></json:string>
          <json:string key="type"><xsl:value-of select="../@type" /></json:string>
          <json:string key="color"><xsl:value-of select="@color" /></json:string>
          <json:string key="description"><xsl:value-of select="@description" /></json:string>
          <json:string key="from">project</json:string>
        </json:map>
      </xsl:for-each>
      <!-- labels defined in document types -->
      <xsl:variable name="custom-labels" select="$document-types//labeling/labels/label[empty(index-of($all-labels/@name, @name))]" />
      <xsl:for-each select="$custom-labels">
        <json:map>
          <json:string key="name"><xsl:value-of select="@name" /></json:string>
          <json:string key="type"><xsl:value-of select="../@type" /></json:string>
          <json:string key="color"><xsl:value-of select="@color" /></json:string>
          <json:string key="description"><xsl:value-of select="@description" /></json:string>
          <json:string key="doctype"><xsl:value-of select="ancestor::document-type/@name" /></json:string>
          <json:string key="from">document-type</json:string>
          <xsl:if test="@fragmenttypes">
            <json:array key="fragmentTypes">
              <xsl:for-each select="tokenize(@fragmenttypes, ',')"><json:string><xsl:value-of select="."/></json:string></xsl:for-each>
            </json:array>
          </xsl:if>
        </json:map>
      </xsl:for-each>
      <!-- xref labels in xref-configs -->
      <xsl:for-each select="$xref-configs/xref-config/xref[@labels]">
        <xsl:variable name="config" select="../@name" />
        <xsl:variable name="doctype" select="ancestor::document-type/@name" />
        <xsl:variable name="editor" select="ancestor::editor-config/@name" />
        <xsl:for-each select="tokenize(@labels, ',')[empty(index-of(($all-labels | $custom-labels)/@name, .))]">
          <json:map>
            <json:string key="name"><xsl:value-of select="." /></json:string>
            <json:string key="type">xref</json:string>
            <json:string key="from"><xsl:value-of select="if ($editor) then 'editor' else if ($doctype) then 'document-type' else 'xref-config'" /></json:string>
            <xsl:if test="$config"><json:string key="config"><xsl:value-of select="$config" /></json:string></xsl:if>
            <xsl:if test="$doctype"><json:string key="doctype"><xsl:value-of select="$doctype" /></json:string></xsl:if>
            <xsl:if test="$editor"><json:string key="editor"><xsl:value-of select="$editor" /></json:string></xsl:if>
          </json:map>
        </xsl:for-each>
      </xsl:for-each>
      <!-- creation labels -->
      <xsl:for-each select="$document-types//creation/document[@labels]">
        <xsl:variable name="doctype" select="ancestor::document-type/@name" />
        <xsl:for-each select="tokenize(@labels, ',')[empty(index-of(($all-labels | $custom-labels)/@name, .))]">
          <json:map>
            <json:string key="name"><xsl:value-of select="." /></json:string>
            <json:string key="type">document</json:string>
            <json:string key="doctype"><xsl:value-of select="$doctype" /></json:string>
            <json:boolean key="creation">true</json:boolean>
            <json:string key="from">document-type</json:string>
          </json:map>
        </xsl:for-each>
      </xsl:for-each>
    </json:array>
  </xsl:variable>
  <xsl:value-of select="json:xml-to-json($labels)"/>
</xsl:function>

<!--
  @return the array of the labels as a JSON string
-->
<xsl:function name="f:to-json-labels-array" as="xs:string">
  <xsl:param name="label-config" as="element(label-config)?"/>
  <xsl:variable name="labels-array">
    <json:array>
      <xsl:for-each select="$label-config/labels">
        <xsl:variable name="type" select="@type"/>
        <xsl:for-each select="label">
          <json:map>
            <json:string key="name"><xsl:value-of select="@name" /></json:string>
            <json:string key="type"><xsl:value-of select="$type" /></json:string>
            <json:string key="description"><xsl:value-of select="@description" /></json:string>
          </json:map>
        </xsl:for-each>
      </xsl:for-each>
    </json:array>
  </xsl:variable>
  <xsl:value-of select="json:xml-to-json($labels-array)"/>
</xsl:function>

</xsl:stylesheet>