<?xml version="1.0"?>
<!--
  Display diffx elements/attributes

  Diff templates are designed to call the next matching template for each element so that they augment
  the templates using `<next-match />`

  Use priorities to ensure the correct sequencing of templates

  1. Regular element specific template (default priority ~ 0.5)
  2. Diff element specific template (priority 10)
  3. Diff genetic template (priority 20)
  3. Diff-info template (priority 30)

  @author Philip Rutherford
  @author Christophe Lauret

  @since 5.8000
-->
<xsl:stylesheet version="3.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:json="http://www.w3.org/2005/xpath-functions"
                xmlns:diff="https://www.pageseeder.org/diffx"
                xmlns:track="https://www.pageseeder.org/diffx/track"
                xmlns:del="https://www.pageseeder.org/diffx/delete"
                xmlns:ins="https://www.pageseeder.org/diffx/insert"
                xmlns:mml="http://www.w3.org/1998/Math/MathML"
                xmlns:svg="http://www.w3.org/2000/svg"
                exclude-result-prefixes="#all">

<!-- Diff-info (priority 30) ================================================================== -->

<!--
  Block level elements need to be wrapped to provide diff-info.
-->
<xsl:template match="block[@del:* or @ins:*]
                   | blockxref[@del:* or @ins:*]
                   | heading[@del:* or @ins:*]
                   | image[@del:* or @ins:*]
                   | list[@del:* or @ins:*]
                   | nlist[@del:* or @ins:*]
                   | para[@del:* or @ins:*]
                   | preformat[@del:* or @ins:*]
                   | table[@del:* or @ins:*]" mode="psml" priority="30">
  <div class="diff-info-holder{if (@diff:insert) then ' is-inserted-quiet' else if (@diff:delete) then ' is-deleted-quiet' else ''}">
    <xsl:apply-templates select="." mode="diff-info" />
  </div>
  <xsl:next-match />
</xsl:template>

<!--
  Inline level elements need to be wrapped to provide diff-info.
-->
<xsl:template match="inline[@del:* or @ins:*]
                   | link[@del:* or @ins:*]" mode="psml" priority="30">
  <xsl:next-match />
  <span class="{if (@diff:insert) then ' is-inserted-quiet' else if (@diff:delete) then ' is-deleted-quiet' else ''}">
    <xsl:apply-templates select="." mode="diff-info" />
  </span>
</xsl:template>

<!--
  Create the `<diff-info>` component if there are any inserted/deleted attributes.
-->
<xsl:template match="*" name="diff-info" mode="diff-info">
  <xsl:if test="@ins:* or @del:*">
    <xsl:variable name="info">
      <json:map>
        <json:string key="element"><xsl:value-of select="name()"/></json:string>
        <json:map key="ins">
          <xsl:for-each select="@ins:*">
            <json:string key="{local-name()}"><xsl:value-of select="../@*[name() = local-name(current())]"/></json:string>
          </xsl:for-each>
        </json:map>
        <json:map key="del">
          <xsl:for-each select="@del:*">
            <json:string key="{local-name()}"><xsl:value-of select="."/></json:string>
          </xsl:for-each>
        </json:map>
      </json:map>
    </xsl:variable>
    <diff-info v-bind="{json:xml-to-json($info)}" />
  </xsl:if>
</xsl:template>


<!-- Generic diff =========================================================================== -->

<!--
  When an element is replaced with another with no change in content (e.g. para becomes a heading),
  we report it as modified.
-->
<xsl:template match="*[@diff:insert = 'true' and (empty(diff:ins) or *[1]/@diff:delete = 'true')]" priority="25" mode="psml">
  <xsl:variable name="result">
    <xsl:next-match />
  </xsl:variable>
  <xsl:for-each select="$result/*">
    <xsl:copy>
      <xsl:attribute name="class"><xsl:value-of select="@class" /> is-modified</xsl:attribute>
      <xsl:copy-of select="@*[name() != 'class']" />
      <xsl:copy-of select="*|text()" />
    </xsl:copy>
  </xsl:for-each>
</xsl:template>

<!--
  Works in tandem with template above (ignore previous element)
-->
<xsl:template match="*[@diff:delete = 'true' and empty(preceding-sibling::*) and ../@diff:insert = 'true']" priority="25" mode="psml">
  <xsl:apply-templates mode="psml" />
</xsl:template>

<!--
  Inserted and deleted elements have classes used for filtering in side-by-side view
-->
<xsl:template match="*[@diff:* or @del:* or @ins:*]" mode="psml" priority="20">
  <xsl:variable name="result"><xsl:next-match /></xsl:variable>
  <xsl:variable name="current" select="." />
  <xsl:for-each select="$result/*">
    <xsl:copy>
      <xsl:copy-of select="@* except @class" />
      <xsl:attribute name="class">
        <xsl:value-of select="@class"/>
        <xsl:choose>
          <xsl:when test="$current/@diff:insert"> is-inserted</xsl:when>
          <xsl:when test="$current/@diff:delete"> is-deleted</xsl:when>
          <xsl:when test="$current/@ins:* or $current/@del:*"> is-modified</xsl:when>
        </xsl:choose>
      </xsl:attribute>
      <xsl:copy-of select="node()" />
    </xsl:copy>
  </xsl:for-each>
</xsl:template>

<!-- Specific elements customization (priority 10) ============================================ -->

<!-- Headings and paragraphs -->
<xsl:template match="heading[@del:* or @ins:*]
                   | para[@del:* or @ins:*]" mode="psml" priority="10">
  <xsl:variable name="result"><xsl:next-match /></xsl:variable>
  <xsl:variable name="current" select="." />
  <xsl:for-each select="$result/*">
    <xsl:copy>
      <xsl:copy-of select="@*[not(name() = 'data-prefix' or name() = 'data-numbered')]" />
      <!-- Any change in the prefix we put in diffx-prefix element -->
      <xsl:choose>
        <xsl:when test="$current/@ins:prefix or $current/@del:prefix">
          <span class="diff-prefix">
            <xsl:if test="$current/@ins:prefix"><ins class="prefix"><xsl:value-of select="$current/@prefix"/></ins></xsl:if>
            <xsl:if test="$current/@del:prefix"><del class="prefix"><xsl:value-of select="$current/@del:prefix"/></del></xsl:if>
          </span>
        </xsl:when>
        <xsl:otherwise>
          <xsl:copy-of select="@data-prefix|@data-numbered"/>
        </xsl:otherwise>
      </xsl:choose>
      <xsl:copy-of select="*|text()" />
    </xsl:copy>
  </xsl:for-each>
</xsl:template>

<!-- Images (Include previous image) -->
<xsl:template match="image[@del:src or @ins:src]" mode="psml" priority="10">
  <xsl:if test="@ins:src">
    <xsl:variable name="current-image">
      <image diff:insert="true">
        <xsl:copy-of select="@* except (@del:*, @ins:*)"/>
      </image>
    </xsl:variable>
    <xsl:apply-templates select="$current-image" mode="psml"/>
  </xsl:if>
  <xsl:if test="@del:src">
    <xsl:variable name="previous-image">
      <image diff:delete="true">
        <xsl:for-each select="@del:*">
          <xsl:attribute name="{local-name()}"><xsl:value-of select="."/></xsl:attribute>
        </xsl:for-each>
      </image>
    </xsl:variable>
    <xsl:apply-templates select="$previous-image" mode="psml"/>
  </xsl:if>
</xsl:template>

<!-- Lists (preserve old type) -->
<xsl:template match="list[@ins:type or @del:type]
                  | nlist[@ins:type or @del:type]" mode="psml" priority="10">
  <xsl:variable name="result"><xsl:next-match /></xsl:variable>
  <xsl:variable name="current" select="." />
  <xsl:for-each select="$result/*">
    <xsl:copy>
      <!-- TODO Should no longre be necessary -->
      <xsl:copy-of select="@*[not(name() = 'class')]" />
      <xsl:attribute name="class"><xsl:value-of select="@class"/> was-type-<xsl:value-of select="$current/@del:type"/></xsl:attribute>
      <xsl:copy-of select="*|text()" />
    </xsl:copy>
  </xsl:for-each>
</xsl:template>

<!-- Sadly, we cannot use syntax highlighting for preformatted content comparisons -->
<xsl:template match="preformat[diff:*|track:*]" mode="psml" priority="10">
  <pre>
    <xsl:apply-templates select="@role" mode="psml-data"/>
    <xsl:if test="@role">
      <xsl:attribute name="class" select="@role" />
    </xsl:if>
    <xsl:apply-templates mode="#current"/>
  </pre>
</xsl:template>

<!-- Inserts and deletions (default priority) ================================================= -->

<!-- Inserted content reported by Diff-X -->
<xsl:template match="diff:ins" mode="psml psml-transclude psml-pre">
  <ins><xsl:apply-templates mode="#current" /></ins>
</xsl:template>

<!-- Deleted content reported by Diff-X -->
<xsl:template match="diff:del" mode="psml psml-transclude psml-pre">
  <del><xsl:apply-templates mode="#current" /></del>
</xsl:template>

<!-- Do as if they weren't here in edit mode -->
<xsl:template match="diff:ins|track:ins" mode="psml-edit">
  <xsl:apply-templates mode="#current" />
</xsl:template>

<!-- Suppress in edit mode -->
<xsl:template match="diff:del|track:del" mode="psml-edit" />

<!-- Display inserted MathML content in green -->
<xsl:template match="math|mml:math" mode="mathml-ins">
  <!-- Colors defined in diff.scss (must be hardcoded in MathML) -->
  <math class="is-inserted" mathcolor="#01b30f" mathbackground="#dfffdf">
    <xsl:copy-of select="@*[not(local-name()='mathcolor' or local-name()='mathbackground')]" />
    <xsl:copy-of select="*|text()" />
  </math>
</xsl:template>

<!-- Display deleted MathML content in red -->
<xsl:template match="math|mml:math" mode="mathml-del">
  <!-- Colors defined in diff.scss (must be hardcoded in MathML) -->
  <math class="is-deleted" mathcolor="#cc0011" mathbackground="#ffdfdf">
    <xsl:copy-of select="@*[not(local-name()='mathcolor' or local-name()='mathbackground')]" />
    <xsl:copy-of select="*|text()" />
  </math>
</xsl:template>

<!-- MathML content with tracker -->
<xsl:template match="*|mml:*" mode="mathml-track">
  <xsl:copy>
    <xsl:choose>
      <!-- Colors defined in diff.scss (must be hardcoded in MathML) -->
      <xsl:when test="track:ins or diff:ins">
        <xsl:attribute name="mathcolor">#01b30f</xsl:attribute>
        <xsl:attribute name="mathbackground">#dfffdf</xsl:attribute>
        <xsl:copy-of select="@* except (@mathcolor|@mathbackground|@track:*|@diff:*|@del:*|@ins:*)" />
      </xsl:when>
      <xsl:when test="track:del or diff:del">
        <xsl:attribute name="mathcolor">#cc0011</xsl:attribute>
        <xsl:attribute name="mathbackground">#ffdfdf</xsl:attribute>
        <xsl:copy-of select="@* except (@mathcolor|@mathbackground|@track:*|@diff:*|@del:*|@ins:*)" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:copy-of select="@* except (@track:*|@diff:*|@del:*|@ins:*)" />
      </xsl:otherwise>
    </xsl:choose>
    <xsl:apply-templates select="*|text()" mode="mathml-track"/>
  </xsl:copy>
</xsl:template>

<xsl:template match="track:*|diff:*" mode="mathml-track">
  <xsl:apply-templates select="*|text()" mode="mathml-track"/>
</xsl:template>

<!-- Inserted SVG in green -->
<xsl:template match="svg|svg:svg" mode="svg-ins">
  <svg class="is-inserted {@class}">
    <xsl:copy-of select="@* except @class" />
    <xsl:copy-of select="*|text()" />
  </svg>
</xsl:template>

<!-- Deleted SVG in red -->
<xsl:template match="svg|svg:svg" mode="svg-del">
  <svg class="is-deleted {@class}">
    <xsl:copy-of select="@* except @class" />
    <xsl:copy-of select="*|text()" />
  </svg>
</xsl:template>

<!-- SVG content with tracker -->
<xsl:template match="*" mode="svg-track">
  <xsl:copy>
    <xsl:copy-of select="@* except @track:*" />
    <xsl:apply-templates select="*|text()" mode="svg-track"/>
  </xsl:copy>
</xsl:template>

<xsl:template match="track:*" mode="svg-track">
  <xsl:apply-templates select="*|text()" mode="svg-track"/>
</xsl:template>

<!-- Legacy templates ================================================ -->

<!-- Display changes to cell or hcell -->
<!--<xsl:template match="cell[@ins:align or @ins:role or @ins:colspan or @ins:rowspan or-->
<!--                        @del:align or @del:role or @del:colspan or @del:rowspan][not(@diff:delete)]|-->
<!--                  hcell[@ins:align or @ins:role or @ins:colspan or @ins:rowspan or-->
<!--                        @del:align or @del:role or @del:colspan or @del:rowspan][not(@diff:delete)]" mode="psml-edit">-->
<!--  <xsl:variable name="result">-->
<!--    <xsl:next-match />-->
<!--  </xsl:variable>-->
<!--  <xsl:for-each select="$result/*">-->
<!--    <xsl:copy>-->
<!--      <xsl:attribute name="class"><xsl:value-of select="@class" /> modified</xsl:attribute>-->
<!--      <xsl:copy-of select="@*[name() != 'class']" />-->
<!--      <xsl:copy-of select="node()" />-->
<!--    </xsl:copy>-->
<!--  </xsl:for-each>-->
<!--</xsl:template>-->

<!-- Display changes to row -->
<!--<xsl:template match="cell[ancestor::row[@ins:align or @ins:role or @ins:part or-->
<!--                                      @del:align or @del:role or @del:part]][not(@diff:delete)]|-->
<!--                  hcell[ancestor::row[@ins:align or @ins:role or @ins:part or-->
<!--                                      @del:align or @del:role or @del:part]][not(@diff:delete)]" mode="psml-edit">-->
<!--  <xsl:variable name="result">-->
<!--    <xsl:next-match />-->
<!--  </xsl:variable>-->
<!--  <xsl:for-each select="$result/*">-->
<!--    <xsl:copy>-->
<!--      <xsl:attribute name="class"><xsl:value-of select="@class" /> modified</xsl:attribute>-->
<!--      <xsl:copy-of select="@*[name() != 'class']" />-->
<!--      <xsl:copy-of select="node()" />-->
<!--    </xsl:copy>-->
<!--  </xsl:for-each>-->
<!--</xsl:template>-->

<!-- Display changes to table or col -->
<!--<xsl:template match="table[@ins:role or @ins:height or @ins:width or-->
<!--                         @del:role or @del:height or @ins:width or-->
<!--                         col[@ins:align or @ins:role or @ins:part or @ins:span or @ins:width or-->
<!--                             @del:align or @del:role or @del:part or @ins:span or @ins:width]]" mode="psml-edit">-->
<!--  <xsl:variable name="result">-->
<!--    <xsl:next-match />-->
<!--  </xsl:variable>-->
<!--  <xsl:for-each select="$result/*">-->
<!--    <xsl:copy>-->
<!--      <xsl:attribute name="class"><xsl:value-of select="@class" /> modified</xsl:attribute>-->
<!--      <xsl:copy-of select="@*[name() != 'class']" />-->
<!--      <xsl:copy-of select="node()" />-->
<!--    </xsl:copy>-->
<!--  </xsl:for-each>-->
<!--</xsl:template>-->

<!-- Tracker (split) -->
<xsl:template match="track:ins|track:del" mode="psml">
  <xsl:variable name="id" select="@editid"/>
  <xsl:variable name="edit" select="(//edit[@id=$id])[1]"/>
  <xsl:element name="{local-name(.)}">
    <xsl:attribute name="data-editid"><xsl:value-of select="$edit/@id"/></xsl:attribute>
    <xsl:attribute name="data-author"><xsl:value-of select="$edit/author/@id"/></xsl:attribute>
    <xsl:attribute name="class">is-edit</xsl:attribute>
    <xsl:attribute name="v-tooltip">'<xsl:value-of select="if (local-name() = 'ins') then 'Inserted' else 'Deleted'"/> by <xsl:value-of select="$edit/author/fullname"/> on <xsl:value-of select="format-dateTime($edit/@created, '[FNn,*-3] [D] [MNn,*-3] [Y] at [H01]:[m01]')"/>'</xsl:attribute>
    <xsl:attribute name="datetime"><xsl:value-of select="$edit/@created"/></xsl:attribute>
    <xsl:choose>
      <xsl:when test="text() = ' '">&#xa0;</xsl:when>
      <xsl:otherwise><xsl:apply-templates mode="psml"/></xsl:otherwise>
    </xsl:choose></xsl:element>
</xsl:template>

<!-- Tracker (combined) -->
<xsl:template match="track:span" mode="psml">
  <xsl:variable name="id" select="@editid"/>
  <xsl:variable name="edit" select="(//edit[@id=$id])[1]"/>
  <mark data-editid="{$edit/@id}" class="is-edit author-{($edit/author/@id * 37) mod 360}"
        data-color="{($edit/author/@id * 37) mod 360}"
        data-author="{$edit/author/@id}"
        v-tooltip="`Edited by {$edit/author/fullname} on {format-dateTime($edit/@created, '[FNn,*-3] [D] [MNn,*-3] [Y] at [H01]:[m01]')}`"
        datetime="{$edit/@created}"
  ><xsl:choose>
    <xsl:when test="text() = ' '">&#xa0;</xsl:when>
    <xsl:otherwise><xsl:apply-templates mode="psml"/></xsl:otherwise>
  </xsl:choose></mark>
</xsl:template>

<xsl:template match="track:span" mode="psml-edit">
  <xsl:apply-templates mode="#current"/>
</xsl:template>

<!-- TODO Check if still required -->
<!--<xsl:template match="document[@tracking]//(block|heading|para)[@track:*]" mode="psml">-->
<!--  <div class="diff-info-holder">-->
<!--    <xsl:variable name="info">-->
<!--      <json:map>-->
<!--        <json:string key="element"><xsl:value-of select="name()"/></json:string>-->
<!--        <json:map key="{ancestor::document/@tracking}">-->
<!--          <xsl:for-each select="@track:*">-->
<!--            <json:string key="{local-name()}"><xsl:value-of select="../@*[name() = local-name(current())]"/></json:string>-->
<!--          </xsl:for-each>-->
<!--        </json:map>-->
<!--        <json:map key="{if (ancestor::document/@tracking = 'ins') then 'del' else 'ins'}" />-->
<!--      </json:map>-->
<!--    </xsl:variable>-->
<!--    <diff-info v-bind="{json:xml-to-json($info)}" />-->
<!--  </div>-->
<!--  <xsl:next-match />-->
<!--</xsl:template>-->

</xsl:stylesheet>
