<!--
  Indexing transformations for PageSeeder tasks

  The source XML follows the structure below:

  ```
    <index-data mediatype="application/vnd.pageseeder.task+xml">
      <comment> ... </comment> ????
    </index-data>
  ```

  @author Christophe Lauret
  @author Jean-Baptiste Reure

  @version 5.9000
-->
<xsl:transform  xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="2.0"
                xmlns:psf="http://www.pageseeder.com/function"
                xmlns:xs="http://www.w3.org/2001/XMLSchema"
                exclude-result-prefixes="#all">

<!-- Contains commons fields -->
<xsl:import href="common.xsl" />

<!-- Using Flint Index Document 5.0 -->
<xsl:output method="xml" indent="no" encoding="utf-8"
            doctype-public="-//Weborganic//DTD::Flint Index Documents 5.0//EN"
            doctype-system="http://weborganic.org/schema/flint/index-documents-5.0.dtd"/>

<!-- Sent by the index processor -->
<xsl:param name="psid-field-name" select="'psid'" />
<xsl:param name="psid-field-value" />

<!--
  Index task
-->
<xsl:template match="/index-data">
  <documents version="5.0">
    <document>
      <xsl:apply-templates select="task"                     mode="ixml"/>
      <xsl:apply-templates select="publications/publication" mode="ixml"/>
    </document>
  </documents>
</xsl:template>

<!--
  Generate the fields specific to a task from a `comment`
-->
<xsl:template match="task" mode="ixml">
  <!-- Intiator for workflow is last comment, otherwise it is last time re-opened -->
  <xsl:variable name="initiator" select="if (comment/@contentrole='Workflow') then comment[last()]
                                         else (comment[@status='Open' and not(preceding-sibling::comment[1]/@status = 'Open')][last()] |
                                         (comment[empty(current()/comment[@status = 'Open'])][@status])[1])"/>

  <!-- Core fields -->
  <xsl:sequence select="psf:ps-id($psid-field-name, @id)"/>
  <xsl:sequence select="psf:ps-title(@title)"/>
  <xsl:sequence select="psf:ps-type('task')"/>
  <xsl:sequence select="psf:ps-mediatype('application/vnd.pageseeder.task+xml')"/>

  <!-- Get latest created or modified date -->
  <xsl:variable name="dates" select="comment/@created | comment/modifiedby/@date" as="xs:string*" />
  <xsl:variable name="sortdates" as="xs:string*">
    <xsl:perform-sort select="$dates">
      <xsl:sort select="." />
    </xsl:perform-sort>
  </xsl:variable>
  <xsl:sequence select="psf:ps-modifieddate(xs:dateTime($sortdates[last()]))"/>

  <!-- Initiator fields -->
  <xsl:sequence select="psf:ps-taskdefinitionid($initiator/@id)"/>
  <xsl:sequence select="psf:ps-subtype(if (comment/@contentrole='Workflow') then 'workflow'
                                       else if ($initiator/@type) then $initiator/@type else 'task')"/>
  <xsl:sequence select="psf:ps-labels($initiator/labels)"/>
  <xsl:sequence select="psf:ps-createddate($initiator/@created)"/>
  <xsl:sequence select="psf:ps-author($initiator/author/fullname)"/>
  <xsl:if test="$initiator/author/@id">
    <xsl:sequence select="psf:ps-authorid($initiator/author/@id)"/>
  </xsl:if>
  <xsl:sequence select="psf:ps-description($initiator/content)"/>

  <!-- Initiator properties -->
  <xsl:for-each select="$initiator//property">
    <xsl:sequence select="psf:ps-property(.)"/>
  </xsl:for-each>

  <!-- Context -->
  <xsl:if test="uri">
    <xsl:sequence select="psf:ps-folder(replace(uri/@decodedpath, '/[^/]*?$', ''))"/>
  </xsl:if>

  <!-- Task specific fields -->
  <xsl:sequence select="psf:ps-status(@status)"/>
  <xsl:sequence select="psf:ps-duedate(@due)"/>
  <xsl:sequence select="psf:ps-priority(@priority)"/>
  <xsl:sequence select="psf:ps-assignedto(assignedto/fullname)"/>
  <xsl:if test="assignedto/@id">
    <xsl:sequence select="psf:ps-assignedtoid(assignedto/@id)"/>
  </xsl:if>
  <xsl:sequence select="psf:ps-contextexternal(uri/@external = 'true')"/>
  <xsl:sequence select="psf:ps-contexturi(uri)"/>
  <xsl:sequence select="psf:ps-contextfragment(if (locator/@fragment) then locator/@fragment else 'default')"/>
  <xsl:variable name="status-changed"
      select="(comment[@status = current()/@status and not(following-sibling::comment[@status != current()/@status])])[1]" />
  <xsl:sequence select="psf:ps-statuschangeddate($status-changed/@created)"/>

  <!-- XXX: Content from all comments? -->
  <xsl:sequence select="psf:ps-content(normalize-space(string-join(comment/content//text(), ' ')))"/>

  <!-- Any other custom fields -->
  <xsl:call-template name="custom-fields" />

</xsl:template>

<!-- TEMPLATES LEFT BLANK FOR CUSTOMISATION =============================== -->

<!--
  Use this template to add extra fields for each task

  <field store="true" name="my_field">field contents</field>

  @context task
-->
<xsl:template name="custom-fields" />

</xsl:transform>