/**
 * $Id: DefaultsValueURIHandlerTest.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.tests;

import  org.apache.tools.ant.Project;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.valueuri.info.DefaultsValueURIHandler;

/**
 * Testsuite for {@linkplain DefaultsValueURIHandler}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class DefaultsValueURIHandlerTest extends ValueURIHandlerTestSkeleton
{
    /**
     * Initializes a new test case for named method.
     * @param methodname test case method's name (non-null)
     **/
    public DefaultsValueURIHandlerTest(String methodname)
    {
        super("DefaultsValueURIHandler::",methodname);
    }


    /**
     * Returns full test suite for DefaultsValueURIHandler.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(DefaultsValueURIHandlerTest.class);
    }

//  ---------------------------------------------------------------------------------------------------------
//  --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    protected String xmlTestFile()
    {
        return "defaults.xml";
    }

    protected ValueURIHandler newOUT()
    {
        return new DefaultsValueURIHandler();
    }
    
    protected void setUp() throws Exception
    {
        super.setUp();
        assertNil(getProject().getProperty("defaults.licenses"),
                 "'defaults.licenses' property");
    }

//  ---------------------------------------------------------------------------------------------------------
//  ------------------------------------------- [ The Test Cases ] ------------------------------------------
//  ---------------------------------------------------------------------------------------------------------

    /**
     * Verify a blank uri fragment is evaluated as null.
     * @since JWare/AntX 0.5
     **/
    public void testBaseline_AntX05()
    {
        checkBaseline();

        String result = newOUT().valueFrom("","$default:",m_rqlink);
        assertNil(result,"$default:");
    }


    /**
     * Verify the capitalization of the key is unimportant.
     * @since JWare/AntX 0.5
     **/
    public void testKeyIsCaseInsensitive_AntX05()
    {
        ValueURIHandler out = newOUT();
        String result1 = out.valueFrom("CONFIGID","$default:CONFIGID",m_rqlink);
        assertEqual(result1,Iteration.configId(),"$default:CONFIGID");
        assertTrue(result1.startsWith("jware.antx"),"configid startswith jware.antx");
        String result2 = out.valueFrom("configid","$default:configid",m_rqlink);
        assertEqual(result2,result1,"$default:config");

        result1 = out.valueFrom("haltiferror","$default:haltiferror",m_rqlink);
        result2 = out.valueFrom("haltIfError","$default:haltIfError",m_rqlink);
        assertEqual(result2,result1,"$default:haltIfError");
        assertEqual(result1,String.valueOf(Iteration.defaultdefaults().isHaltIfError()));
    }


    /**
     * Verify can include a fallback value if handler unable to 
     * determine default value from context.
     * @since JWare/AntX 0.5
     **/
    public void testCanInlineDefaultsIfNotPredefined_AntX05()
    {
        final Project myproject = getProject();

        ValueURIHandler out = newOUT();
        String license = out.valueFrom("licenses?LGPL","$df:licenses?LGPL",m_rqlink);
        assertEqual(license,"LGPL","$df:licenses?LGPL");

        assertNil(myproject.getProperty("fallback"),"'fallback' property");
        myproject.setProperty("fallback", "MIT");
        license = out.valueFrom("licenses?@(fallback)/latest","$df:licenses?@(fallback)/latest",m_rqlink);
        assertEqual(license,"MIT/latest","$df:licenses?@(fallback)/latest with $fallback=MIT");

        myproject.setProperty("defaults.licenses","Apache2");
        license = out.valueFrom("licenses?LGPL","$df:licenses?LGPL",m_rqlink);
        assertEqual(license,"Apache2","$df:licenses?LGPL with defaults.license=Apache2");
    }
    
    
    
    /**
     * Verifies that the key of the URI fragment must be a resolved
     * value. Property references are supported only as part of the 
     * inline default string.
     * @since JWare/AntX 0.5
     **/
    public void testLiteralKeyRequired_AntX05()
    {
        final Project myproject = getProject();
        myproject.setProperty("dotd","assertions");
        
        ValueURIHandler out = newOUT();
        String result = out.valueFrom("@(dotd)","$df:@(dotd)",m_rqlink);
        assertNil(result,"$df:@(dotd)");
        
        result = out.valueFrom("@(dotd)?@(dotd)","$df:@(dotd)?@(dotd)",m_rqlink);
        assertEqual(result,"assertions","$df:@(dotd)?@(dotd)");
    }



    /**
     * Verify some things from within an Ant-only environment.
     * @since JWare/AntX 0.5
     **/
    public void testDefaultsHandlerInAntScript_AntX05()
    {
        runTarget("testInstallDefaultsHandler_AntX05");
    }


    /**
     * Ensures the general "getDefault" API can extract inlined defaults.
     * @since JWare/AntX 0.5
     **/
    public void testDefaultsValueApiCanExtractInlineDefaults_AntX05()
    {
        final Project myproject = getProject();
        myproject.setProperty("defaults.licenses","MIT");
        
        ValueURIHandler out = newOUT();
        String result = out.getDefaultValue("$df:licenses?GPL",m_rqlink);
        assertEqual(result,"GPL","getDefault('$df:licenses?GPL')");

        myproject.setProperty("fallback","CollabNet");
        result = out.getDefaultValue("$df:licenses?@(fallback)",m_rqlink);
        assertEqual(result,"CollabNet","getDefault('$df:licenses?@(fallback)') where $fallback=CollabNet");
    }
}

/* end-of-DefaultsValueURIHandlerTest.java */