/**
 * $Id: PasswordValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.info;

import  java.io.File;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.ValueURIHandler;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.InputFileLoader;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.TransformHelper;
import  com.idaremedia.antx.parameters.ValueTransform;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Example value uri handler that lets a script reference a password without embedding
 * the password directly in script. Whether the referred to password is clear text
 * or cipher text is application dependent.
 * <p/>
 * The handler looks for a set of properties of username=password settings. It
 * looks for these properties in four places:<ol>
 *   <li>A <span class="src">passwordfile</span> artifact value URI. If available,
 *       the artifact must specify either a file path or a file path URL.</li>
 *   <li>A local &lt;propertyset&gt; or &lt;properties&gt; object under refid
 *       <span class="src">build&#46;passwords</span>. This option exists primarily
 *       to aid debugging and script development.</li>
 *   <li>The file named in the default iteration password file property. See
 *       {@linkplain com.idaremedia.antx.Defaults#passwordFileLocation
 *        Defaults.passwordFileLocation} for more information.</li>
 *   <li>A <span class="src">build&#46;passwords</span> file stored under the 
 *       directory <span class="src">${user&#46;home}/&#46;ant</span>.</li>
 * </ol>
 * At least <em>one</em> of these places must contain a valid set of properties. If no
 * properties are found, this value uri handler signals an error. (This not the same
 * response the handler generates if it does not find a matching entry for the password. 
 * If the properties exist but there is no match, the handler returns <i>null</i> or
 * the password stored as the default.)
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;baseurl name="data" uri="http://qalab.mysoft.com/myproject/data/&gt;
 *       &lt;parameter name="auth:username" value="nightlybuild"/&gt;
 *       &lt;parameter name="auth:password" value="${$password:nightlybuild}"/&gt;
 *    &lt;/baseurl&gt;
 * 
 *   -- To Install --
 *    &lt;valueuri-interpreter action="install"&gt;
 *       &lt;parameter name="password"
 *             value="com.idaremedia.antx.valueuri.info.PasswordValueURIHandler"/&gt;
 *    &lt;/valueuri-interpreter&gt;
 * </pre>
 *
 * @since     AntXtras/Oofs 1.0
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    helper,example
 * @.caveat   The password file is loaded <em>everytime</em> a handler is
 *            asked for a value. Nothing is cached.
 **/

public final class PasswordValueURIHandler extends ValueURIHandlerSkeleton
{
    private static final boolean OPTIONAL= true;

    /**
     * Tries to load the passwords properties file to search for
     * a single entry. (A bit excessive but we are just a sample.)
     * @param f password file
     * @param optional <i>true</i> if file does not have to exist.
     * @param clnt problem handler (non-null)
     * @return password file as Properties object (null if optional
     *         and cannot find/load file).
     * @throws BuildException if unable to load file as properties 
     *         file for any reason and is not optional
     **/
    private Properties readFile(File f, boolean optional, Requester clnt)
    {
        try {
            return InputFileLoader.loadProperties
                (AntXFixture.fileUtils().getFileURL(f), null);
        } catch(Exception anyX) {
            clnt.problem(anyX.getMessage(), 
                optional? Project.MSG_WARN : Project.MSG_ERR);
            if (!optional) {
                throw new BuildException(anyX,clnt.getLocation());
            }
        }
        return null;
    }



    /**
     * Get password file according to our search algorithm. First we
     * look for an artifact "<span class="src">passwordfile</span>".
     * Next we look for a local propertyset (or properties) called 
     * "<span class="src">build&#46;passwords</span>". Next we look
     * for a client-supplied file using the iteration's passwordfile
     * configuration setting. If no such setting, we look for a file
     * <span class="src">build&#46;passwords</span> in the user's ant 
     * directory at <span class="src">${user&#46;home}/&#46;ant</span>.
     * @param clnt error handler (non-null)
     * @return username/password pairs as a properties object.
     **/
    private Properties getPWFile(Requester clnt)
    {
        Project project = clnt.getProject();
        AntX.verify_(project!=null, "PasswordValueURIHandler",
                     "valueFrom- called from project");

        String generated = Iteration.valueURIHandler("artifact").valueFrom
            ("passwordfile","$artifact:passwordfile",clnt);
        if (generated!=null) {
            File f = new File(TransformHelper.apply
                (ValueTransform.OSPATH,generated, project));
            return readFile(f,!OPTIONAL,clnt);
        }

        Properties passwords = FixtureExaminer.getReferencedProperties
            (project, "build.passwords", null);
        if (passwords!=null) {
            return passwords;
        }
        Object o = project.getReference("build.passwords");
        if (o!=null) {
            clnt.problem("Unrecognized data under 'build.passwords': "+
                o.getClass().getName(), Project.MSG_WARN);
        }

        String filename = Iteration.defaultdefaults().passwordFileLocation(project);
        if (filename!=null) {
            File f = project.resolveFile(filename);
            return readFile(f,!OPTIONAL,clnt);
        }

        filename = Tk.getTheProperty(project,"user.home");
        File f = new File(filename);
        f = new File(f,".ant"+File.separator+"build.passwords");
        return readFile(f,OPTIONAL,clnt);
    }



    /**
     * Tries to extract the named password from the current iteration's
     * password file. The password file is loaded everytime a uri is
     * evaluated (yick).
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        String value = null;
        Properties pwfile = getPWFile(clnt);
        if (pwfile!=null) {
            String owner = Tk.resolveString(clnt.getProject(),uriFragment,true);
            value = pwfile.getProperty(owner);
            pwfile.clear();
        }
        if (value==null) {
            ValueURIHandler helper = Iteration.valueURIHandler("default");
            value = helper.valueFrom("password","$default:password",clnt);
        }
        return value;
    }
}

/* end-of-PasswordValueURIHandler.java */