/**
 * $Id: ManageCollectionValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.info;

import  java.util.Iterator;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Starter class for handlers that manipulate common collection types like lists and
 * maps. Provides support for a common set of operation selectors and uri fragment
 * parsing.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 **/

abstract class ManageCollectionValueURIHandler extends ValueURIHandlerSkeleton
{
    static final int UNKNOWN=-1;
    static final int SIZE=UNKNOWN+1;
    static final int KEYSET=SIZE+1;
    static final int VALUES=KEYSET+1;
    static final int LOOKUP=VALUES+1;
    static final int DUMP=LOOKUP+1;


    /**
     * Initializes a new collection examiner value uri handler.
     **/
    protected ManageCollectionValueURIHandler()
    {
        super();
    }


    /**
     * Returns the default operation if one is not set explicitly.
     * Will use "DUMP" by default.
     **/
    int getDefaultOp()
    {
        return DUMP;
    }



    /**
     * Extracts the bit of collection information requested. Currently supports
     * four special operations: dump (also default if no fragment), keyset,
     * values, and size.
     */
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        final Project P = clnt.getProject();
            
        String refid = uriFragment;
        int op = getDefaultOp();
        String key = null;
        String delim = null;

        int i = uriFragment.lastIndexOf("?");
        if (i>=0) {
            refid = uriFragment.substring(0,i++);
            if (i<uriFragment.length()) {
                String instr = Tk.resolveString(P,uriFragment.substring(i),true);
                int j = instr.lastIndexOf(",,");
                if (j>=0) {
                    delim = instr.substring(j+2);
                    instr = instr.substring(0,j);
                    if ("\\n".equals(delim)) {
                        delim= Strings.NL;
                    }
                }
                op = opFrom(instr);
                if (op==LOOKUP) {
                    key = instr;
                }
            }
        }
        refid = Tk.resolveString(P,refid,true);
        return valueFromTyped(refid,op,delim,key,clnt);
    }



    /**
     * Does query work that is specific to the particular collection or map that
     * is being handled.
     * @param refid extracted (normalized) reference id (or inline collection)
     * @param op the determined operation from original URI fragment
     * @param delim the client-supplied delimiter or <i>null</i> if none in URI
     * @param key the client-supplied "key" or <i>null</i> if none in URI
     * @param clnt call controls (non-null
     * @return the handler's results or <i>null</i> if cannot determine.
     **/
    abstract String valueFromTyped(String refid, int op, String delim, String key,
        Requester clnt);



    /**
     * Determine the requested collection operation for the handler.
     * @param opstring incoming operation or key value (non-null)
     * @return operation id
     **/
    int opFrom(String opstring)
    {
        if ("size".equals(opstring)) {
            return SIZE;
        }
        if ("keys".equals(opstring)) {
            return KEYSET;
        }
        if ("values".equals(opstring)) {
            return VALUES;
        }
        if ("dump".equals(opstring)) {
            return DUMP;
        }
        return opstring.length()>0 ? LOOKUP : getDefaultOp();
    }



    /**
     * Generates a listing string of the items returned by 
     * the given iterator. The listing elements are delimited 
     * by the incoming marker string.
     * @param itr iterator that returns listing elements (non-null)
     * @param delim delimiter used between elements (non-null) 
     * @return listing string (never <i>null</i>)
     **/
    String listFrom(Iterator itr, String delim)
    {
        StringBuffer sb = new StringBuffer(100);
        int i=0;
        while (itr.hasNext()) {
            if (i>0) {
                sb.append(delim);
            }
            sb.append(itr.next());
            i++;
        }
        return sb.toString();
    }
}

/* end-of-ManageCollectionValueURIHandler.java */