/**
 * $Id: ListFriendlyValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.valueuri.info;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.FixtureExaminer;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ListFriendly;

/**
 * Value URI handler that lets you extract information from a list friendly data
 * object. This handler is not installed automatically by the AntX runtime, you must
 * install it explicitly like the example below. List value URIs are often linked to the
 * <span class="src">$list:</span> scheme. The general form of the URI: 
 * <span class="src"><nobr>$list:listrefid|comma-delimited-list[?[values|size|dump|<i>index</i>][,,delim]]</nobr></span>.
 * The delim option is only relevant to the values operation.
 * <p/>
 * This handler assumes that the named item is a reference to a list-friendly
 * data object unless the name is surrounded by square brackets or no such reference
 * exists. If not a reference, this handler treats the name as an inlined comma-
 * delimited list(as would be the case for a list stored in a property value).
 * <p/>
 * You can use this handler to extract a particular string from the targetted
 * list. Set the URI's query string (the bit after the "&#63;") to the item's 
 * 0-based index; for example: <span class="src">$list:javadoc.servers?3</span>.
 * You can also extract the size of a list by setting the query string to 
 * "<span class="src">size</span>" like: <span class="src">$list:logs?size</span>.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;urls id="javadoc.servers" prefix="http://" suffix="/apis/packagelist"&gt;
 *       &lt;url value="antxtras.org/thirdparty/junit"/&gt;
 *       &lt;url value="antxtras.org/stable/antunit/docs"/&gt;
 *       ...
 *    &lt;/urls&gt;
 *    ...
 *    &lt;echo message="There are ${<b>$list:</b>javadoc.servers?size}" servers"/&gt;
 *    &lt;foreach i="server_" list="${<b>$list:</b>javadoc.servers}".../&gt;
 * 
 *    &lt;macrodef name="run-programmertests"&gt;
 *       &lt;attribute name="jvm.args" default=""/&gt;
 *       ...
 *       &lt;do true="${$isdefined:@{jvm.args}}"&gt;
 *          &lt;foreach i="arg" in="0,${<b>$list:</b>@{jvm.args}?size}" mode="local"&gt;
 *             &lt;altertask name="myjunit" resolveproperties="yes"&gt;
 *                &lt;jvmarg value="${<b>$list:</b>@{jvm.args}?@(arg)}"/&gt;
 *             &lt;/altertask&gt;
 *          &lt;/foreach&gt;
 *       &lt;/do&gt;
 *
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="list"
 *             value="com.idaremedia.antx.valueuri.info.ListFriendlyValueURIHandler"/&gt;
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    api,helper
 **/

public final class ListFriendlyValueURIHandler extends ManageCollectionValueURIHandler
{
    /**
     * Initializes a new list examiner URI handler.
     **/
    public ListFriendlyValueURIHandler()
    {
        super();
    }


    /**
     * Returns "VALUES" always.
     */
    int getDefaultOp()
    {
        return VALUES;
    }


    /**
     * Extracts the bit of list information requested. Currently supports
     * three special operations: dump, values (also default if no fragment),
     * and size. Any other query string is considered the lookup index.
     **/
    String valueFromTyped(String refid, int op, String delim, String key, 
        Requester clnt)
    {
        final Project P = clnt.getProject();

        Object o;
        if (refid.startsWith("[") && refid.endsWith("]")) {
            refid = refid.substring(1,refid.length()-1);
            o = null;
        } else {
            o = FixtureExaminer.trueReference(P,refid);
        }
        if (o instanceof ListFriendly) {
            ListFriendly l = (ListFriendly)o;
            switch(op) {
                case SIZE: {
                    return String.valueOf(l.size());
                }
                case VALUES:
                case KEYSET: {
                    if (delim!=null) {
                        return listFrom(l.readonlyStringIterator(P),delim);
                    }
                    return l.stringFrom(P);
                }
                case LOOKUP: {
                    int i = Tk.integerFrom(key,-1);
                    if (i>=0) {
                        Iterator itr= l.readonlyStringIterator(P);
                         int n=0;
                         while (itr.hasNext()) {
                             String s = itr.next().toString();
                             if (n==i) {
                                 return s;
                             }
                             n++;
                         }
                    }
                    break;
                }
                case DUMP: {
                    return String.valueOf(o);
                }
            }
        } else if (o==null || (o instanceof List)) {
            List l = o==null ? Tk.splitList(refid.trim()) : (List)o;
            switch(op) {
                case SIZE: {
                    return String.valueOf(l.size());
                }
                case LOOKUP: {
                    int i = Tk.integerFrom(key,-1);
                    if (i>=0 && i<l.size()) {
                        return (String)l.get(i);
                    }
                    break;
                }
                case VALUES:
                case KEYSET: {
                    if (delim==null) {
                        delim = AntX.DEFAULT_DELIMITER;
                    }
                    return listFrom(l.iterator(),delim);
                }
                default: {
                    return String.valueOf(l);
                }
            }
        }
        return null;
    }
}

/* end-of-ListFriendlyValueURIHandler.java */