/**
 * $Id: TaskSet.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2002-2004,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.RuntimeConfigurable;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.TaskContainer;
import  org.apache.tools.ant.UnknownElement;
import  org.apache.tools.ant.util.CollectionUtils;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.helpers.Empties;
import  com.idaremedia.antx.helpers.TaskHandle;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;

/**
 * Base implementation of AntX task containers. Implements basic support for the
 * following behavior:<ul>
 *   <li>Unmasking of "UnknownElements" proxies for true AntX tasks.</li>
 *   <li>Delay configuration of nested tasks.</li>
 *   <li>Filtering of nested tasks (by default allows anything).</li>
 *   <li>Template method(s) for nested task execution.</li>
 * </ul>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  single
 * @.group   impl,infra
 * @see      StrictInnerTask
 * @see      StrictOuterTask
 **/

public class TaskSet extends AssertableTask implements TaskContainer
{
    /**
     * Default setting of the delay-configure behavior of this
     * container (<i>true</i>).
     **/
    public static final boolean DEFAULT_DELAY_CONFIGURE= true;


    /**
     * Used to mark the number of tasks to be added as "unknown".
     * @since JWare/AntXtras 0.5.1
     */
    private static final int INFINITE_TASKS = Integer.MAX_VALUE;



    /**
     * Initializes a new empty taskset.
     **/
    public TaskSet()
    {
        super(AntX.starters);
    }


    /**
     * Initializes a new CV-labeled empty taskset.
     * @param iam CV-label (non-null)
     **/
    protected TaskSet(String iam)
    {
        super(iam);
    }



    /**
     * Empties this taskset's current list of tasks. This method
     * is only triggered if a taskset subclass implements the
     * <span class="src">Reconfigurable</span> interface.
     * @since JWare/AntX 0.5
     */
    public void unconfigure()
    {
        super.unconfigure();
        getTasksList().clear();
    }



    /**
     * Initializes new configuration-delaying empty taskset.
     * @param iam CV-label (non-null)
     * @param delayConfigure <i>true</i> if nested tasks configured
     *        when they're actually executed
     **/
    protected TaskSet(String iam, boolean delayConfigure)
    {
        super(iam);
        delayChildrenConfigure(delayConfigure);
    }



    /**
     * Explicitly tells this taskset whether to delay configuration
     * of nested tasks until they're executed.
     * @param delay <i>true</i> if configuration delayed
     **/
    protected final void delayChildrenConfigure(boolean delay)
    {
        m_delayConfigure = delay;
    }



    /**
     * Returns <i>true</i> if this taskset prefers to delay child
     * configuration until they're executed(performed).
     **/
    protected final boolean isDelayChildrenConfigure()
    {
        return m_delayConfigure;
    }



    /**
     * Optionally delays configuration of nested tasks until they're
     * executed (like standard '<i>sequential</i>' task).
     * @see TaskExaminer#ignoreConfigure TaskExaminer.ignoreConfigure(&#8230;)
     **/
    public void maybeConfigure()
        throws BuildException
    {
        if (!TaskExaminer.ignoreConfigure(this)) {//@since AntX0.4+Ant1.6
            if (isInvalid() || !isDelayChildrenConfigure()) {
                super.maybeConfigure();
            } else {
                getRuntimeConfigurableWrapper().maybeConfigure(getProject(),false);
                if (AntX.ANT_PRE17) {//order: children,configure
                    maybeConfigureSpecialTasks();
                }
            }
        }
    }


    /**
     * Half of workaround for the switch in ordering of the UnknownElement's
     * maybeConfigure processing since Ant 1&#46;7.
     * @since JWare/AntX 0.5.1
     */
    public void setRuntimeConfigurableWrapper(RuntimeConfigurable wrapper)
    {
        if (!AntX.ANT_PRE17) {//Ant 1.7 hack: try calc how many calls to 'addTask' possible!
            if (wrapper!=null) {
                int n=0;
                for (Iterator itr=CollectionUtils.asIterator(wrapper.getChildren());itr.hasNext();) {
                    itr.next();
                    n++;
                }
                if (n>0) {
                    this.m_configureThreshold = n;
                }
            }
        }
        super.setRuntimeConfigurableWrapper(wrapper);
    }



    /**
     * Called by {@linkplain #maybeConfigure maybeConfigure} if
     * configuration of nested tasks has been delayed. This allows
     * subclasses to configure special nested tasks that <em>should</em>
     * be configured at the same time as this especially if these
     * tasks are not included in the list of tasks returned by
     * {@linkplain #getTasks getTasks}
     **/
    protected void maybeConfigureSpecialTasks()
    {
        //no-op for this class.
    }



    /**
     * Returns this taskset's list of nested tasks.
     **/
    protected List getTasksList()
    {
        return m_nestedTasks;
    }



    /**
     * Returns <i>true</i> if this taskset considers itself empty.
     * @since JWare/AntX 0.5
     **/
    protected boolean isEmpty()
    {
        return getTasksList().isEmpty();
    }



    /**
     * Returns <i>true</i> if task inside handle is allowed to nest within
     * this taskset. Returns <i>true</i> by default; all tasks are allowed.
     * This method can also switch the task to be included by this taskset
     * by updating the incoming handle. Warning: the task should never be
     * set to <i>null</i> if this method returns <i>true</i>.
     * @param taskH the task to be nested (as handle)
     * @see #candidateTask candidateTask(TaskHandle,Class[])
     **/
    protected boolean includeTask(TaskHandle taskH)
    {
        return true;
    }



    /**
     * Adds another task if allowed.
     * @param task task to be added
     * @throws BuildException if task cannot be nested here
     * @see TaskExaminer
     **/
    public void addTask(Task task)
    {
        require_(task!=null,"addTsk- nonzro task");

        TaskHandle taskH = new TaskHandle(task);

        if (includeTask(taskH)) {
            verify_(!taskH.isEmpty(),"addTsk- nonzro task");
            getTasksList().add(taskH.getTask());
        } else {
            String ermsg = uistrs().get("taskset.nested.task.disallowed",
                                    task.getTaskName(), this.getTaskName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        addedTask();
    }



    /**
     * Called after a task has been added to this task set's underlying
     * set. For Ant 1&#46;7 or later ensures the task set's special tasks (if any)
     * are properly configured before others. Only used if Ant 1&#46;7 or later.
     * @since JWare/AntXtras 0.5.1
     **/
    protected final void addedTask()
    {
        if (!AntX.ANT_PRE17) {
            m_configureThreshold--;
            if (m_configureThreshold==0) {
                finishedAddingTasks();
            }
        }
    }
 


    /**
     * If Ant 1&#46;7 or later runtime, this method is called after this task 
     * set's children have all been added to esure the task set's special tasks
     * (if any)  are properly configured before others. 
     * Only used if Ant 1&#46;7 or later.
     * @since JWare/AntXtras 0.5.1
     **/
    private void finishedAddingTasks()
    {
        m_configureThreshold = INFINITE_TASKS;
        maybeConfigureSpecialTasks();
    }



    /**
     * An {@linkplain #includeTask includeTask()} helper that ensures
     * an Ant 1.6+ placeholder task is replaced if AntX's strict script
     * loading property is turned on. The handle should be updated with
     * the true task.
     * @param taskH the task to be examined (non-null)
     * @param coi classes of interest (non-null)
     * @return the task to examine (might be flipped, might not)
     * @see TaskExaminer#candidateTask TaskExaminer.candidateTask()
     * @since JWare/AntX 0.4
     **/
    protected final Task candidateTask(TaskHandle taskH, Class[] coi)
    {
        return TaskExaminer.candidateTask(taskH,coi,this);
    }



    /**
     * Returns a new array of this taskset's nested tasks. The array
     * is new, the tasks are live references.
     **/
    public Task[] getTasks()
    {
        return (Task[])getTasksList().toArray(Empties.EMPTY_TASK_ARRAY);
    }



    /**
     * Runs this task's nested tasks in order. Each task's '<i>perform</i>'
     * method is called.
     **/
    protected final void performTheTasksList() throws BuildException
    {
        Iterator itr = getTasksList().listIterator();
        while (itr.hasNext()) {
            Task nestedTask = (Task)itr.next();
            nestedTask.perform();
        }
    }



    /**
     * Runs a <em>new copy</em> of this task's nested tasks in order in 
     * targetted project. Each new task instance's 
     * <span class="src">perform</span> method is called. Expects this 
     * taskset's internals have not been resolved and contain the original
     * proxies installed by Ant parsing process.
     * @see TaskExaminer#copyUEProxy  TaskExaminer.copyUEProxy(&#8230;)
     * @throws BuildException if any nested task does.
     * @since JWare/AntX 0.5
     * @.safety multiple if tasklist is frozen.
     **/
    protected final void performIterationOfTheTasksList(final Project theProject,
        final Target parent) throws BuildException
    {
        require_(theProject!=null,"preformIterOfItems- nonzro project");
        require_(parent!=null,"preformIterOfItems- nonzro parent target");
        
        Iterator itr = getTasksList().listIterator();

        while (itr.hasNext()) {
            UnknownElement template = (UnknownElement)itr.next();
            UnknownElement instance = TaskExaminer.copyUEProxy(template,theProject,parent);
            instance.perform();
            instance = null;
        }
    }



    /**
     * Runs a <em>new copy</em> of this task's nested tasks in order.
     * Like {@linkplain #performIterationOfTheTasksList(Project,Target)
     * performIterationOfTheTasksList(&#8230;)} but for this taskset's
     * own project.
     * @since JWare/AntX 0.5
     **/
    protected final void performIterationOfTheTasksList() throws BuildException
    {
        performIterationOfTheTasksList(getProject(),getOwningTarget());
    }



    /**
     * Runs nested tasks in order. Each task's '<i>perform</i>' method
     * is called. Subclasses can override this method to decorate the
     * central {@linkplain #performTheTasksList} method which actually
     * performs each task in this container.
     * @throws BuildException if any nested task does
     **/
    protected void performNestedTasks() throws BuildException
    {
        performTheTasksList();
    }



    /**
     * Same as {@linkplain #performNestedTasks performNestedTasks} after
     * execution verification.
     * @throws BuildException if unable to execute for any reason
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");

        performNestedTasks();
    }



    /**
     * Helper that ensures one and only one of a particular strict nested
     * task is defined for this taskset at any time.
     * @throws BuildException if current task is defined
     **/
    protected final void verifyIsUndefined_(Task currentTask, Task newTask)
        throws BuildException
    {
        if (currentTask!=null) {
            String ermsg = uistrs().get("taskset.only.one.specialtask",
                                        newTask.getTaskName(),
                                        currentTask.getTaskName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    private boolean m_delayConfigure= DEFAULT_DELAY_CONFIGURE;
    private List m_nestedTasks= AntXFixture.newList(10);//NB: only access thru getTasksList()!
    private int m_configureThreshold= INFINITE_TASKS;
}

/* end-of-TaskSet.java */
