/**
 * $Id: EchoThingTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.starters;

import  java.io.ByteArrayOutputStream;
import  java.io.File;
import  java.io.FileOutputStream;
import  java.io.IOException;
import  java.io.OutputStream;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.parameters.InformationSaver;

/**
 * Skeleton for a helper task that displays some internal single AntX variable or
 * single data type without using the extended AntX display registry mechanism. Note
 * that starting with AntX 0.4, {@linkplain com.idaremedia.antx.print.DisplayStrategy
 * DisplayStrategies} are the preferred display mechanism for AntX types. If an
 * Echo*Task is written, it must use the type's display strategy implementation
 * to ensure it remains in-sync.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public abstract class EchoThingTask extends MsgTask
    implements InformationSaver
{
    /**
     * Platform-specific newline.
     **/
    protected static final String NL= Strings.NL;



    /**
     * Initializes new CV-labeled EchoThingTask.
     * @param iam CV-label (non-null)
     **/
    protected EchoThingTask(String iam)
    {
        super(iam);
    }

// ---------------------------------------------------------------------------------------
// Public build parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets the target file path for this echo task's generated
     * output. If the path is relative, the output file will be
     * determined relative to this task's project's base directory.
     * @param outputFile the output file's path (non-null)
     **/
    public void setToFile(String outputFile)
    {
        require_(outputFile!=null,"setToFil- nonzro file");
        m_toFile= getProject().resolveFile(outputFile);
    }



    /**
     * Sets the target file for this echo task's generated output.
     * The file reference is used as-is.
     * @param outputFile the output file (non-null)
     * @since JWare/AntX 0.4
     **/
    public final void setToProjectFile(File outputFile)
    {
        require_(outputFile!=null,"setToProjFile- nonzro file");
        m_toFile = outputFile;
    }


    /**
     * Returns this task's target file for output. Returns
     * <i>null</i> if never set.
     * @since JWare/AntX 0.4
     **/
    public final String getToFilePath()
    {
        File f = getToFile();
        return f!=null ? f.getPath() : null;
    }


    /**
     * Returns this task's target file for output. Returns
     * <i>null</i> if never set.
     **/
    public File getToFile()
    {
        return m_toFile;
    }


    /**
     * Marks whether this task should try to append its new information
     * to its output stream.
     * @param append <i>true</i> if should try to append
     **/
    public void setAppend(boolean append)
    {
        m_tryAppend= append;
    }


    /**
     * Returns <i>true</i> if this task will try to append any new
     * information to an existing output sink. By default is <i>false</i>.
     **/
    public final boolean willTryAppend()
    {
        return m_tryAppend;
    }

// ---------------------------------------------------------------------------------------
// Implementation helpers:
// ---------------------------------------------------------------------------------------

    /**
     * Initializes the reference id of this task's thing.
     **/
    protected final void setThingRefId(String refId)
    {
        require_(refId!=null,"setrefId- nonzro refid");
        m_refId = refId;
    }


    /**
     * Returns this task's thing reference id; can be <i>null</i>
     * if never set. Required for task to execute properly.
     **/
    protected final String getThingRefId()
    {
        return m_refId;
    }


    /**
     * Returns the parameter name of this task's thing refid.
     * Used in diagnostic messages.
     * @since JWare/AntX 0.4
     **/
    protected String getThingRefParameterName()
    {
        return "refid";//backward compatible
    }


    /**
     * Returns the actual thing instance echoed by this task. Never
     * returns <i>null</i>.
     * @param ofKind thing's required class (non-null)
     * @param msgid id of error message used if reference invalid
     * @throws BuildException if refid undefined or not a a valid thing
     **/
    protected Object getReferencedThing(Class ofKind, String msgid)
        throws BuildException
    {
        require_(ofKind!=null,"getRefTing- nonzro clas");
        String refid = getThingRefId();
        if (msgid==null) {
            return getReferencedObject(null,refid,ofKind);
        }
        Object object = getProject().getReference(refid);
        if (!ofKind.isInstance(object)) {
            String ermsg = uistrs().get(msgid,refid);
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        return object;
    }


    /**
     * Ensures we're in a valid target/project and have a defined
     * thing reference id. Subclasses must ensure the referenced thing
     * is of the expected type.
     * @see #getReferencedThing
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (getThingRefId()==null) {
            String ermsg = uistrs().get("task.needs.this.attr",
                                        getTaskName(),getThingRefParameterName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }
    }


    /**
     * Returns an output stream suitable for this task. The returned
     * stream is the raw output stream, <em>not</em> a buffered stream.
     **/
    protected OutputStream getOutputStream()
    {
        File f = getToFile();
        if (f!=null) {
            try {
                return new FileOutputStream(f.getPath(), willTryAppend());
            } catch(IOException ioX) {
                String ermsg = uistrs().get("task.echo.unable.use.file",f.getPath(),
                                            ioX.getMessage());
                log(ermsg,Project.MSG_WARN);
            }
        }
        return new ByteArrayOutputStream(1024);
    }


    /**
     * Returns <i>true</i> if this task should try to echo the given
     * output stream to the Ant runtime's logging system.
     **/
    protected boolean tryAntLog(OutputStream os)
    {
        return (os instanceof ByteArrayOutputStream);
    }


    /**
     * Returns the information from the given stream to log to the
     * Ant logging system. By default if the stream is a byte buffer,
     * returns the entire contents; otherwise returns the empty string.
     **/
    protected String getAntLogString(OutputStream os)
    {
        if (os instanceof ByteArrayOutputStream) {
            return ((ByteArrayOutputStream)os).toString();
        }
        return "";
    }


    /**
     * Echoes this task's referenced thing iff execution conditions
     * have been met. Subclass should implement the {@linkplain #echoThing
     * echoThing} method.
     * @throws BuildException if invalid refid or a file I/O
     *         error occurs.
     * @see #echoThing
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");

        if (testIfCondition() && testUnlessCondition()) {
            echoThing();
        }
    }


    /**
     * Actual task echoing work; subclass must provide.
     * @throws BuildException if invalid refid or a file I/O
     *         error occurs.
     **/
    protected abstract void echoThing() throws BuildException;



    private String m_refId;
    private File m_toFile;
    private boolean m_tryAppend;//NB:false=>overwrite existin'
}

/* end-of-EchoThingTask.java */
