/**
 * $Id: Modification.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.EnumSkeleton;

/**
 * Enumeration of variable modifications. Exposed as standalone class since reused by
 * multiple AntX tasks and conditions. The following list explains what the various
 * settings mean:<ul>
 *    <li><span class="src">+s</span>: The value (a string) is concatenated to the 
 *         current value of the variable (also a string).</li>
 *    <li><span class="src">len</span> or <span class="src">=s</span>: The variable is set 
 *         to the source's (a string)length.</li>
 * </ul>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 **/

public class Modification extends EnumSkeleton
{
    //@.impl ORDERING of static declarations is important! ------

    //1) Indices (in order)

    /** Index of {@linkplain #SET SET}. **/
    public static final int SET_INDEX= 0;
    /** Index of {@linkplain #INC INC}. **/
    public static final int INC_INDEX= SET_INDEX+1;
    /** Index of {@linkplain #DEC DEC}. **/
    public static final int DEC_INDEX= INC_INDEX+1;
    /** Index of {@linkplain #INC_BY1 INC_BY1}. **/
    public static final int INC_BY1_INDEX= DEC_INDEX+1;
    /** Index of {@linkplain #DEC_BY1 DEC_BY1}. **/
    public static final int DEC_BY1_INDEX= INC_BY1_INDEX+1;
    /** Index of {@linkplain #NOW NOW}. **/
    public static final int NOW_INDEX= DEC_BY1_INDEX+1;
    /** Index of {@linkplain #MINUS_NOW MINUS_NOW}. **/
    public static final int MINUS_NOW_INDEX= NOW_INDEX+1;

    /** Index of {@linkplain #STRCAT STRCAT}.
     *  @since JWare/AntX 0.5 **/
     public static final int STRCAT_INDEX= MINUS_NOW_INDEX+1;
     
    /** Index of {@linkplain #STRLEN STRLEN}.
     *  @since JWare/AntX 0.5 **/
     public static final int STRLEN_INDEX= STRCAT_INDEX+1;



    /** The number of base modifications. Subclasses need.
     *  @since JWare/AntX 0.4
     **/
    protected static final int BASE_VALUE_COUNT= STRLEN_INDEX+1;


    //2) Values (in order)

    /** Values in same order as public indices. **/
    private static final String[] VALUES_= new String[] {
        "=", "+", "-", "++", "--", "now", "-now", "+s", "len"
    };


    //3) Singletons (depend on Indices and Values already existing!)

    /** Singleton "set" operation. **/
    public static final Modification SET =
        new Modification(VALUES_[SET_INDEX],SET_INDEX);
    /** Singleton "increment" operation. **/
    public static final Modification INC =
        new Modification(VALUES_[INC_INDEX],INC_INDEX);
    /** Singleton "decrement" operation. **/
    public static final Modification DEC =
        new Modification(VALUES_[DEC_INDEX],DEC_INDEX);
    /** Singleton "increment-by-1" operation. **/
    public static final Modification INC_BY1 =
        new Modification(VALUES_[INC_BY1_INDEX],INC_BY1_INDEX);
    /** Singleton "decrement-by-1" operation. **/
    public static final Modification DEC_BY1 =
        new Modification(VALUES_[DEC_BY1_INDEX],DEC_BY1_INDEX);
    /** Singleton "now" operation. **/
    public static final Modification NOW =
        new Modification(VALUES_[NOW_INDEX],NOW_INDEX);
    /** Singleton "minus-now" operation. **/
    public static final Modification MINUS_NOW =
        new Modification(VALUES_[MINUS_NOW_INDEX],MINUS_NOW_INDEX);

    /** Singleton "strcat" operation.
     *  @since JWare/AntX 0.5 **/
    public static final Modification STRCAT =
        new Modification(VALUES_[STRCAT_INDEX],STRCAT_INDEX);

    /** Singleton "strlen" operation.
     *  @since JWare/AntX 0.5 **/
    public static final Modification STRLEN =
        new Modification(VALUES_[STRLEN_INDEX],STRLEN_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public Modification()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensure it's initialized
     * as with default Ant Introspector helper thingy.
     **/
    private Modification(String v, int i)
    {
        super(v);
    }



    /**
     * Returns a <em>copy</em> of the standard modification
     * operators in order. Subclasses can use this method to
     * pre-fill their value arrays with the inherited list.
     * @param fillin [optional] array of strings to update with values.
     **/
    public static String[] copyOfDefaultValues(String[] fillin)
    {
        if (fillin==null) {
            fillin = new String[VALUES_.length];
        }
        System.arraycopy(VALUES_,0,fillin,0,VALUES_.length);
        return fillin;
    }



    /**
     * Returns copy of all possible modification values as an ordered
     * string array. Note: ordering should be same as singletons indice.
     **/
    public String[] getValues()
    {
        return Modification.copyOfDefaultValues(null);
    };



    /**
     * Helper that converts a scalar to a known modification.
     * Returns <i>null</i> if value does not match any of five
     * expected modifications.
     * @param i the index to be matched
     **/
    public static Modification from(int i)
    {
        switch(i) {
            case SET_INDEX: return SET;
            case INC_INDEX: return INC;
            case DEC_INDEX: return DEC;
            case INC_BY1_INDEX: return INC_BY1;
            case DEC_BY1_INDEX: return DEC_BY1;
            case NOW_INDEX: return NOW;
            case MINUS_NOW_INDEX: return MINUS_NOW;
            case STRCAT_INDEX: return STRCAT;
            case STRLEN_INDEX: return STRLEN;
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known modification
     * index.
     * @param i the index to be matched
     * @param dfltMod the default modification if necessary
     **/
    public static Modification from(int i, Modification dfltMod)
    {
        Modification mod= from(i);
        return (mod==null) ? dfltMod : mod;
    }


    /**
     * Helper that converts a string to a known modification
     * singleton. Returns <i>null</i> if string unrecognized.
     * String can be either modification's symbolic name or
     * its index.
     **/
    public static Modification from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if ("default".equals(s))       { return getAntXDefault(); }
                if (SET.value.equals(s))       { return SET; }
                if (INC.value.equals(s))       { return INC; }
                if (DEC.value.equals(s))       { return DEC; }
                if (INC_BY1.value.equals(s))   { return INC_BY1; }
                if (DEC_BY1.value.equals(s))   { return DEC_BY1; }
                if (NOW.value.equals(s))       { return NOW; }
                if (MINUS_NOW.value.equals(s)) { return MINUS_NOW; }
                if (STRCAT.value.equals(s))    { return STRCAT; }
                if ("strcat".equals(s))        { return STRCAT; }
                if (STRLEN.value.equals(s))    { return STRLEN; }
                if ("=s".equals(s))            { return STRLEN; }
                if ("strlen".equals(s))        { return STRLEN; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(java.lang.String) from(String)} but
     * with a default value if value does not match any known
     * modification name.
     * @param s the symbolic name to be matched
     * @param dfltMod the default modification if necessary
     **/
    public static Modification from(String s, Modification dfltMod)
    {
        Modification mod= from(s);
        return (mod==null) ? dfltMod : mod;
    }


    /**
     * Returns the <i>AntX</i> default modification (SET).
     **/
    public static final Modification getAntXDefault()
    {
        return Modification.SET;
    }
}

/* end-of-Modification.java */
