/**
 * $Id: ExportTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.text.DecimalFormat;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.ExportScope;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.TransformEnabled;
import  com.idaremedia.antx.parameters.TransformHelper;
import  com.idaremedia.antx.parameters.ValueTransform;

/**
 * Helper task that manipulates a {@linkplain com.idaremedia.antx.ExportedProperties
 * modifiable property}. Intended for unit tests and other controlled build environments.
 * When combined with assertions can make test build files almost auto-testing.
 * <p>
 * <b>Example Usage:</b><pre>
 *  &lt;assign var="loopcount" value="0"/&gt;
 *  &lt;assign var="loopcount" op="++"/&gt;
 *  &lt;assign var="__bn" fromproperty="build.number"/&gt;
 *  &lt;assign var="num.changes" op="+" value="${diff.count}"/&gt;
 *  &lt;assign var="num.changes" copyproperty="__loop.num.changes"/&gt;
 *  &lt;assign var="time" op="now"/&gt;
 *  &lt;assign var="duration" op="-now" fromvariable="time"/&gt;
 *  &lt;assign var="time" op="-now"/&gt;
 *  &lt;assign var="totaltime" op="+" fromvariable="duration"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      ImportTask
 **/

public class ExportTask extends AssertableTask
    implements TransformEnabled, AntLibFriendly
{
    private static final String SRC_ATTRS = "value|fromproperty|fromvariable";

// ---------------------------------------------------------------------------------------
// Construction:
// ---------------------------------------------------------------------------------------

    /**
     * Initializes a new export task instance.
     **/
    public ExportTask()
    {
        super(AntX.nopackage);
    }


    /**
     * Initializes a new CV-labeled export task instance.
     **/
    public ExportTask(String iam)
    {
        super(iam);
    }


    /**
     * Initializes this task's project.
     **/
    public void setProject(Project project)
    {
        super.setProject(project);
        m_value.setProject(project);
    }


// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets this export task's effective scope. Value must be one
     * of "<i>project</i>","<i>thread</i>", or "<i>all</i>".  Defaults
     * to "<i>thread</i>".
     * @since JWare/AntX 0.2
     **/
    public void setScope(ExportScope scope)
    {
        require_(scope!=null,"setScop- nonzro");
        m_scope = scope;
    }


    /**
     * Returns this export task's effective scope. Never returns
     * <i>null</i>; defaults to "<i>thread</i>".
     * @since JWare/AntX 0.2
     **/
    public final ExportScope getScope()
    {
        return m_scope;
    }


    /**
     * Sets this export task's modifiable property name.
     * @param name property's name (non-null)
     **/
    public void setName(String name)
    {
        require_(!Tk.isWhitespace(name),"setNam- nonwspc name");
        m_name = name;
    }


    /**
     * Synonym for {@linkplain #setName setName}.
     **/
    public final void setVar(String name)
    {
        setName(name);
    }


    /**
     * Returns this export task's property's name. Will return
     * <i>null</i> if never set.
     **/
    public final String getName()
    {
        return m_name;
    }


    /**
     * Sets this export task's new value as a literal as-is string.
     * @param value new value (non-null)
     * @see #setOp
     **/
    public void setValue(String value)
    {
        require_(value!=null,"setValu- nonzro valu");
        m_value.set(value);
        m_value.setIsLiteral();
    }


    /**
     * Sets this export task's new value from a property's value.
     * @param property property's name (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setFromProperty(String property)
    {
        require_(property!=null,"setFromProp- nonzro name");
        m_value.set(property);
        m_value.setIsProperty(true);
    }


    /**
     * Sets this export task's new value from another variable's
     * value. The source variable is assumed in the thread-scope.
     * @param variable variable's name (non-null)
     * @since JWare/AntX 0.3
     **/
    public void setFromVariable(String variable)
    {
        require_(variable!=null,"setFromVar- nonzro name");
        m_value.set(variable);
        m_value.setIsExported(true);
    }


    /**
     * Synonym for {@linkplain #setFromVariable setFromVariable}.
     * @since JWare/AntX 0.4
     **/
    public final void setFromVar(String variable)
    {
        setFromVariable(variable);
    }


    /**
     * Returns this export task's source value. Can return <i>null</i>
     * if this task's source was never defined or source is missing.
     * The returned value represents a literal ({@linkplain #setValue
     * value}), a property's value ({@linkplain #setFromProperty
     * fromproperty}), or another variable's value ({@linkplain
     * #setFromVariable fromvariable}).
     **/
    public final String getValue()
    {
        return m_value.getValue();
    }


    /**
     * Changes this export task's default <i>assign</i> operation.
     * @param op required modification (non-null)
     **/
    public final void setOp(Modification op)
    {
        require_(op!=null,"setModf- nonzro op");
        m_Op= op;
    }


    /**
     * Returns this export task's modification operation. Never
     * returns <i>null</i>. Defaults to simple assignment.
     **/
    public final Modification getOp()
    {
        return m_Op;
    }


    /**
     * Sets the value transform for the value copied to
     * the variable.
     * @since JWare/AntX 0.4
     **/
    public final void setTransform(ValueTransform t)
    {
        m_T = t==null ? null : ValueTransform.from(t.getIndex());//normalize
    }


    /**
     * Returns the value transformation this task will
     * perform on the to-be-copied value. Will return
     * {@linkplain ValueTransform#NONE} if never set.
     * @since JWare/AntX 0.4
     **/
    public final ValueTransform getTransform()
    {
        return m_T==null ? ValueTransform.NONE : m_T;
    }



    /**
     * Sets a name for a regular property to be updated with new
     * exported property's value. This update is done <em>in
     * addition</em> to the exported property's update.
     * @param property property's name (non-null)
     **/
    public void setCopyProperty(String property)
    {
        require_(property!=null,"setP- nonzro propnam");
        m_copyProperty = property;
    }


    /**
     * Returns name of regular property to be updated in
     * addition to the exported property. Returns <i>null</i>
     * if not set.
     **/
    public final String getCopyProperty()
    {
        return m_copyProperty;
    }

// ---------------------------------------------------------------------------------------
// Managing Modifiable Variable:
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if this task's source (value) has
     * not be explicitly defined.
     * @since JWare/AntX 0.3
     **/
    protected final boolean isUndefined()
    {
        return m_value.isUndefined();
    }


    /**
     * Updates the modifiable property with given value. What
     * this method actually effects depends on this task's scope.
     * @see ExportScope#setTheProperty ExportScope
     **/
    protected final void setTheProperty(String value)
    {
        ExportScope.setTheProperty(getScope(),getProject(),getName(),
                                   value,true);
    }


    /**
     * Returns the modifiable property's current value. Can return
     * <i>null</i> if property never set.
     * @see ExportScope#getTheProperty ExportScope
     **/
    protected final String getTheProperty()
    {
        return ExportScope.getTheProperty(getScope(),getProject(),
                                          getName());
    }


    /**
     * Sets a cached predetermined value for this task. This allows
     * pre-checks to cache values w/o changing the task's formal
     * definition.
     * @since JWare/AntX 0.3
     **/
    protected final void setShortcutValue(String value)
    {
        m_shortcutValue = value;
    }


    /**
     * Ensures auto-assigned delta values for certain modification
     * are defined; for example, "++" is shorthand for an increase of
     * one.
     **/
    protected void fixDefaults()
    {
        if (isUndefined()) {
            int op= getOp().getIndex();
            switch (op) {
                case Modification.INC_BY1_INDEX:
                case Modification.DEC_BY1_INDEX: {
                    setShortcutValue("1");
                    break;
                }
                //JWare/AntX 0.3
                case Modification.NOW_INDEX:
                case Modification.MINUS_NOW_INDEX: {
                    setShortcutValue(String.valueOf(System.currentTimeMillis()));
                    break;
                }
            }
        }
    }


    /**
     * Returns the initializing variable value. Defaults to a zero
     * Long. Could be different for typed variables; for instance
     * a timestamp might return the current system time.
     * @param like [optional] hint about the type of operation 
     *             value to be used in
     * @since JWare/AntX 0.2
     **/
    protected Number initialValue(Number like)
    {
        if (like instanceof Double) {
            return INITIAL_NUMF;
        }
        return INITIAL_NUMI;
    }


    /**
     * Returns any predetermined value for this task. This allows
     * pre-checks to cache values w/o changing the task's attribute
     * definitions.
     * @since JWare/AntX 0.3
     **/
    protected String shortcutValue()
    {
        return m_shortcutValue;
    }


    /**
     * Returns the value with which this task should operate. Takes
     * any cached shortcut values and/or flex conversion into acccount.
     * @see #getValue
     * @see #shortcutValue
     * @since JWare/AntX 0.3
     **/
    protected String getWorkValue()
    {
        String value = shortcutValue();
        if (value==null) {
            value = getValue();
        }
        return value;
    }



    /**
     * Applies any value transformation instruction(s) to the given
     * value. Extension point; by default we let another task like
     * {@linkplain CopyPropertyTask} do transformation work.
     * @param value [optional the value to be transformed
     * @param numberValue [optional] the numeric value to be transformed
     * @return the transformed value or same if no transformation necessary
     * @since JWare/AntX 0.4
     **/
    protected String getExportValue(String value, Number numberValue)
    {
        if (value==null) {
            // NB: do what most people expect in terms of float formatting!
            if (m_T==ValueTransform.DECIMAL) {
                synchronized(DFI) {
                    return DFI.format(numberValue);
                }
            }
            value = numberValue.toString();
        }
        if (m_T!=null) {
            value = TransformHelper.apply(m_T,value,getProject());
        }
        return value;
    }



    /**
     * Ensures this task is in a valid project and has an assigned
     * value.
     * @throws BuildException if value not defined
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);

        if (getName()==null) {
            String ermsg = uistrs().get("task.needs.name",getTaskName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }

        fixDefaults();

        if (isUndefined() && shortcutValue()==null) {
            if (getCopyProperty()!=null &&
                getOp().getIndex()==Modification.SET_INDEX) {//justCopy
                return;
            }
            String ermsg = uistrs().get
                ("task.needs.this.attr",getTaskName(),SRC_ATTRS);
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }



    /**
     * Performs a delta operation on the existing property value (which
     * must be either undefined or a valid number).
     * @param negate <i>true</i> if operation is a decrement
     * @throws BuildException if either the current value or the delta
     *         value are not valid numbers
     **/
    protected void doDeltaOperation(boolean negate)
    {
        String value = getWorkValue();
        Number delta = Tk.numberFrom(value);
        if (delta==Tk.NO_NUM_NUM) {
            String ermsg = uistrs().get("export.value.NAN",value);
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }

        String oldvalue = getTheProperty();
        Number current;
        if (oldvalue!=null) {
            current = Tk.numberFrom(oldvalue);
        } else {
            current = initialValue(delta);//!?
        }
        if (current==Tk.NO_NUM_NUM) {
            String ermsg = uistrs().get("export.current.NAN",getName(),oldvalue);
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }
        
        if (current instanceof Double || delta instanceof Double) {
            double delta_ = delta.doubleValue();
            if (negate) {
                delta_ = -delta_;
            }
            current = new Double(current.doubleValue()+delta_);
        } else {
            long delta_ = delta.longValue();
            if (negate) {
                delta_ = -delta_;
            }
            current = new Long(current.longValue()+delta_);
        }
        setTheProperty(getExportValue(null,current));
    }



    /**
     * Performs either of two string-specific operations (len or
     * strcat).
     * @since JWare/AntX 0.5
     **/
    protected void doStringOperation()
    {
        String value = null;
        int opI= getOp().getIndex();
        if (opI==Modification.STRLEN_INDEX) {
            value = String.valueOf(getWorkValue().length());
        }
        else if (opI==Modification.STRCAT_INDEX) {
            value = getTheProperty();
            if (value==null) {
                value = "";
            }
            value = value + getWorkValue();
        }
        setTheProperty(getExportValue(value,null));
    }




    /**
     * Performs the assignment operation of the current value to
     * the exported variable.
     * @since JWare/AntX 0.3
     **/
    protected void doAssignOperation()
    {
        String value = getWorkValue();
        if (value==null) {
            String ermsg = uistrs().get("export.value.undefined",m_value.get());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
        setTheProperty(getExportValue(value,Tk.NO_NUM_NUM));
    }



    /**
     * Returns <i>true</i> if is just copy to property operation.
     **/
    private boolean isJustCopy()
    {
        return isUndefined() &&
            shortcutValue()==null &&
            getCopyProperty()!=null;
    }



    /**
     * Modifies this task's named property according to operation
     * and value definitions.
     * @throws BuildException if invalid parameters or current value
     *        cannot be modified in manner requested
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");

        if (isJustCopy()) {
            getProject().setNewProperty(getCopyProperty(),
                              getExportValue(getTheProperty(),Tk.NO_NUM_NUM));
            return;
        }

        switch (getOp().getIndex()) {
            case Modification.NOW_INDEX: {
                if (!isUndefined()) {
                    String warning = uistrs().get
                        ("task.one.or.other.attr", "now", SRC_ATTRS);
                    log(warning, Project.MSG_WARN);
                }
                //+fall-through to assign
            }
            case Modification.SET_INDEX: {
                doAssignOperation();
                break;
            }
            case Modification.MINUS_NOW_INDEX: {
                if (!isUndefined()) {
                    Number from = Tk.numberFrom(getValue());
                    if (from!=Tk.NO_NUM_NUM) {
                        long diff = System.currentTimeMillis() - from.longValue();
                        if (diff<0) {
                            from= Tk.NO_NUM_NUM;
                        } else {
                            setShortcutValue(String.valueOf(diff));
                            doAssignOperation();
                            break;
                        }
                    }
                    if (from==Tk.NO_NUM_NUM) {
                        String ermsg = uistrs().get("export.value.NAN",m_value.get());
                        log(ermsg, Project.MSG_ERR);
                        throw new BuildException(ermsg, getLocation());
                    }
                }
                doDeltaOperation(true);
                break;
            }
            case Modification.INC_INDEX:
            case Modification.INC_BY1_INDEX: {
                doDeltaOperation(false);
                break;
            }
            case Modification.DEC_INDEX:
            case Modification.DEC_BY1_INDEX: {
                doDeltaOperation(true);
                break;
            }
            default: {
                doStringOperation();
                break;
            }
        }

        if (getCopyProperty()!=null) {
            getProject().setNewProperty(getCopyProperty(), getTheProperty());
        }
    }


    private static final Long INITIAL_NUMI= new Long(0);
    private static final Double INITIAL_NUMF= new Double(0.0f);
    private static final DecimalFormat DFI = new DecimalFormat("#.###");//NB:90+% expects

    private String m_name;
    private FlexString m_value = new FlexString();
    private String m_copyProperty;
    private Modification m_Op= Modification.SET;
    private ExportScope m_scope= ExportScope.DEFAULT_SCOPE;
    private String m_shortcutValue;
    private ValueTransform m_T;//NB: none-by-default
}

/* end-of-ExportTask.java */
