/**
 * $Id: CopyReferenceTask.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2002-2005,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.solo;

import  java.lang.reflect.Method;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.Empties;
import  com.idaremedia.antx.parameters.FeedbackLevel;

/**
 * Fixture configuration instruction to copy a reference within a single Project or
 * from a parent Project to a child Project. Must be a Task derivative since expected to
 * be nested inside TaskContainers (which only allow nested tasks).
 * <p>
 * <b>Example Usage:</b><pre>
 *   &lt;copyreference refid="default.buildnum" torefid="buildnum"/&gt;
 *   &lt;copyreference refid="default.buildnum" torefid="buildnum" overwrite="no"/&gt;
 *   &lt;copyreference refid="default.buildnum" torefid="buildnum" haltifexists="yes"/&gt;
 *   &lt;copyreference refid="allfilters" torefid="javafilters" shallow="yes"/&gt;
 *
 * -OR with flow-control tasks-
 *
 *   &lt;callforeach targets="&#46;&#46;&#46;" &#46;&#46;&#46;&gt;
 *     &lt;reference refid="default.buildnum"/&gt;
 *   &lt;/callforeach&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2002-2005,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @.pattern GoF.Adapter (when used with build-flowcontrol [call] tasks)
 * @see      CopyPropertyTask
 **/

public final class CopyReferenceTask extends AssertableTask
    implements AntLibFriendly
{
    /**
     * Initializes a new CopyReferenceTask adapter task.
     **/
    public CopyReferenceTask()
    {
        super(AntX.nopackage);
    }


    /**
     * Set the source reference's identifier
     * @param srcRefId existing reference (non-null).
     **/
    public void setRefId(String srcRefId)
    {
        require_(srcRefId!=null, "setRefId- nonzro id");
        m_srcRefId = srcRefId;

        if (m_dstRefId==null) {
            m_dstRefId= srcRefId;
        }
    }


    /**
     * Returns the source reference's identifer.
     * Returns <i>null</i> if never set.
     **/
    public String getRefId()
    {
        return m_srcRefId;
    }


    /**
     * Set the target reference's identifier (non-null)
     * @param dstRefId new reference (non-null)
     * @see #setHaltIfExists
     **/
    public void setToRefId(String dstRefId)
    {
        require_(dstRefId!=null, "setRefToId- nonzro id");
        m_dstRefId = dstRefId;
    }


    /**
     * Returns the (new) target reference's identifer.
     * Returns <i>null</i> if never set and source never
     * set.
     **/
    public String getToRefId()
    {
        return m_dstRefId;
    }


    /**
     * Sets if this task will generate a build error if the
     * destination reference id already exists within its project.
     * Trumps the 'overwrite' option.
     * @param halt <i>true</i> to throw build exception
     * @see #setToRefId
     * @see #setOverwrite
     **/
    public void setHaltIfExists(boolean halt)
    {
        m_haltIfExists = halt;
    }


    /**
     * Returns <i>true</i> if this task will generate a build error
     * if the destination reference id already exists within its
     * project. Defaults <i>false</i> (supports overwrites).
     **/
    public boolean isHaltIfExists()
    {
        return m_haltIfExists;
    }


    /**
     * Sets if this task will overwrite an existing reference.
     * This option is ignored if the 'haltiferror' option
     * is turned on.
     * @see #setHaltIfExists
     * @param allowOverwrite <i>true</i> if can overwrite old reference
     **/
    public void setOverwrite(boolean allowOverwrite)
    {
        m_allowOverwrite = allowOverwrite;
    }


    /**
     * Returns <i>true</i> if this task will overwrite an existing
     * reference. Defaults <i>true</i>. Ignored if 'haltiferror'
     * option is turned on.
     **/
    public boolean willAllowOverwrite()
    {
        return m_allowOverwrite;
    }



    /**
     * Tells this copy to only copy the reference pointer; no cloning
     * is necessary. Useful for moving a reference to another name.
     * @param shallowCopy <i>true</i> to make copy shallow.
     * @since JWare/AntX 0.5
     **/
    public void setShallow(boolean shallowCopy)
    {
        m_shallowCopy = shallowCopy;
    }
    
    
    
    /**
     * Returns <i>true</i> if this copy will be shallow (by reference
     * only). Default <i>false</i> if never set explicitly.
     * @since JWare/AntX 0.5
     **/
    public boolean willCopyShallow()
    {
        return m_shallowCopy;
    }


    /**
     * Tells this task how much non-diagnostic feedback to generate.
     * Really only has "loud" vs. "quiet-ish" interpretation. If
     * set quiet, this task will not issue a warning if it hits a
     * hard limit.
     * @param level feedback level (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setFeedback(String level)
    {
        require_(level!=null,"setFeedback- nonzro level");
        FeedbackLevel fbl = FeedbackLevel.from(level);
        if (fbl==null) {
            String e = getAntXMsg("task.illegal.param.value",
                           getTaskName(), level,"feedback");
            log(e, Project.MSG_ERR);
            throw new BuildException(e, getLocation());
        }
        m_fbLevel = fbl;
    }



    /**
     * Returns this task's assigned feedback level. Will return
     * <i>null</i> by default.
     * @since JWare/AntX 0.5
     **/
    public final FeedbackLevel getFeedbackLevel()
    {
        return m_fbLevel;
    }



    /**
     * Tries to copy the source reference object to a new copy
     * within this task's enclosing project.
     * @throws BuildException if this task in incompletely defined,
     *            the source object cannot be cloned, or an object
     *            already exists with destination refid
     **/
    public void execute()
    {
        verifyCanExecute_("execute");

        final Project P  = getProject();
        final boolean quiet = FeedbackLevel.isQuietish(m_fbLevel,true);
        String src_refid = getRefId();
        String dst_refid = getToRefId();

        //NB: when executed as standalone the 'to' refid must be unique
        if (src_refid.equals(dst_refid)) {
            String ermsg = uistrs().get("task.needs.this.attr",getTaskName(),"torefid");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }

        //NB: warn if about to stomp on some poor existing reference!
        if (P.getReference(dst_refid)!=null) {
            String msg = uistrs().get("task.warn.refid.exists",dst_refid);
            if (isHaltIfExists()) {
                log(msg,Project.MSG_ERR);
                throw new BuildException(msg,getLocation());
            }
            if (!willAllowOverwrite()) {
                if (!quiet && m_fbLevel!=FeedbackLevel.NORMAL) {
                    log(msg,Project.MSG_VERBOSE);
                }
                return;
            }
            if (!quiet) {
                log(msg,Project.MSG_WARN);
            }
        }

        //NB: because the interface exists, doesn't mean it's implemented
        //    properly (ssmc)
        Object src = P.getReference(src_refid);
        Object dst = null;
        try {
            if (willCopyShallow()) {
                dst = src;
            } else if (src instanceof String) {//FIXME:String-hack for now (ssmc)
                dst = new String((String)src);
            } else {
                Method cloneit = src.getClass().getMethod("clone", new Class[0]);
                dst = cloneit.invoke(src,Empties.EMPTY_CLASS_ARRAY);
            }
        } catch (Exception anyX) {
            String ermsg = uistrs().get("task.bad.refid",src_refid,
                                        "PUBLIC "+Cloneable.class.getName(),
                                        src.getClass().getName());
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,anyX,getLocation());
        }

        log("Copying reference "+src_refid+" to "+dst_refid,Project.MSG_DEBUG);
        if (quiet) {
            P.getReferences().put(dst_refid, dst);
        } else {
            P.addReference(dst_refid, dst);
        }
        
    }


    /**
     * Verifies that this task has at least its source reference
     * identifier set.
     * @throws BuildException if this task's source reference undefined
     **/
    public void verifyIsDefined()
    {
        if (getRefId()==null) {
            String ermsg = uistrs().get("task.needs.this.attr",getTaskName(),"refid");
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg,getLocation());
        }
    }


    /**
     * Verifies that this task refers to an existing project entity. If
     * the project and/or location is not specified this task will use its
     * own information.
     * @param theProject [optional]project in which reference must exist
     * @param theLocation [optional] location from which any error reported
     * @param ofKind [optional] reference must be assignable as this class
     * @throws BuildException if reference doesn't exist or is incompatible
     **/
    public void verifyExists(Project theProject, Location theLocation, Class ofKind)
    {
        verifyIsDefined();

        if (theProject==null) {
            theProject = getProject();
            if (theProject==null) {
                throw new IllegalStateException("verifyExists- no project");
            }
        }
        if (theLocation==null) {
            theLocation = getLocation();
        }

        String error = null;

        String refid = getRefId();

        Object o = theProject.getReference(refid);
        if (o==null) {
            error = uistrs().get("task.missing.refid",refid);
        } else if (ofKind!=null && !ofKind.isAssignableFrom(o.getClass())) {
            if ((ofKind==Cloneable.class) && (o instanceof String)) {
                return;//FIXME:String-hack for now (ssmc)
            }
            error = uistrs().get("task.bad.refid",refid,ofKind.getName(),
                                 o.getClass().getName());
        }
        if (error!=null) {
            log(error,Project.MSG_ERR);
            throw new BuildException(error,theLocation);
        }
    }


    /**
     * Verifies this task is completely defined, the source reference
     * exists, and is publicly Cloneable.
     * @throws BuildException if not valid
     **/
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);
        
        Class ofKind = Cloneable.class;
        if (willCopyShallow()) {
            ofKind = null;
        }
        verifyExists(null,null,ofKind);
    }

    private String m_srcRefId,m_dstRefId;
    private boolean m_haltIfExists;//NB:false allow overwrites!
    private boolean m_allowOverwrite=true;//NB:true allow overwrites!
    private boolean m_shallowCopy;//NB:true=>no-cloning
    private FeedbackLevel m_fbLevel=FeedbackLevel.NORMAL;//NB:=>overwrite noise
}

/* end-of-CopyReferenceTask.java */
