/**
 * $Id: TransformHelper.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2004-2005,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  java.text.DecimalFormat;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.util.FileUtils;

import  com.idaremedia.antx.helpers.DateTimeFormat;
import  com.idaremedia.antx.helpers.DurationFormat;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;


/**
 * Utilities to do simple formatting of (raw) string values based on our various
 * transform instructions.
 * <p>
 * <b>Example Usage:</b><pre>
 *     public class MyClass ...
 *     {
 *         private ValueTransform m_T= ValueTransform.NONE;
 *         private String m_value;
 *
 *         public final void setValue(String s) {
 *            m_value = s;
 *         }
 *         public final void setTransform(String t) {
 *            m_T = (t==null) ? ValueTransform.NONE : ValueTransform.from(t,m_T);
 *         }
 *
 *         public final String getValue() {
 *            return <b>TransformHelper.apply</b>(m_T,m_value,getProject());
 *         }
 *         ...
 *     }
 * </pre>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004-2005,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  multiple
 * @.group   impl,helper
 * @see      CaseTransform
 * @see      ValueTransform
 **/

public final class TransformHelper
{
    /**
     * Converts a string value to a long number or returns a
     * symbolic {@linkplain Tk#NO_NUM NO_NUM} if string could
     * not be converted.
     * @param s number string (non-null)
     * @since JWare/AntX 0.4
     **/
    public static final long toNumber(String s)
    {
        return Tk.longFrom(s,Tk.NO_NUM);
    }



    /**
     * Like {@linkplain #toNumber toNumber(&#8230;)} except if
     * the incoming string equals "<span class="src">now</span>"
     * the system's current time is returned as a long
     * (milliseconds).
     * @param s number string or the value "now" (non-null)
     * @since JWare/AntX 0.5
     **/
    public static final long toTimeNumber(String s)
    {
        if (Strings.NOW.equals(s)) {
            return System.currentTimeMillis();
        }
        return Tk.longFrom(s,Tk.NO_NUM);
    }



    /**
     * Inverts the case of every character in a string. Will
     * invert characters according to current locale response
     * to 'isLowerCase' and 'isUpperCase'. Returns <i>null</i>
     * if incoming string is <i>null</i>.
     * @param s string to be inverted
     * @since JWare/AntX 0.4
     **/
    public static final String toInvertCase(String s)
    {
        if (s==null) {
            return s;
        }
        StringBuffer sb = new StringBuffer(s);
        for (int i=0,n=s.length();i<n;i++) {
            char c = s.charAt(i);
            if (Character.isLowerCase(c)) {
                sb.setCharAt(i,Character.toUpperCase(c));
            }
            else if (Character.isUpperCase(c)) {
                sb.setCharAt(i,Character.toLowerCase(c));
            }
        }
        s = sb.substring(0);
        return s;
    }



    /**
     * Converts a general value transform instruction into a specific
     * case transform. Returns <i>null</i> if unable to match by name.
     * @param gt the general transform instruction (non-null)
     * @since JWare/AntX 0.4
     **/
    public static CaseTransform toCaseTransform(ValueTransform gt)
    {
        if (gt==null) {
            throw new IllegalArgumentException();
        }
        return CaseTransform.from(gt.getValue());
    }




    /**
     * Converts a case transform instruction into a generic value
     * transform. Returns <i>null</i> if unable to match by name.
     * @param ct the case transform instruction (non-null)
     * @since JWare/AntX 0.4
     **/
    public static ValueTransform toValueTransform(CaseTransform ct)
    {
        if (ct==null) {
            throw new IllegalArgumentException();
        }
        return ValueTransform.from(ct.getValue());
    }



    /**
     * Short hand for transforming a file system path or file URL
     * into a normalized (possibly project-based) full path. Common
     * operation within AntX tools.
     * @param fileRef description of file (path or URL)
     * @param P project to use if fileref is not absolute
     * @return normalized absolute path (non-null)
     * @since JWare/AntX 0.5
     **/
    public static final String toPath(String fileRef, Project P)
    {
        return apply(ValueTransform.OSPATH,fileRef,P);
    }



    /**
     * Utility that implements simple lowercasing, uppercasing,
     * and invertcasing transformation.
     * @param t the transform instruction (non-null)
     * @param s the value to be transformed
     * @since JWare/AntX 0.4
     **/
    public static final String apply(CaseTransform t, String s)
    {
        if (t==null) {
            throw new IllegalArgumentException();
        }
        if (s==null) {
            return null;
        }
        switch(t.getIndex()) {
            case CaseTransform.LOWERCASE_INDEX: {
                s = s.toLowerCase();
                break;
            }
            case CaseTransform.UPPERCASE_INDEX: {
                s = s.toUpperCase();
                break;
            }
            case CaseTransform.INVERTCASE_INDEX:{
                s = toInvertCase(s);
                break;
            }
        }
        return s;
    }



    /**
     * Utility that implements simple interpretation of value
     * transformation instructions. Good as a generic fallback.
     * @param t the transform instruction (non-null)
     * @param s the value to be transformed
     * @param P controlling project for problem feedback (non-null)
     * @see DateTimeFormat
     * @see DurationFormat
     * @since JWare/AntX 0.4
     **/
    public static final String apply(ValueTransform t, String s, Project P)
    {
        if (t==null || P==null) {
            throw new IllegalArgumentException();
        }
        if (s==null) {
            return null;
        }
        switch(t.getIndex()) {
            case ValueTransform.TRIM_INDEX: {
                s = s.trim();
                break;
            }
            case ValueTransform.STRIPWS_INDEX: {
                s = Tk.stripWhitespace(s);
                break;
            }
            case ValueTransform.LOWERCASE_INDEX: {
                s = s.toLowerCase();
                break;
            }
            case ValueTransform.UPPERCASE_INDEX: {
                s = s.toUpperCase();
                break;
            }
            case ValueTransform.INVERTCASE_INDEX: {
                s = toInvertCase(s);
                break;
            }
            case ValueTransform.DURATION_INDEX:  {
                long l= toNumber(s);
                if (l!=Tk.NO_NUM) {
                    s = DurationFormat.INSTANCE.format(l,true);
                } else {
                    P.log("Unable to convert '"+s+"' to time duration.",
                          Project.MSG_WARN);
                }
                break;
            }
            case ValueTransform.DATETIME_INDEX: {
                long l= toTimeNumber(s);
                if (l!=Tk.NO_NUM) {
                    s = DateTimeFormat.shortformat(l);
                } else {
                    P.log("Unable to convert '"+s+"' to datetime.",
                          Project.MSG_WARN);
                }
                break;
            }
            case ValueTransform.CHECKPOINT_INDEX:  {
                long l= toTimeNumber(s);
                if (l!=Tk.NO_NUM) {
                    s = DateTimeFormat.GMTformat(l);
                } else {
                    P.log("Unable to convert '"+s+"' to GMT timestamp.",
                          Project.MSG_WARN);
                }
                break;
            }
            case ValueTransform.LONGDATETIME_INDEX: {
                long l= toTimeNumber(s);
                if (l!=Tk.NO_NUM) {
                    s = DateTimeFormat.format(l);
                } else {
                    P.log("Unable to convert '"+s+"' to full datetime.",
                          Project.MSG_WARN);
                }
                break;
            }
            case ValueTransform.OSPATH_INDEX: {
                FileUtils fu= FileUtils.getFileUtils();
                if (s.startsWith("file:")) {
                    s = fu.fromURI(s);
                } else {
                    //s = fu.normalize(s).getPath();
                    s = P.resolveFile(s).getPath();
                }
                break;
            }
            case ValueTransform.OSPATHURL_INDEX: {
                FileUtils fu= FileUtils.getFileUtils();
                if (s.startsWith("file:")) {
                    s = fu.fromURI(s);
                } else {
                    s = P.resolveFile(s).getPath();
                }
                s = fu.toURI(s);
                break;
            }
            case ValueTransform.DECIMAL_INDEX: {
                Number n = Tk.numberFrom(s);
                if (n==Tk.NO_NUM_NUM) {
                    s = "";
                } else {
                    synchronized(DFI) {
                        s = DFI.format(n);
                    }
                }
                break;
            }
        }
        return s;
    }

    private static final DecimalFormat DFI = new DecimalFormat("#.###");//NB:90+% expects
}

/* end-of-TransformHelper.java */
