/**
 * $Id: PropertySource.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.parameters;

import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Enumeration that represents the possible sources of a project property in the Ant
 * runtime. The following list explains what kinds of properties are lumped under
 * the various source categories:<ul>
 *    <li><span class="src">all</span>: All project properties regardless of origin.</li>
 *    <li><span class="src">all--</span>: Like "<span class="src">all</span>" but
 *        with the standard immutable Java/Sun properties removed.</li>
 *    <li><span class="src">command</span>: Project properties that were either defined
 *        as command arguments to main Ant process (via CLI or GUI) or is one of a small
 *        set of special Ant runtime properties (like
 *        <span class="src">ant.version</span>).</li>
 *    <li><span class="src">control</span>: Non-command properties that were defined
 *        by the project's Ant launch context (like the nested properties inside an
 *        originating <span class="src">&lt;antcall&gt;</span>).</li>
 *    <li><span class="src">script</span>: Properties that are neither command nor
 *        control properties. This group includes the regular Java runtime's
 *        <span class="src">System</span> properties and properties defined within the
 *        current Ant context or <span class="src">Project</span>.</li>
 *    <li><span class="src">script--</span>: Like "<span class="src">script</span>" but
 *        with the standard immutable Java/Sun properties removed.</li>
 *    <li><span class="src">jre</span>: Properties that are defined in the
 *        standard Java runtime's <span class="src">System Properties</span>.</li>
 *    <li><span class="src">fixture</span>: Properties that are defined in either the
 *        AntX fixture or the <span class="src">System</span> properties.</li>
 * </ul>
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,helper
 **/

public final class PropertySource extends EnumSkeleton
{
    /** Index of {@linkplain #ALL ALL}. **/
    public static final int ALL_INDEX = 0;
    /** Index of {@linkplain #ALLlite ALLlite}. **/
    public static final int ALLlite_INDEX = ALL_INDEX+1;

    /** Index of {@linkplain #COMMAND COMMAND}. **/
    public static final int COMMAND_INDEX = ALLlite_INDEX+1;
    /** Index of {@linkplain #CONTROL CONTROL}. **/
    public static final int CONTROL_INDEX = COMMAND_INDEX+1;

    /** Index of {@linkplain #SCRIPT SCRIPT}. **/
    public static final int SCRIPT_INDEX = CONTROL_INDEX+1;
    /** Index of {@linkplain #SCRIPTlite SCRIPTlite}. **/
    public static final int SCRIPTlite_INDEX = SCRIPT_INDEX+1;

    /** Index of {@linkplain #JRE JRE}. **/
    public static final int JRE_INDEX = SCRIPTlite_INDEX+1;
    /** Index of {@linkplain #FIXTURE FIXTURE}. **/
    public static final int FIXTURE_INDEX = JRE_INDEX+1;


    /** Singleton "<span class="src">all</span>" choice. **/
    public static final PropertySource ALL=
        new PropertySource("all",ALL_INDEX);

    /** Singleton "<span class="src">all--</span>" choice. **/
    public static final PropertySource ALLlite=
        new PropertySource("all--",ALLlite_INDEX);

    /** Singleton "<span class="src">command</span>" choice. **/
    public static final PropertySource COMMAND=
        new PropertySource("command",COMMAND_INDEX);

    /** Singleton "<span class="src">control</span>" choice. **/
    public static final PropertySource CONTROL=
        new PropertySource("control",CONTROL_INDEX);

    /** Singleton "<span class="src">script</span>" choice. **/
    public static final PropertySource SCRIPT=
        new PropertySource("script",SCRIPT_INDEX);

    /** Singleton "<span class="src">script--</span>" choice. **/
    public static final PropertySource SCRIPTlite=
        new PropertySource("script--",SCRIPTlite_INDEX);

    /** Singleton "<span class="src">jre</span>" choice. **/
    public static final PropertySource JRE=
        new PropertySource("jre",JRE_INDEX);

    /** Singleton "<span class="src">fixture</span>" choice. **/
    public static final PropertySource FIXTURE=
        new PropertySource("fixture",FIXTURE_INDEX);



    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public PropertySource()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensures this enum is
     * initialized as if with the default Ant Introspector
     * helper thingy.
     **/
    private PropertySource(String v, int i)
    {
        super(v);
    }


    /**
     * Returns copy of all possible source values as an ordered
     * string array. Note: ordering should be same as our
     * singleton indices.
     **/
    public String[] getValues()
    {
        return new String[] {"all", "all--",
                             "command", "control",
                             "script", "script--",
                             "jre", "fixture"};
    };



    /**
     * Helper that converts a scalar to a known PropertySource.
     * Returns <i>null</i> if value does not match any of expected
     * source.
     * @param i the index to be matched
     **/
    public static PropertySource from(int i)
    {
        if (i==COMMAND.index) { return COMMAND; }
        if (i==ALL.index)     { return ALL; }
        if (i==ALLlite.index) { return ALLlite; }
        if (i==CONTROL.index) { return CONTROL; }
        if (i==SCRIPT.index)  { return SCRIPT; }
        if (i==SCRIPTlite.index)  { return SCRIPTlite; }
        if (i==FIXTURE.index) { return FIXTURE; }
        if (i==JRE.index)     { return JRE; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known
     * PropertySource's index.
     * @param i the index to be matched
     * @param dflt the default PropertySource if necessary
     **/
    public static PropertySource from(int i, PropertySource dflt)
    {
        PropertySource choice= from(i);
        return (choice==null) ? dflt : choice;
    }


    /**
     * Helper that converts a string to a known PropertySource
     * singleton. Returns <i>null</i> if string unrecognized. String
     * can be either PropertySource's symbolic name or its index.
     **/
    public static PropertySource from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                // NOTE: We must support the pre AntX-0.4 choices of
                //       "all", "user", etc. as well as the names used
                //       by the standard <propertyset> command [ssmc]
                if (COMMAND.value.equals(s))    { return COMMAND; }
                if (Strings.USER.equals(s))     { return COMMAND; }
                if (ALL.value.equals(s))        { return ALL; }
                if (ALLlite.value.equals(s))    { return ALLlite; }
                if (SCRIPT.value.equals(s))     { return SCRIPT; }
                if (SCRIPTlite.value.equals(s)) { return SCRIPTlite; }
                if (CONTROL.value.equals(s))   { return CONTROL; }
                if (FIXTURE.value.equals(s))   { return FIXTURE; }
                if (JRE.value.equals(s))       { return JRE; }
                if (Strings.DEFAULT.equals(s)) { return ALL; }
                if ("commandline".equals(s))   { return COMMAND; }//<propertyset>
                if ("system".equals(s))        { return JRE; }
                if ("antx".equals(s))          { return FIXTURE; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but with a
     * default value if supplied value does not match any known
     * PropertySource's name.
     * @param s the symbolic name to be matched
     * @param dflt the default PropertySource if necessary
     **/
    public static PropertySource from(String s, PropertySource dflt)
    {
        PropertySource choice= from(s);
        return (choice==null) ? dflt : choice;
    }
}

/* end-of-PropertySource.java */
