/**
 * $Id: ConditionalParametersTest.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers.tests;

import  java.util.Map;
import  java.util.Properties;

import  junit.framework.TestSuite;

import  com.idaremedia.antx.helpers.InnerNameValuePair;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.ownhelpers.ConditionalInnerNameValuePair;
import  com.idaremedia.antx.ownhelpers.ConditionalParameters;
import  com.idaremedia.antx.parameters.Conditional;
import  com.idaremedia.antx.ut.HTC;
import  com.idaremedia.antx.ut.HTCUtils;

/**
 * Test suite for {@linkplain com.idaremedia.antx.ownhelpers.ConditionalParameters}.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,test
 **/

public final class ConditionalParametersTest extends HTC
{
    /** <i>PET</i> Test Category. **/
    public static final String TEST_CATEGORY="CLASS";


    /**
     * Create new ConditionalParametersTest testcase.
     **/
    public ConditionalParametersTest(String methodName)
    {
        super("ConditionalParameters::",methodName);
    }


    /**
     * Create full test suite for ConditionalParameters.
     **/
    public static TestSuite suite()
    {
        return new TestSuite(ConditionalParametersTest.class);
    }


    /**
     * Create baseline test suite for ConditionalParameters (same as full).
     **/
    public static TestSuite baseline()
    {
        return suite();
    }


    /**
     * Make this test (standalone) self-running.
     **/
    public static void main(String[] argv)
    {
        HTCUtils.quickCheck(suite());
    }


// ---------------------------------------------------------------------------------------------------------
// --------------------------------------- [ Misc Factory Methods ] ----------------------------------------
// ---------------------------------------------------------------------------------------------------------

    protected void setUp() throws Exception
    {
        configureProjectFromResource("blank.xml");
    }


    private ConditionalParameters newOUT()
    {
        return new ConditionalParameters(getProject());
    }

    private ConditionalInnerNameValuePair newIf(String name, String value, String property)
    {
        ConditionalInnerNameValuePair arg = new ConditionalInnerNameValuePair(name,value);
        arg.setIf(property);
        if (value==null) {
            arg.setValue(Strings.UNDEFINED);
        }
        return arg;
    }

    private ConditionalInnerNameValuePair newUn(String name, String value, String property)
    {
        ConditionalInnerNameValuePair arg = new ConditionalInnerNameValuePair(name,value);
        arg.setUnless(property);
        if (value==null) {
            arg.setValue(Strings.UNDEFINED);
        }
        return arg;
    }

    private ConditionalInnerNameValuePair newAlways(String name, String value)
    {
        ConditionalInnerNameValuePair arg = new ConditionalInnerNameValuePair(name,value);
        if (value==null) {
            arg.setValue(Strings.UNDEFINED);
        }
        return arg;
    }


// ---------------------------------------------------------------------------------------------------------
// ------------------------------------------- [ The Test Cases ] ------------------------------------------
// ---------------------------------------------------------------------------------------------------------

    public void checkBaseline()
    {
        //--Ensures setUp() works and can find our xml file!
        ConditionalParameters out = newOUT();
        assertTrue(out.isEmpty());
    }


    public void testBaseline()
    {
        checkBaseline();
    }


    /**
     * Verifies that if-constraints paid attention to as expected.
     * @since JWare/AntX 0.5
     **/
    public void testIfFiltersParameters_AntX05()
    {
        ConditionalParameters out = newOUT();
        out.addConfiguredArg(newAlways("name","frank"));
        out.addConfiguredArg(newIf("height","5'7","verbose"));
        out.addConfiguredArg(newIf("weight","197lbs","verbose"));
        out.addConfiguredArg(newIf("longname","Frank Burns","longnames"));
        assertEqual(out.size(),4,"ParamCount");

        out.getProject().setProperty("verbose","anyvalue");
        Properties filtered = out.copyOfSimpleKeyValues(getProject(),true);
        assertEqual(filtered.size(),3,"If-Filtered ParamCount");
        assertTrue(filtered.containsKey("name"),"'name' found");
        assertTrue(filtered.containsKey("height"),"'height' found");
        assertTrue(filtered.containsKey("weight"),"'weight' found");
    }



    /**
     * Verifies that if-constraints paid attention to as expected.
     * @since JWare/AntX 0.5
     **/
    public void testUnlessFiltersParameters_AntX05()
    {
        ConditionalParameters out = newOUT();
        out.addConfiguredArg(newAlways("name","billy"));
        out.addConfiguredArg(newUn("height","5'5","brief"));
        out.addConfiguredArg(newUn("weight","227lbs","brief"));
        out.addConfiguredArg(newUn("longname","Billy Joel","shortnames"));
        assertEqual(out.size(),4,"ParamCount");

        out.getProject().setProperty("shortnames","anyvalue");
        Properties filtered = out.copyOfSimpleKeyValues(getProject(),true);
        assertEqual(filtered.size(),3,"Unless-Filtered ParamCount");
        assertTrue(filtered.containsKey("name"),"'name' found");
        assertTrue(filtered.containsKey("height"),"'height' found");
        assertTrue(filtered.containsKey("weight"),"'weight' found");
    }


    /**
     * Verifies that both if-unless ocnstraints applied as expected.
     * @since JWare/AntX 0.5
     **/
    public void testBothFiltersAppliedAllTrue_AntX05()
    {
        ConditionalParameters out = newOUT();
        out.addConfiguredArg(newAlways("name","billy"));
        out.addConfiguredArg(newIf("height","5'5","verbose"));
        out.addConfiguredArg(newIf("weight","227lbs","verbose"));
        out.addConfiguredArg(newUn("longname","Billy Joel","shortnames"));
        assertEqual(out.size(),4,"ParamCount");

        out.getProject().setProperty("verbose","anyvalue");
        Properties filtered = out.copyOfSimpleKeyValues(getProject(),true);
        assertEqual(filtered.size(),4,"If/Unless-Filtered ParamCount");
        assertTrue(filtered.containsKey("name"),"'name' found");
        assertTrue(filtered.containsKey("height"),"'height' found");
        assertTrue(filtered.containsKey("weight"),"'weight' found");
        assertTrue(filtered.containsKey("longname"),"'longname' found");
    }




    /**
     * Verifies that both if-unless ocnstraints applied as expected.
     * @since JWare/AntX 0.5
     **/
    public void testBothFiltersAppliedAllFalse_AntX05()
    {
        ConditionalParameters out = newOUT();
        out.addConfiguredArg(newAlways("name","billy"));
        out.addConfiguredArg(newIf("height","5'5","verbose"));
        out.addConfiguredArg(newIf("weight","227lbs","verbose"));
        out.addConfiguredArg(newUn("longname","Billy Joel","shortnames"));
        assertEqual(out.size(),4,"ParamCount");

        out.getProject().setProperty("shortnames","anyvalue");
        Properties filtered = out.copyOfSimpleKeyValues(getProject(),true);
        assertEqual(filtered.size(),1,"If/Unless-Filtered ParamCount");
        assertTrue(filtered.containsKey("name"),"'name' found");
    }



    /**
     * Ensures the parameters object can handle different typed
     * contents (because client passes in references) with general
     * Conditional interface being key to inclusion/exclusion.
     * @since JWare/AntX 0.5
     **/
    public void testHandlesHetergenousItems_AntX05()
    {
        ConditionalParameters out = newOUT();
        out.addConfiguredArg(new InnerNameValuePair("name","WildThing"));
        out.addConfiguredArg(newIf("streak","continue","winning"));
        out.addConfiguredArg(newUn("hairstyle","mohawk","losing"));
        out.addConfiguredArg(new SeekritNVItem("team","indians"));
        out.addConfiguredArg(new SeekritConditionalNVItem("status","avail"));
        assertEqual(out.size(),5,"ParamCount");

        out.getProject().setProperty("losing","as-always");
        out.getProject().setProperty("freeagent","");

        Properties filtered = out.copyOfSimpleKeyValues(getProject());
        assertEqual(filtered.size(),3,"Custom NVItems included");
        assertTrue(filtered.containsKey("name"),"'name' found");
        assertTrue(filtered.containsKey("team"),"'team' found");
        assertEqual(filtered.getProperty("status"),"avail","status");
    }


    private static class SeekritNVItem extends InnerNameValuePair {
        SeekritNVItem(String name, String value) {
            super();
            setNV(name,value);
        }
    };

    private static class SeekritConditionalNVItem extends InnerNameValuePair
        implements Conditional {
        SeekritConditionalNVItem(String name, String value) {
            super();
            setNV(name,value);
        }
        public void setIf(String property) {
        }
        public String getIfProperty() {
            return "freeagent";
        }
        public void setUnless(String property) {
        }
        public String getUnlessProperty() {
            return null;
        }
    };


    /**
     * Ensures the filters are applied to copies of name-value objects
     * that are returned by CUT.
     * @since JWare/AntX 0.5
     **/
    public void testFiltersAppliedToRawItemsToo_AntX05()
    {
        ConditionalParameters out = newOUT();
        out.addConfiguredParameter(newIf("slow","turtle","slow"));
        out.addConfiguredProperty(newUn("fast","hare","slow"));
        out.addConfiguredArg(new SeekritNVItem("medium","gerbil"));
        ConditionalInnerNameValuePair item = new ConditionalInnerNameValuePair("superfast","cheetah");
        item.setIf("superfast");
        item.setUnless("superslow");
        out.addConfiguredProperty(item);
        assertEqual(out.size(),4,"ParamCount");

        getProject().setProperty("slow","yes");
        getProject().setProperty("superfast","yes");

        Map copy = out.copyOfParameterObjects();
        assertEqual(copy.size(),3,"Filtered ParamCount");
        assertNotNil(copy.get("slow"),"'slow'");
        assertNotNil(copy.get("medium"),"'medium'");
        assertNotNil(copy.get("superfast"),"'superfast'");

        assertIdent(copy.get("medium").getClass(),SeekritNVItem.class,"SeekritNVItem");
    }
}

/* end-of-ConditionalParametersTest.java */