/**
 * $Id: UEContainerProxy.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;
import  java.util.StringTokenizer;

import  org.apache.tools.ant.RuntimeConfigurable;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.TaskContainer;
import  org.apache.tools.ant.UnknownElement;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Helper when constructing UnknownElements(UEs) that contain other UEs.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @see      TaskExaminer#newUEProxy TaskExaminer.newUEProxy(&#8230;)
 **/

public final class UEContainerProxy implements TaskContainer
{
    /**
     * The element that must be "filled-in" with other UE items.
     **/
    private final UnknownElement m_rootItem;



    /**
     * Initialize a new container proxy for an unknown element.
     * @param impl the unknown element (acts like container)
     **/
    public UEContainerProxy(UnknownElement impl)
    {
        AntX.require_(impl!=null,AntX.nopackage,"ctor- nonzro UE");
        m_rootItem= impl;
    }




    /**
     * Ensures the incoming UE and its wrapper are added to our
     * underlying UE properly.
     * @param task the unknown element task proxy (non-null)
     **/
    public void addTask(Task task)
    {
        AntX.require_(task instanceof UnknownElement,
                      AntX.nopackage, "addWrappedTask- UE");
        UnknownElement ue = (UnknownElement)task;
        m_rootItem.addChild(ue);
        m_rootItem.getWrapper().addChild(ue.getWrapper());
    }



    /**
     * Sets an attribute on the underlying UE element. Useful if
     * a macro maker or custom macro definition needs to tweak
     * the item's attributes as it is itself defined.
     * @param name attribute's name (non-null)
     * @param value attribute's value (non-null)
     **/
    public void setAttribute(String name, String value)
    {
        RuntimeConfigurable wrapper = m_rootItem.getWrapper();
        wrapper.setAttribute(name,value);
    }



    /**
     * Sets a collection of attributes on the underlying UE
     * element. Shorthand for calling {@linkplain #setAttribute
     * setAttribute()} repeatedly. The passed map must contain
     * string-based name/value pairs.
     * @param attrs map of attribute name/value pairs (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setAttributes(Map attrs)
    {
        RuntimeConfigurable wrapper = m_rootItem.getWrapper();
        Iterator itr = attrs.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            wrapper.setAttribute((String)e.getKey(),(String)e.getValue());
        }
    }



    /**
     * Like {@linkplain #addTask addTask()} except the new element
     * is added to a nested child element. Incoming is therefore a
     * grandchild to root element.
     * @param task the unknown element grandchild proxy (non-null)
     * @param parentname local type name of parent (non-null)
     * @since JWare/AntX 0.5
     **/
    public void addGrandchildTask(Task task, String parentname)
    {
        AntX.require_(task instanceof UnknownElement,
                      AntX.nopackage, "addWrappedTask- UE");
        UnknownElement target = getBestTarget(parentname);
        UnknownElement ue = (UnknownElement)task;
        target.addChild(ue);
        target.getWrapper().addChild(ue.getWrapper());
    }



    /**
     * Like {@linkplain #setAttribute setAttribute()} except the new
     * attribute is set on a nested child element.
     * @param name attribute's name (non-null)
     * @param value attribute's value (non-null)
     * @param parentname local type name of parent (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setGrandchildAttribute(String name, String value,
        String parentname)
    {
        UnknownElement target = getBestTarget(parentname);
        RuntimeConfigurable wrapper = target.getWrapper();
        wrapper.setAttribute(name,value);
    }



    /**
     * Like {@linkplain #setAttributes setAttributes()} except the new
     * attributes are set on a nested child element.
     * @param attrs map of attribute name/value pairs (non-null)
     * @param parentname local type name of parent (non-null)
     * @since JWare/AntX 0.5
     **/
    public void setGrandchildAttributes(Map attrs, String parentname)
    {
        RuntimeConfigurable wrapper = getBestTarget(parentname).getWrapper();
        Iterator itr = attrs.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            wrapper.setAttribute((String)e.getKey(),(String)e.getValue());
        }
    }



    /**
     * Returns the main underlying unknown element. Never
     * returns <i>null</i>.
     * @since JWare/AntX 0.5
     * @see TaskExaminer#copyUEProxy TaskExaminer.copyUEProxy(&#8230;)
     **/
    public UnknownElement getUE()
    {
        return m_rootItem;
    }


    private UnknownElement getBestTarget(String itempath)
    {
        UnknownElement match = m_rootItem;
        if (!Tk.isWhitespace(itempath)) {
            StringTokenizer st= new StringTokenizer(itempath,",");
            while (st.hasMoreTokens()) {
                String nextname = st.nextToken();
                UnknownElement next = childFor(nextname,match);
                if (next==null) {
                    return m_rootItem;
                }
                match = next;
            }
        }
        return match;
    }



    private UnknownElement childFor(String childname, UnknownElement parent)
    {
        UnknownElement child = null;
        if (!Tk.isWhitespace(childname)) {
            List l = parent.getChildren();
            if (l!=null && !l.isEmpty()) {
                for (int i=0,N=l.size();i<N;i++) {
                    UnknownElement next = (UnknownElement)l.get(i);
                    if (childname.equals(next.getTaskName())) {
                        child = next;
                        break;
                    }
                }
            }
        }
        return child;
    }
}

/* end-of-UEContainerProxy.java */
