/**
 * $Id: OptionalExecuteHelper.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.ownhelpers;

import  java.util.Iterator;
import  java.util.Map;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.Iteration;
import  com.idaremedia.antx.go.Go;
import  com.idaremedia.antx.go.Iff;
import  com.idaremedia.antx.go.IffAll;
import  com.idaremedia.antx.go.IffAnt;
import  com.idaremedia.antx.go.IffOs;
import  com.idaremedia.antx.go.IffValue;
import  com.idaremedia.antx.parameters.Conditional;
import  com.idaremedia.antx.parameters.FlexConditional;

/**
 * Common implementation of the {@linkplain FlexConditional} interface. Public-facing
 * items can implement all or part of an AntX conditional interface by delegating to
 * an instance of this class. Note that the owning element must synchronize this helper's
 * project reference with its own. This class is <em>not</em> synchronized against
 * concurrent modification.
 *
 * @since    JWare/AntX 0.4
 * @author   ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,helper
 * @.caveat  If your task wants to track why it did not execute (ala logs) it must
 *           stop testing on the first failed test because subsequent tests can
 *           erase the "failure reason string" from the helper's memory.
 **/

public class OptionalExecuteHelper extends ProjectDependentSkeleton
    implements FlexConditional
{
    private static final String IAM_=
        AntX.AntX+"OptionalExecuteHelper:";


    /**
     * Initializes a new helper.
     **/
    public OptionalExecuteHelper()
    {
    }


    /**
     * Initializes a new helper bound to a particular
     * project.
     * @param P the project
     **/
    public OptionalExecuteHelper(Project P)
    {
        setProject(P);
    }


    /**
     * Initializes a new helper based on a placeholder item's
     * execution attributes. The attribute names must be normalized
     * to all lowercase (US locale).
     * @param attrs collection of controller's parameters (non-null)
     * @.impl Only 'Conditional' and 'PlatformConditional' and "true|false".
     **/
    public OptionalExecuteHelper(Map attrs)
    {
        AntX.require_(attrs!=null,IAM_,"ctor- nonzro attrs");
        String v;

        v = (String)attrs.get("if");
        if (v!=null) { setIf(v); }
        v = (String)attrs.get("unless");
        if (v!=null) { setUnless(v); }

        v = (String)attrs.get("true");
        if (v!=null) { setTrue(v); }
        v = (String)attrs.get("false");
        if (v!=null) { setFalse(v); }

        v = (String)attrs.get("ifos");
        if (v!=null) { setIfOS(v); }
        v = (String)attrs.get("unlessos");
        if (v!=null) { setUnlessOS(v); }

        v = (String)attrs.get("ifantlike");
        if (v!=null) { setIfAntLike(v); }
        v = (String)attrs.get("unlessantlike");
        if (v!=null) { setUnlessAntLike(v); }
   }


    /**
     * Adds a custom if-condition to this helper.
     * @param t the test (non-null)
     **/
    public void addIfTest(Go.Test t)
    {
        m_ifTests.add(t);
    }


    /**
     * Adds a custom unless-condition to this helper.
     * @param t the test (non-null)
     **/
    public void addUnlessTest(Go.Test t)
    {
        m_unlessTests.add(t);
    }


    /**
     * Clears <em>all</em> of this helper's tests.
     **/
    public final void clearTests()
    {
        m_ifTests.clear();
        m_unlessTests.clear();
    }


    public void setIf(String property)
    {
        addIfTest(new Iff.Exists(property));
    }


    public void setUnless(String property)
    {
        addUnlessTest(new Iff.NotExists(property));
    }


    public void setTrue(String booleanString)
    {
        addIfTest(new IffValue.IsTrue(booleanString));
    }

    public void setFalse(String booleanString)
    {
        addIfTest(new IffValue.IsNotTrue(booleanString));
    }


    public void setIfTrue(String property)
    {
        addIfTest(new Iff.IsTrue(property));
    }


    public void setUnlessTrue(String property)
    {
        addUnlessTest(new Iff.IsNotTrue(property));
    }


    public void setIfAll(String properties)
    {
        addIfTest(new IffAll.Exists(properties));
    }


    public void setUnlessAll(String properties)
    {
        addUnlessTest(new IffAll.NotExists(properties));
    }


    public void setIfAllTrue(String properties)
    {
        addIfTest(new IffAll.IsTrue(properties));
    }


    public void setUnlessAllTrue(String properties)
    {
        addUnlessTest(new IffAll.IsNotTrue(properties));
    }


    public void setIfOS(String choice)
    {
        addIfTest(new IffOs.Is(choice));
    }


    public void setUnlessOS(String choice)
    {
        addUnlessTest(new IffOs.IsNot(choice));
    }


    public void setIfAntLike(String versionRE)
    {
        addIfTest(new IffAnt.IsLike(versionRE));
    }


    public void setUnlessAntLike(String versionRE)
    {
        addUnlessTest(new IffAnt.IsNotLike(versionRE));
    }


    public String getIfProperty()
    {
        String p= "";
        int N= m_ifTests.size();
        if (N>0) {
            if (N==1) {
                p= m_ifTests.get(0).getParameter();
            }
            //Else look for Conditional API's tests bits(?!)
            else {
                Go.Test t= m_ifTests.get(Iff.Exists.class);
                if (t==null) {
                    t= m_ifTests.get(Iff.IsTrue.class);
                }
                if (t!=null) {
                    p = t.getParameter();
                }
            }
        }
        return p;
    }


    public String getUnlessProperty()
    {
        String p= "";
        int N= m_unlessTests.size();
        if (N>0) {
            if (N==1) {
                p= m_unlessTests.get(0).getParameter();
            }
            //Else look for Conditional API's tests bits(?!)
            else {
                Go.Test t= m_unlessTests.get(Iff.NotExists.class);
                if (t==null) {
                    t= m_unlessTests.get(Iff.IsNotTrue.class);
                }
                if (t!=null) {
                    p = t.getParameter();
                }
            }
        }
        return p;
    }


    /**
     * Returns <i>null</i> if <em>all</em> of the given "go"
     * tests are <i>true</i>; otherwise, will return a descriptive
     * string of the first test that failed.
     * @param P project under test (non-null)
     * @param tests list of <span class="src">Go.Test</span>s.
     * @see #getLastFailure
     **/
    public static String testTests(Project P, Go.TestList tests)
    {
        AntX.require_(P!=null && tests!=null,IAM_,"test- nonzro proj+list");
        if (!tests.isEmpty()) {
            Iterator itr= tests.iterator();
            while (itr.hasNext()) {
                Go.Test t= (Go.Test)itr.next();
                if (!t.pass(P)) {
                    return t.toString();
                }
            }
            itr = null;
        }
        return null;
    }



    /**
     * Returns the last failed test's descriptive string.
     * Will return <i>null</i> if this helper's test
     * {@linkplain #testTests method} was never called or
     * it did not fail (on last call).
     **/
    public final String getLastFailure()
    {
        return m_lastFailure;
    }



    /**
     * Manually sets the last failure reason for this helper.
     * @param lastFailure last failure string.
     * @since JWare/AntX 0.5
     **/
    public final void setLastFailure(String lastFailure)
    {
        m_lastFailure = lastFailure==null ? "" : lastFailure;
    }



    /**
     * Returns <i>true</i> if <em>all</em> of this helper's
     * if-conditions are <i>true</i>. Returns <i>true</i> if
     * no if-conditions exist.
     * @.sideeffect If failure, updates this helper's last
     *       {@linkplain #getLastFailure failure} information.
     **/
    public boolean testIfCondition()
    {
        String failure = testTests(getProjectNoNull(), m_ifTests);
        if (failure!=null) {
            m_lastFailure = failure;
        }
        return failure==null;
    }


    /**
     * Returns <i>true</i> if <em>all</em> of this helper's
     * unless-conditions are <i>true</i>. Returns <i>true</i>
     * if no unless-conditions exist.
     * @.sideeffect If failure, updates this helper's last
     *       {@linkplain #getLastFailure failure} information.
     **/
    public boolean testUnlessCondition()
    {
        String failure = testTests(getProjectNoNull(), m_unlessTests);
        if (failure!=null) {
            m_lastFailure = failure;
        }
        return failure==null;
    }


    /**
     * Returns <i>true</i> if this helper represents a simple
     * (zero or single condition) if/unless test.
     **/
    public final boolean isSimple()
    {
        return m_ifTests.size()<=1 && m_unlessTests.size()<=1;
    }



    /**
     * Returns <i>true</i> if this helper contains no tests.
     * @since JWare/AntX 0.5
     **/
    public final boolean isEmpty()
    {
        return m_ifTests.isEmpty() && m_unlessTests.isEmpty();
    }



    /**
     * Helper that shows why this helper said "no" (if/unless).
     * Message is logged at requested level.
     * @param from enclosing task (non-null)
     * @param passedIf <i>true</i> if the 'if' bits passed
     * @param lvl preferred log level
     * @.impl Uses this helper's last recorded failure's
     *        information.
     **/
    public void logSkippedBecause(Task from,
                                  boolean passedIf, int lvl)
    {
        Project P= getProject();

        if (!passedIf) {
            P.log(from, Iteration.uistrs().get
                ("task.skipped.if.failed", from.getTaskName(), getLastFailure()),
                  lvl);
        }
        else {
            P.log(from, Iteration.uistrs().get
                ("task.skipped.unless.failed", from.getTaskName(), getLastFailure()),
                  lvl);
        }
    }



    /**
     * Based on current setting of a parameter's constraints,
     * determines if caller should include parameter. This utility
     * method works for any InnerNameValuePair that implements the
     * standard AntX <span class="src">Conditional</span> interface.
     * @param P caller's project (used to resolve constraint properties)
     * @param param source of if/unless condition
     * @return <i>true</i> if pass if/unless test; otherwise <i>false</i>
     * @since JWare/AntX 0.5
     **/
    public static boolean include(final Project P, Conditional param)
    {
        boolean can = true;
        String ifProp = param.getIfProperty();
        String unProp = param.getUnlessProperty();

        if (ifProp!=null || unProp!=null) {
            OptionalExecuteHelper go = new OptionalExecuteHelper(P);
            if (ifProp!=null) {
                go.setIf(ifProp);
            }
            if (unProp!=null) {
                go.setUnless(unProp);
            }
            can = go.testIfCondition() && go.testUnlessCondition();
        }
        return can;
    }



    private Go.TestList m_ifTests= Go.TestList.newList(4);
    private Go.TestList m_unlessTests= Go.TestList.newList(4);
    private String m_lastFailure;//None
}

/* end-of-OptionalExecuteHelper.java */
